
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found
// in the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin      dmartin@cliftonlabs.com
//          Timothy J. McBrayer
//          Malolan Chetlur    
//          Umesh Kumar V. Rajasekaran
//          Narayanan Thondugulam

#include "IIRScram_ProcedureCallStatement.hh"
#include "IIR_ProcedureCallStatement.hh"
#include "IIR_ProcedureDeclaration.hh"
#include "IIR_SimpleName.hh"
#include "IIR_Identifier.hh"
#include "IIR_IndexedName.hh"
#include "IIR_DeclarationList.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_TypeDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_Label.hh"
#include "IIR_List.hh"
#include "IIR_AssociationList.hh"
#include "IIR_AboveAttribute.hh"
#include "set.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "savant.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

extern bool parse_error;

IIRScram_ProcedureCallStatement::~IIRScram_ProcedureCallStatement() {}

void 
IIRScram_ProcedureCallStatement::_publish_vhdl(ostream &_vhdl_out) {

  int parameter_count = actual_parameter_part.num_elements();

  _publish_vhdl_stmt_label(_vhdl_out);

  get_procedure_name()->_publish_vhdl(_vhdl_out);
  
  if(actual_parameter_part.num_elements() != 0) {
    _vhdl_out << "(";
    
    if (get_procedure_name()->get_kind() == IIR_PROCEDURE_DECLARATION) {
      IIR_ProcedureDeclaration *procDecl = (IIR_ProcedureDeclaration *) get_procedure_name();
      parameter_count = procDecl->interface_declarations.num_elements();
    }
    
    if (parameter_count != actual_parameter_part.num_elements()) {
      actual_parameter_part._publish_vhdl(_vhdl_out);
    }
    else {
      actual_parameter_part._publish_vhdl_without_formals(_vhdl_out);
    }
    _vhdl_out << ")";
  }
}

// Note:
// 1. Default label for a procedure call statement is constructed by
//    concatenating the string "PL" with the string representation of the
//    pointer to that IIRScram_ProcedureCallStatement instance.
// 2. The wait label for the procedure call statement is constructed by
//    taking the integer value of the "this" pointer of the
//    IIRScram_ProcedureCallStatement instance.
void 
IIRScram_ProcedureCallStatement::_publish_cc( published_file &_cc_out ) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ProcedureCallStatement::_publish_cc" );

  _publish_cc_proc_label( _cc_out );
  _cc_out << OS( ":{" );
  ASSERT (get_procedure_name() != NULL);
  ASSERT (get_procedure_name()->_is_iir_procedure_declaration() == TRUE);

  IIR_ProcedureDeclaration *proc_decl
    = (IIR_ProcedureDeclaration *)get_procedure_name();
  
  actual_parameter_part._publish_cc_necessary_temporaries_for_TC( _cc_out,
								  &proc_decl->interface_declarations);
  
  // Special handling is required for "writeline" call within a
  // subprogram.
  if(_is_currently_publishing_subprogram() == TRUE) {
    IIR_TextLiteral *declarator = ((IIR_Declaration *) get_procedure_name())->get_declarator();
    if(IIRScram_Identifier::_cmp(declarator, "writeline") == 0 ||
       IIRScram_Identifier::_cmp(declarator, "readline") == 0) {
      _cc_out << "processPtr->";
      get_procedure_name()->_publish_cc_lvalue( _cc_out );
      _cc_out << OS( "(processPtr," );
      if(actual_parameter_part.num_elements() != 0) {
	actual_parameter_part._publish_cc_lvalue( _cc_out );
      }
      _cc_out << CS( ");" ) << CS( "}" );
      return;
    }
  }

  // Assume being called from a process.
  _cc_out << OS( "switch(" );
  get_procedure_name()->_publish_cc_subprogram_call_name( _cc_out );
  _cc_out << OS( "(processPtr" );

  if(actual_parameter_part.num_elements() != 0) {
    _cc_out << ", " << NL();
    actual_parameter_part._publish_cc_with_temporaries_for_TC( _cc_out, 
							       &proc_decl->interface_declarations);
  }
  _cc_out << CS(")") << CS(")") << OS("{")
	  << OS( "case EXECUTE_WAIT_RETURN:" );
  if(_get_currently_publishing_unit() == PROCEDURE) {
    _cc_out << "callStack->push(" << this << ", 0);";
  }
  else if(_get_currently_publishing_unit() == FUNCTION) {
    // Do Nothing.
  } else {
    _cc_out << "(static_cast<VHDLKernel_state *>(getState()))->stack.push(" << this << ", 0);";
  }
  _cc_out << NL();
  if(_get_currently_publishing_unit() == PROCEDURE) {
    _cc_out << "return EXECUTE_WAIT_RETURN;";
  } else if(_get_currently_publishing_unit() == FUNCTION) {
    // Do Nothing.
  } else {
    _cc_out << "return;";
  }
  _cc_out << CS("") << OS( "case RESUME_WAIT_RETURN:" );
  if(_get_currently_publishing_unit() == PROCEDURE) {
    _cc_out << "return RESUME_WAIT_RETURN;";
  }
  else if(_get_currently_publishing_unit() == FUNCTION) {
    // Do Nothing.
  }
  else {
    _cc_out << "return;";
  }
  _cc_out << CS("");
  _cc_out << OS( "case NORMAL_RETURN:" );
  actual_parameter_part._publish_cc_restore_from_temporaries( _cc_out, 
							      &proc_decl->interface_declarations );
  _cc_out << "break;"
	  << CS("")
	  << CS("}")
	  << CS("}");
}

void
IIRScram_ProcedureCallStatement::_publish_cc_proc_label( published_file &_cc_out ) {
  if (get_label() != NULL) {
    get_label()->_publish_cc_lvalue( _cc_out );
  }
  else {
    _cc_out << "PL";
    _cc_out << this;
  }
}

void
IIRScram_ProcedureCallStatement::_build_wait_list(dl_list<IIRScram_WaitStatement> * ){}

void 
IIRScram_ProcedureCallStatement::_build_procedure_call_stmt_list(dl_list<IIR_ProcedureCallStatement> *list) {
  list->append((IIR_ProcedureCallStatement *) this);
}

void 
IIRScram_ProcedureCallStatement::_type_check(){
  // Blatantly stolen from IIR_ProcedureCallStatement.  Changes need to be
  // propagate back to that file.

  set<IIR_TypeDefinition> proc_rval_set(IIR_ProcedureDeclaration::_get_procedure_return_type()); 
  set_procedure_name( get_procedure_name()->_semantic_transform( &proc_rval_set ));
  
  if( get_procedure_name()->get_kind() != IIR_FUNCTION_CALL ){
    set<IIR_Declaration> *my_decls;
    if( get_procedure_name()->get_kind() == IIR_INDEXED_NAME ){
      my_decls = ((IIR_IndexedName *)get_procedure_name())->get_prefix()->_symbol_lookup();
    }
    else{
      my_decls = get_procedure_name()->_symbol_lookup();
    }

    if( my_decls == NULL ){
      report_undefined_symbol( get_procedure_name() );
      return;
    }

    IIR_Declaration *current_decl = my_decls->get_element();
    while( current_decl != NULL ){
      if( current_decl->get_kind() != IIR_PROCEDURE_DECLARATION ){
	my_decls->remove( current_decl );
      }
      current_decl = my_decls->get_next_element();
    }
    
    resolve_subprogram_decls( my_decls, &actual_parameter_part, &proc_rval_set );
    
    switch( my_decls->num_elements() ){
    case 0:{
      ostringstream err;
      err << "No declaration of procedure with signature matching |"
	  << *get_procedure_name() << "|.";
      report_error( this, err.str() );
      break;
    }
    case 1:{
      set_procedure_name( get_procedure_name()->_decl_to_decl( my_decls->get_element() ) );

      IIR *temp_name = get_procedure_name();
      ASSERT( temp_name->_is_iir_subprogram_declaration() == TRUE );
      IIR_SubprogramDeclaration *my_decl = (IIR_SubprogramDeclaration *)temp_name;

      actual_parameter_part._resolve_and_order( &my_decl->interface_declarations, NULL, this );
      actual_parameter_part._fill_in_defaults( this, &my_decl->interface_declarations );

      break;
    }
    default:{
      report_ambiguous_error( get_procedure_name(), my_decls );
    }
    }
  
    delete my_decls;
  }
  else{
    
    IIR_FunctionCall *old_proc_name =  (IIR_FunctionCall *)get_procedure_name();
    set_procedure_name( old_proc_name->get_implementation() );
    IIR_List::_listcopy( actual_parameter_part,
			 old_proc_name->parameter_association_list );

    delete old_proc_name;
  }
    
  ASSERT( get_procedure_name()->_is_resolved() == TRUE );  
}

void
IIRScram_ProcedureCallStatement::_get_list_of_input_signals( set<IIR> *list ){
  actual_parameter_part._get_list_of_input_signals(list);
}

void
IIRScram_ProcedureCallStatement::_get_signal_source_info( set<IIR> *siginfo ){
  ASSERT(get_procedure_name()->get_kind() == IIR_PROCEDURE_DECLARATION);

  actual_parameter_part._get_signal_source_info( siginfo );
}

IIR_Boolean
IIRScram_ProcedureCallStatement::_is_above_attribute_found() {
  IIR_Boolean retval = FALSE;
  retval = retval || actual_parameter_part._is_above_attribute_found();
  return retval;
}

void
IIRScram_ProcedureCallStatement::
_build_above_attribute_set(set<IIR_AboveAttribute> *to_build) {
  actual_parameter_part._build_above_attribute_set(to_build);
}

IIR*
IIRScram_ProcedureCallStatement::_clone() {
  IIR_ProcedureCallStatement *callstmt = new IIR_ProcedureCallStatement();

  IIR_SequentialStatement::_clone(callstmt);
  callstmt->set_procedure_name(get_procedure_name());
  
  IIR_AssociationElement *newassoc, *assoc = actual_parameter_part.first();
  while (assoc != NULL) {
    newassoc = (IIR_AssociationElement*)assoc->_clone();
    callstmt->actual_parameter_part.append(newassoc);
    assoc = actual_parameter_part.successor(assoc);
  }

  return callstmt;
}

visitor_return_type *IIRScram_ProcedureCallStatement::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ProcedureCallStatement(this, arg);
};
