from config.ConfigManager import ConfigManager
from DisplayTarget import DisplayTarget
from TargetGroup import TargetGroup
from main import UNSET_COORD
from scripting.Script import Script
from utils import actionparser
from utils.Struct import Struct
from utils.Observable import Observable
from utils import Unit

import NullObject
import utils.dialog

import gtk
import time
import random
import sys
import os
import re

try:
    import gnomevfs
except ImportError:
    import gnome.vfs as gnomevfs


#
# Class for display windows.
#
class Display(Observable):

    # observer commands
    OBS_CLOSE = 0
    OBS_RESTART = 1


    # regular expression to test whether a path is absolute
    __IS_ABSOLUTE_PATH_RE = re.compile("[a-zA-Z]+://.+")


    def __init__(self, ident, rep):

        # the display menu
        self.__DISPLAY_MENU = [
            (_("_Configure display"), self.__handle_configure),
            (_("_Move display"), self.__handle_move),
            ("", None),
            (_("_View Source"), self.__handle_source),
            ("", None),
            (_("Re_start display"), self.__handle_restart),
            (_("_Remove display"), self.__handle_remove)]

        self.__motion_index = 0
        self.__leave_index = 0

        # the controls handling the deprecated sensor stuff: id -> control
        self.__sensor_controls = {}

        # mapping of the elements to their parent arrays: child_id -> array_id
        self.__arrays = {}

        # the deprecated sensor menu
        # FIXME: remove eventually
        self.__sensor_menu = []

        # the container of this display
        self.__container = None
        self.__container_settings = {}

        # the root TargetGroup
        self.__group = None

        # the scripting environment
        self.__script = Script(ident)

        # the configurator object
        self.__configurator = None

        # the path of the .display file
        self.__path = os.path.dirname(rep)
        self.__display_file = rep

        # the unique ID of this display
        self.__id = ident

        # the last position of the mouse pointer (used for filtering out
        # unwanted events)
        self.__last_pointer_pos = (-1, -1)

        # mapping between sensors and targets; which target watches
        # which sensor?
        # (sensor, port) -> (target, property)
        self.__mapping = {}

        # temporary data for remembering the position of the last mouse click
        self.__pointer_pos = (0, 0)

        # whether the display reacts on events
        self.__is_sensitive = True

        # the menu to open
        self.__menu = None


    def get_next_child_index(self): return -1
    def get_index_path(self): return []



    #
    # Sets the container of the display.
    #
    def set_container(self, container):

        self.__container = container
        container.add_observer(self.__on_observe_container)
        container.add(self.__group.get_widget())

        for k, v in self.__container_settings.items():
            self.set_prop(k, v)
        self.__container_settings.clear()

        # set geometry
        x, y, w, h = self.__group.get_geometry()
        ux, uy, uw, uh = self.__group.get_user_geometry()
        if (ux.is_unset() or uy.is_unset()):
            container.set_position(UNSET_COORD, UNSET_COORD)
        else:
            container.set_position(x.as_px(), y.as_px())
        container.set_size(w.as_px(), h.as_px())
        container.connect("delete-event", self.__on_close)

        container.connect("button-press-event", self.__on_button, False)
        container.connect("button-release-event", self.__on_button, True)
        container.connect("key-press-event", self.__on_key, False)
        container.connect("key-release-event", self.__on_key, True)
        container.connect("scroll-event", self.__on_scroll)
        container.connect("motion-notify-event", self.__on_motion, False)
        container.connect("leave-notify-event", self.__on_motion, True)




    #
    # Executes the given script.
    #
    def execute_script(self, code):

        self.__script.execute(code)



    #
    # Executes the given callback script.
    #
    def execute_callback_script(self, code, this):

        self.__script.add_target("self", this, True)
        self.__script.execute(code)



    #
    # Adds the given target to the scripting environment.
    #
    def add_target_to_script(self, name, target):

        self.__script.add_target(name, target)



    #
    # Sets the sensitive flag of the display. Insensitive displays don't react
    # on user events.
    #
    def set_sensitive(self, value):

        self.__is_sensitive = value



    #
    # Returns the path of the .display file.
    #
    def get_path(self):

        return self.__path


    #
    # Returns the full path of the given path which may be relative to the
    # .display file.
    #
    def get_full_path(self, path):

        # a path is absolute iff it starts with "/" or with a protocol name
        # such as "http://", otherwise it's relative
        if (path.startswith("/") or self.__IS_ABSOLUTE_PATH_RE.match(path)):
            return path
        else:
            return os.path.join(self.__path, path)


    #
    # Returns the display.
    #
    def _get_display(self): return self



    def new_child(self, childtype, settings, children):

        import targetregistry
        self.__group = targetregistry.create(childtype, self)
        self.__group.get_widget().show()
        cid = settings["id"]
        self.add_target_to_script(cid, self.__group)

        for t, s, c in children:
            self.__group.new_child(t, s, c)

        for key, value in settings.items():
            self.__group.set_xml_prop(key, value)


    #
    # Sets the configurator object for storing and changing settings.
    #
    def set_configurator(self, configurator):

        self.__configurator = configurator
        self.__configurator.set_getter_and_setter_and_caller(
            self.__script.get_value,
            self.__script.set_value,
            self.__script.call_function)

        # init configurator
        self.__configurator.initialize(self.__id)
        

    #
    # Opens the configuration dialog for this display.
    #
    def __open_configurator(self):
        assert (self.__configurator)

        configurators = [s.configurator
                         for s in self.__sensor_controls.values()]

        if (configurators):
            # support old deprecated sensor stuff
            from DisplayConfigurator import DisplayConfigurator
            dconf = DisplayConfigurator(configurators)
            
        else:
            self.__configurator.show()



    #
    # Removes this display.
    #
    def remove_display(self):

        self.drop_observers()
        self.__container.drop_observers()
        self.__script.stop()
        for s in self.__sensor_controls.values():
            s.stop = True

        self.__container.remove(self.__group.get_widget())
        self.__group.delete()
        self.__container.close()

        del self.__sensor_controls
        del self.__mapping
        del self.__group


    #
    # Purges this display.
    #
    def purge_display(self):

        print "Info: Purging ID '%s'." % (self.__id)
        config  = ConfigManager()
        config.clear(self.__id)

        # remove the state file
        self.__script.remove()


    #
    # Reacts on closing the window.
    #
    def __on_close(self, src, event):

        self.update_observer(self.OBS_CLOSE, self.__id)



    #
    # Sends the given action with an event object to the display.
    #
    def send_action(self, src, action, event):

        call = src.get_action_call(action)
        if (call):

            # analyze call to see if it's a legacy call
            # FIXME: remove eventually :)
            import re
            if (re.match("\w+:.*", call)):
                import warnings
                warnings.warn("Please use new style call", DeprecationWarning)
                
                try:
                    legacy_args = event._args
                except:
                    legacy_args = []

                legacy_call = actionparser.parse(call)
                path = src.get_index_path()
                self.call_sensor(legacy_call, path, *legacy_args)

            else:
                src.notify_handle_action(True)
                src._setp("event", event)
                self.execute_callback_script(call, src)


    #
    # Has to be called when a menu is to open. Do not open the menu yourself.
    #
    def queue_menu(self, menu):

        self.__menu = menu


    #
    # Reacts on keypresses.
    #
    def __on_key(self, src, event, is_release):

        key = gtk.gdk.keyval_name(event.keyval)
            
        # F10 opens the menu
        if (key == "F10"):
            action = DisplayTarget.ACTION_MENU
            px, py = self.__pointer_pos
            self.__group.handle_action(action,
                                       Unit.Unit(px, Unit.UNIT_PX),
                                       Unit.Unit(py, Unit.UNIT_PX),
                                       Struct(button = 0,
                                              _args = [0]))
            self.__group.notify_handle_action(True)
            
            self.__make_menu(0, event.time)

        else:
            if (is_release): action = DisplayTarget.ACTION_KEY_PRESS
            else: action = DisplayTarget.ACTION_KEY_RELEASE
            px, py = self.__container.get_pointer()
            self.__group.handle_action(action,
                                       Unit.Unit(px, Unit.UNIT_PX),
                                       Unit.Unit(py, Unit.UNIT_PX),
                                       Struct(key = key))
            self.__group.notify_handle_action(True)
            


    #
    # Reacts on button events.
    #
    def __on_button(self, src, event, is_release = False):

        if (not self.__is_sensitive): return

        px, py = self.__container.get_pointer()
        lx, ly = self.__pointer_pos
        button = event.button

        # determine action
        if (not is_release):
            self.__pointer_pos = (px, py)
            if (button == 1 and event.type == gtk.gdk._2BUTTON_PRESS):
                action = DisplayTarget.ACTION_DOUBLECLICK
            elif (button == 1):
                action = DisplayTarget.ACTION_PRESS
            elif (button == 2):
                return
            elif (button == 3):
                return
            else:
                return

        else:
            if (button == 1):
                if (abs(lx - px) < 10 and abs(ly - py) < 10):
                    action = DisplayTarget.ACTION_CLICK
                else:
                    action = DisplayTarget.ACTION_RELEASE
            elif (button == 2):
                return
            elif (button == 3):
                action = DisplayTarget.ACTION_MENU
            else:
                return

        self.__group.handle_action(action,
                                   Unit.Unit(px, Unit.UNIT_PX),
                                   Unit.Unit(py, Unit.UNIT_PX),
                                   Struct(button = button,
                                                          _args = [button]))
        self.__group.notify_handle_action(True)

        # extend the menu or create one if there's none
        if (action == DisplayTarget.ACTION_MENU):
            self.__make_menu(0, event.time)


    #
    # Builds the menu.
    #
    def __make_menu(self, button, eventtime):

        need_separator = False
        if (not self.__menu):
            ident = Display.make_id()
            self.__menu = self.__group.new_child("menu",
                                                 {"id": ident},
                                                 [])
            self.__group.set_prop("on-menu", "Dsp.%s.popup()" % (ident))
            
        elif (not "display" in self.__menu.get_slots()):
            # add separator
            need_separator = True
        #end if

        # FIXME: remove eventually :)
        if (self.__sensor_menu and
            not "deprecated sensor menu" in self.__menu.get_slots()):
            for entry in self.__sensor_menu:
                if (not entry):
                    label, active, submenu, h, args = \
                           "", True, None, None, []
                else:
                    label, active, submenu, h, args = entry

                item = self.__menu.new_child("menu-item",
                                             {"id": Display.make_id(),
                                              "slot": "deprecated sensor menu",
                                              "label": label},
                                             [])
                args = [item.get_widget()] + list(args)
                if (h): item.set_handler(h, *args)
                if (not active): item.set_prop("active", False)
        #end if

        if (not "display" in self.__menu.get_slots()):
            if (need_separator):
                self.__menu.new_child("menu-item",
                                      {"id": Display.make_id(),
                                       "slot": "display"}, [])

            for label, h in self.__DISPLAY_MENU:
                item = self.__menu.new_child("menu-item",
                                             {"id": Display.make_id(),
                                              "slot": "display",
                                              "label": label},
                                             [])
                if (h): item.set_handler(h)
        #end if

        self.__menu.prepare()
        self.__menu.get_widget().popup(None, None, None, button,
                                       eventtime)
        self.__menu = None




    #
    # Reacts on moving the mouse.
    #
    def __on_motion(self, src, event, is_leave):

        if (not self.__is_sensitive): return
        px, py = self.__container.get_pointer()
        w, h = self.__group.get_widget().size_request()

        if (is_leave):
            self.__leave_index += 1
            gtk.idle_add(self.__queue_motion, self.__leave_index,
                         px, py, w, h, is_leave)
        else:
            self.__motion_index += 1
            gtk.idle_add(self.__queue_motion, self.__motion_index,
                         px, py, w, h, is_leave)



    def __queue_motion(self, index, px, py, w, h, is_leave):

        # we can drop events which are not recent
        if (is_leave and index != self.__leave_index): return
        elif (not is_leave and index != self.__motion_index): return
        #if (index != self.__motion_queue_index): return
        # some window managers send a LEAVE event for mouse clicks;
        # work around this
        if (is_leave and (px, py) == self.__last_pointer_pos
            and (0 <= px <= w) and (0 <= py <= h)):
            is_leave = False

        # don't do redundant work
        if (not is_leave and (px, py) == self.__last_pointer_pos): return
        else: self.__last_pointer_pos = (px, py)

        
        if (not is_leave):
            ux = Unit.Unit(px, Unit.UNIT_PX)
            uy = Unit.Unit(py, Unit.UNIT_PX)
            self.__group.handle_action(DisplayTarget.ACTION_MOTION,
                                       ux, uy, Struct())
            self.__group.notify_handle_action(True)
        else:
            self.__group.notify_handle_action(False)



    #
    # Reacts on rolling the mouse wheel.
    #
    def __on_scroll(self, src, event):

        if (not self.__is_sensitive): return

        px, py = self.__container.get_pointer()

        if (event.direction == gtk.gdk.SCROLL_UP):
            direction = 0
        elif (event.direction == gtk.gdk.SCROLL_DOWN):
            direction = 1
        else:
            direction = -1

        self.__group.handle_action(DisplayTarget.ACTION_SCROLL,
                                   Unit.Unit(px, Unit.UNIT_PX),
                                   Unit.Unit(py, Unit.UNIT_PX),
                                   Struct(direction = direction,
                                          _args = [direction]))
        self.__group.notify_handle_action(True)



    #
    # Observer for the container.
    #
    def __on_observe_container(self, src, cmd):

        if (cmd == src.OBS_GEOMETRY):
            x, y, w, h = src.get_geometry()
            x = Unit.Unit(x, Unit.UNIT_PX)
            y = Unit.Unit(y, Unit.UNIT_PX)
            w = Unit.Unit(w, Unit.UNIT_PX)
            h = Unit.Unit(h, Unit.UNIT_PX)
            cx, cy, cw, ch = self.__group.get_geometry()

            if ((cx, cy) != (x, y)):
                ax, ay = self.__group.get_anchored_coords(x, y, cw, ch)
                dx, dy= x - ax, y - ay
                self.set_position(x.as_px() + dx.as_px(),
                                  y.as_px() + dy.as_px())

        elif (cmd == src.OBS_LOCK): self.set_sensitive(False)
        elif (cmd == src.OBS_UNLOCK): self.set_sensitive(True)


    #
    # Observer for the root group.
    #
    def child_observer(self, src, cmd):

        if (cmd == src.OBS_GEOMETRY):
            x, y, w, h = self.__group.get_geometry()
            ux, uy = self.__group.get_user_geometry()[:2]

            if (self.__container):
                self.__container.set_position(x.as_px(), y.as_px())

                if (w != 0 and h != 0):
                    self.__container.set_size(w.as_px(), h.as_px())



    #
    # Sets the anchored position of the display.
    #
    def set_position(self, x = UNSET_COORD, y = UNSET_COORD):

        if (x == y == UNSET_COORD):
            #self.__container.set_position(x, y)
            pass
        else:
            self.__group.set_xml_prop("x", str(x))
            self.__group.set_xml_prop("y", str(y))



    #
    # Sets the configuration settings.
    # FIXME: remove eventually :)
    #
    def __set_settings(self, settings, sensor):

        for key, value in settings.get_entries():
            # get all (target, property) tuples that are watching the given
            # sensor key and notify the targets
            entries = self.__mapping.get((sensor, key), [])

            # if an array does not have enough elements, create them
            if (not entries and "[" in key):
                pos = key.find("[")
                port = key[:pos]
                path = key[pos + 1:-1].split("][")
                new_length = int(path.pop()) + 1
                if (port in self.__arrays):
                    array = self.__arrays[port]
                    array.set_prop("length", new_length)
                    entries = self.__mapping.get((sensor, key), [])

            for target, prop in entries:
                target.set_xml_prop(prop, value)



    #
    # Sets the window configuration.
    #
    def set_prop(self, key, value):

        if (not self.__container):
            # save for later when we have a container
            self.__container_settings[key] = value

        elif (key == "window-flags"):
            self.__container.set_window_flags(value)

        elif (key == "title"):
            self.__container.set_title(value)

        elif (key == "icon"):
            filename = self.get_full_path(value)
            try:
                loader = gtk.gdk.PixbufLoader()
                data = gnomevfs.read_entire_file(filename)
                loader.write(data, len(data))
                loader.close()
                pixbuf = loader.get_pixbuf()
                self.__container.set_icon(pixbuf)
            except:
                print "could not set icon ", filename

        elif (key == "shape"):
            filename = self.get_full_path(value)
            try:
                loader = gtk.gdk.PixbufLoader()
                data = gnomevfs.read_entire_file(filename)
                loader.write(data, len(data))
                loader.close()
                pixbuf = loader.get_pixbuf()
                pix, mask = pixbuf.render_pixmap_and_mask(1)
                self.__container.set_shape(mask)
            except:
                print "could not set shape ", filename



    #
    # Adds a sensor to this display.
    # FIXME: remove eventually :)
    #
    def add_sensor(self, ident, sensor):

        def set_menu(menu): self.__sensor_menu = menu

        self.__sensor_controls[ident] = sensor
        sensor.bind("output", self.__set_settings, ident)
        sensor.bind("menu", set_menu)


    #
    # Binds a sensor's output to an element.
    # FIXME: remove eventually :)
    #
    def bind_sensor(self, sensorplug, element, prop):

        def h(value, element, prop, port):
            for k, v in value.items():
                if (k == port):
                    element.set_xml_prop(prop, v)

        ident, port = sensorplug.split(":")
        if (not (element, prop) in
              self.__mapping.setdefault((ident, port), [])):
            self.__mapping[(ident, port)].append((element, prop))


    #
    # Calls a function of a Sensor.
    # FIXME: Remove eventually
    #
    def call_sensor(self, cmd, path, *args):
        assert(cmd)

        args = list(args)
        for ident, callname, userargs in cmd:
            sensorctrl = self.__sensor_controls[ident]

            #sensor = self.__get_sensor(ident)
            allargs = args + userargs

            # the sensor is an external module, so we make sure it cannot crash
            # the application
            try:
                #sensor.send_action(callname, path, allargs)
                sensorctrl.action = (callname, path, allargs)

            except StandardError, exc:
                import traceback; traceback.print_exc()
                print >>sys.stderr, "The sensor produced an error: %s" % (exc)



    #
    # Unbinds a sensor's output from an element.
    # FIXME: remove eventually :)
    #
    def unbind_sensor(self, sensorplug, element, prop):

        ident, port = sensorplug.split(":")
        try:
            self.__mapping[(ident, port)].remove((element, prop))
        except KeyError:
            pass


    #
    # Registers an array to be parent array of the given child.
    # FIXME: remove eventually :)
    #
    def register_array_for_port(self, array, sensorplug):

        ident, port = sensorplug.split(":")
        self.__arrays[port] = array



    #
    # Returns the anchored geometry of this display.
    #
    def get_geometry(self):

        try:
            x, y, w, h = self.__group.get_geometry()
            ax, ay = self.__group.get_anchored_coords(x, y, w, h)
            dx, dy = x - ax, y - ay
            return (x + dx, y + dy, w, h)
        except:
            return (Unit.ZERO, Unit.ZERO, Unit.ZERO, Unit.ZERO)


    def __handle_configure(self, *args):

        self.__open_configurator()


    def __handle_move(self, *args):

        self.__container.set_position(UNSET_COORD, UNSET_COORD)


    def __handle_source(self, *args):

        os.system("gnome-text-editor %s & disown" % self.__display_file)


    def __handle_restart(self, *args):

        self.update_observer(self.OBS_RESTART, self.__id)


    def __handle_remove(self, *args):

        def remove_display(*args): self.update_observer(self.OBS_CLOSE, self.__id)
        

        utils.dialog.question(None,
                              _("Do you really want to remove this display ?"),
                              _("This display will no longer be displayed "
                                "and its configuration will be purged."),
                              (gtk.STOCK_CANCEL, Null),
                              (gtk.STOCK_DELETE, remove_display)
                              )

    
    #
    # Returns a unique ID string.
    #
    def make_id():
        return "id%d%d" % (int(time.time() * 100), random.randrange(0xffff))

    make_id = staticmethod(make_id)
