# -*- coding: UTF-8 -*-

import struct

__doc__ = """
Transport of Data



Format of Transported Data
--------------------------

In order to communicate with external non-python programs, the xdr module
implements a very simple protocol à la XDR.

The send function expect variable length arguments of str type.
The arguments will be escaped, joint and packed into binary.
All the arguments have to be joint into a '\\n' delimited str.

e.g. send(s, "a", "b", "c") -> "a\\nb\\nc"

If one argument contains '\n' it has to be escaped

e.g. send(s, "a\\n", "b", "c") -> "a\\\\n\\nb\\nc"

If their are no arguments, an empty str is sent.

To be safely and efficiently exchanged, arguments are packed into the
following representation :

   0     1     2     3   ...
+-----+-----+-----+-----+-----+---+
| length n  |byte0|byte1|...| n-1 |
+-----+-----+-----+-----+-----+---+
|<-2 bytes->|<------n bytes------>|


The length is encoded into a two bytes unsigned integer in native
epresentation.

e.g. send(s, "a", "b", "c") -> "\\x05\\x00a\\nb\\nc" (05 00 61 0a 62 0a 63)

Recieved data has to be unpacked, splitted and unescaped.
"""

# we should have used xrdlib
# but i don't know if this add another dependency
# and a Packer object has to be created each call
# so it may be slower
# if we get issues with the current protocol, we should
# switch to xdrlib

# 64k is large enough

_HEADER_FORMAT = 'H'
_SIZEOF_HEADER = struct.calcsize(_HEADER_FORMAT)


# 16 bits unsigned integer header in native order representif the length of
# the following string. The string represents a '\n'-seperated tuple.


def send(s, *args):

    """
    Writes data on a socket

    @param s: given socket
    @type  s: socket object

    @param *args: variable length arguments
    @type  *args: tuple

    NB: *args is a tuple and therefor a tuple is transmitted on the socket.
    That's why recv returns a tuple equals to args.
    """

    data = '\n'.join([ str(a).replace('\n', '\\n') for a in args ])
    assert len(data) != 0
    header = struct.pack(_HEADER_FORMAT, len(data))

    s.send(header)
    s.send(data)



def recv(s):

    """
    Reads data on a socket

    @param s: given socket
    @type  s: socket object

    @return : incoming data
    @rtype  : tuple
    """

    header = s.recv(_SIZEOF_HEADER)

    if not header:
        return

    size = struct.unpack(_HEADER_FORMAT, header)[0]
    data = s.recv(size)

    return tuple([ a.replace('\\n', '\n') for a in data.split('\n') ])
