/* $Id: cleanup.c,v 1.1.1.1.2.3 2003/11/19 11:30:12 cegger Exp $
******************************************************************************

   LibGG - Functions for adding and removing cleanup callbacks

   Copyright (C) 1998 Marcus Sundberg	[marcus@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include "plat.h"
#include <stdlib.h>
#include <ggi/gg.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SIGNAL_H
#include <signal.h>
#endif

typedef struct funclist {
	ggcleanup_func	*func;
	void		*arg;
	struct funclist	*next;
} funclist;

static funclist *firstfunc = NULL;
static ggcleanup_func *currentfunc = NULL;
static int is_registered = 0;
static int force_exit = 0;


static void do_cleanup(void)
{
	funclist *curr = firstfunc, *prev;
	
	while (curr != NULL) {
		currentfunc = curr->func;
		curr->func(curr->arg);
		prev = curr;
		curr = curr->next;
		currentfunc = NULL;
		/* Make sure we don't call the function again */
		ggUnregisterCleanup(prev->func, prev->arg);
	}
}

static void cleanup_function(void)
{
	do_cleanup();
#ifdef HAVE__EXIT
	if (force_exit) {
		_exit(0);
	}
#endif
}

static void cleanup_function_ret(int retval)
{
	do_cleanup();
#ifdef HAVE__EXIT
	if (force_exit) {
		_exit(retval);
	}
#endif
}



#if defined(HAVE_SIGNAL) || defined(HAVE_SIGACTION)

typedef void (ggsighandler)(int);

typedef struct {
	int		  sig;
	ggsighandler     *oldhandler;
#ifdef HAVE_SIGACTION
	struct sigaction  oldsa;
#endif
} gg_siginfo;


static gg_siginfo siglist[] = {
#ifdef SIGHUP
	{ SIGHUP,  SIG_ERR },
#endif
#ifdef SIGINT
	{ SIGINT,  SIG_ERR },
#endif
#ifdef SIGQUIT
	{ SIGQUIT, SIG_ERR },
#endif
#ifdef SIGILL
	{ SIGILL,  SIG_ERR },
#endif
#ifdef SIGTRAP
	{ SIGTRAP, SIG_ERR },
#endif
#ifdef SIGABRT
	{ SIGABRT, SIG_ERR },
#endif
#ifdef SIGBUS
	{ SIGBUS,  SIG_ERR },
#endif
#ifdef SIGFPE
	{ SIGFPE,  SIG_ERR },
#endif
#ifdef SIGSEGV
	{ SIGSEGV, SIG_ERR },
#endif
#ifdef SIGPIPE
	{ SIGPIPE, SIG_ERR },
#endif
#ifdef SIGALRM
	{ SIGALRM, SIG_ERR },
#endif
#ifdef SIGTERM
	{ SIGTERM, SIG_ERR },
#endif
#ifdef SIGSTKFLT
	{ SIGSTKFLT, SIG_ERR },
#endif
#ifdef SIGIO
	{ SIGIO,   SIG_ERR },
#endif
#if defined(SIGPOLL) && (SIGPOLL != SIGIO)
	{ SIGPOLL, SIG_ERR },
#endif
#ifdef SIGXCPU
	{ SIGXCPU, SIG_ERR },
#endif
#ifdef SIGXFSZ
	{ SIGXFSZ, SIG_ERR },
#endif
#ifdef SIGVTALRM
	{ SIGVTALRM, SIG_ERR },
#endif
#ifdef SIGPROF
	{ SIGPROF, SIG_ERR },
#endif
#ifdef SIGPWR
	{ SIGPWR,  SIG_ERR },
#endif
#ifdef SIGLOST
	{ SIGLOST, SIG_ERR },
#endif
#ifdef SIGUNUSED
	{ SIGUNUSED, SIG_ERR }
#endif
};

#define SIGLIST_LEN	sizeof(siglist)/sizeof(gg_siginfo)


static void sighandler(int signum)
{
	unsigned int i;
	ggsighandler *oldfunc = NULL;

	for (i = 0; i < SIGLIST_LEN; i++) {
		if (siglist[i].sig == signum) {
			if (siglist[i].oldhandler == SIG_IGN) {
				signal(signum, sighandler);
				return;
			}
			if (siglist[i].oldhandler != SIG_DFL) {
				oldfunc = siglist[i].oldhandler;
				break;
			}
		}
	}
	if (oldfunc) {
		oldfunc(signum);
	}
	fprintf(stderr, "Terminating on signal %d\n", signum);
	cleanup_function_ret(signum);
	exit(signum);
}


#ifdef HAVE_SIGACTION

static void setsa(struct sigaction *sa, ggsighandler *func)
{
	sa->sa_handler = func;
	sa->sa_flags = 0;
	sigemptyset(&sa->sa_mask);
}

static void unregister_sighandler(void)
{
	unsigned int i;

	for (i = 0; i < SIGLIST_LEN; i++) {
		struct sigaction sa;

		if (siglist[i].oldhandler == SIG_ERR) continue;
		if (sigaction(siglist[i].sig, NULL, &sa) != 0) continue;
		if (sa.sa_flags == 0 && sa.sa_handler == sighandler) {
			sigaction(siglist[i].sig, &siglist[i].oldsa, NULL);
		} else {
			/* Someone else has changed this signal action, so
			   we shouldn't touch it */
		}

		siglist[i].oldhandler = SIG_ERR;
	}
}		

static void register_sighandler(void)
{
	unsigned int i;

	for (i = 0; i < SIGLIST_LEN; i++) {
		if (sigaction(siglist[i].sig, NULL, &siglist[i].oldsa) != 0) {
			continue;
		}
		if (siglist[i].oldsa.sa_handler == SIG_DFL ||
		    siglist[i].oldsa.sa_handler == SIG_IGN) {
			struct sigaction sa;

			setsa(&sa, sighandler);
			if (sigaction(siglist[i].sig, &sa, NULL) != 0) {
				continue;
			}
			siglist[i].oldhandler = siglist[i].oldsa.sa_handler;
		}
	}
}

#else /* HAVE_SIGACTION */

static void unregister_sighandler(void)
{
	unsigned int i;

	for (i = 0; i < SIGLIST_LEN; i++) {
		ggsighandler *oldfunc;

		if (siglist[i].oldhandler == SIG_ERR) continue;
		oldfunc = signal(siglist[i].sig, siglist[i].oldhandler);
		if (oldfunc != sighandler) {
			/* Someone else has changed this signal handler, so
			   we shouldn't touch it */
			signal(siglist[i].sig, oldfunc);
		}
		siglist[i].oldhandler = SIG_ERR;
	}
}		

static void register_sighandler(void)
{
	unsigned int i;

	for (i = 0; i < SIGLIST_LEN; i++) {
		ggsighandler *oldfunc;

		oldfunc = signal(siglist[i].sig, sighandler);
		if (oldfunc != SIG_DFL &&
		    oldfunc != SIG_IGN) {
			/* If there's a sighandler registered already we don't
			   have to care */
			if (oldfunc != SIG_ERR) {
				signal(siglist[i].sig, oldfunc);
			}
			continue;
		}
		siglist[i].oldhandler = oldfunc;
	}
}
#endif /* HAVE_SIGACTION */

#else /* HAVE_SIGNAL || HAVE_SIGACTION */

#define register_sighandler() /* empty */
#define unregister_sighandler() /* empty */

#endif /* HAVE_SIGNAL || HAVE_SIGACTION */			

int ggRegisterCleanup(ggcleanup_func *func, void *arg)
{
	int ret = 0;
	funclist *newlist;

	ggLock(_gg_global_mutex);
	register_sighandler();
	if (!is_registered) {
		if (atexit(cleanup_function) != 0) {
			ret = GGI_EUNKNOWN;
			goto out;
		}
		is_registered = 1;
	}
	if ((newlist = malloc(sizeof(funclist))) == NULL) {
		ret = GGI_ENOMEM;
		goto out;
	}
	newlist->func = func;
	newlist->arg = arg;
	
	/* Insert first in list */
	newlist->next = firstfunc;
	firstfunc = newlist;

  out:
	ggUnlock(_gg_global_mutex);

	return ret;
}


int ggUnregisterCleanup(ggcleanup_func *func, void *arg)
{
	funclist *curr = firstfunc, *prev = NULL;

	if (func == currentfunc) {
		/* Don't unregister a function while it is being called */
		return 0;
	}
	ggLock(_gg_global_mutex);
	while (curr != NULL) {
		if (curr->func == func && curr->arg == arg) {
			if (curr == firstfunc) firstfunc = curr->next;
			else prev->next = curr->next;
			free(curr);
			if (firstfunc == NULL) unregister_sighandler();
			ggUnlock(_gg_global_mutex);
			return 0;
		}
		prev = curr;
		curr = curr->next;
	}

	ggUnlock(_gg_global_mutex);
	return GGI_ENOTALLOC;
}


void ggCleanupForceExit(void)
{
	force_exit = 1;
}
