/***************************************************************************/
/* 		This code is part of WWW graber called pavuk		   */
/*		Copyright (c) 1997,1998,1999 Ondrejicka Stefan		   */
/*		(ondrej@idata.sk)					   */
/*		Distributed under GPL 2 or later			   */
/***************************************************************************/

#include "config.h"

#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <errno.h>
#ifdef HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

#include "lfname.h"
#include "url.h"
#include "tools.h"
#include "tr.h"
#include "dlhash_tools.h"

struct lfname_lsp_interp {
	char	*urlstr;
	char	*scheme;
	char	*passwd;
	char	*user;
	char	*host;
	char	*domain;
	char	*port;
	char	*path;
	char	*name;
	char	*basename;
	char	*extension;
	char	*query;
	char	*deflt;
	lfname	*orig;
};

enum lfname_lsp_type {
	LF_LSP_UNKNOWN,	/*** unknown ***/
	LF_LSP_STR,	/*** string variable ***/
	LF_LSP_NUM,	/*** number variable ***/
	LF_LSP_MACRO,	/*** macro variable ***/
	LF_LSP_SUB,	/*** subpart from regex ***/
	LF_LSP_SC,	/*** strcat function ***/
	LF_LSP_SS,	/*** substr function ***/
	LF_LSP_HASH,	/*** hash function ***/
	LF_LSP_MD5,	/*** md5 function ***/
	LF_LSP_LOWER,	/*** lowerstr function ***/
	LF_LSP_UPPER,	/*** uperstr function ***/
	LF_LSP_UENC,	/*** urlencode function ***/
	LF_LSP_DELCHR,	/*** delchr function ***/
	LF_LSP_TRCHR,	/*** trchr function ***/
	LF_LSP_TRSTR,	/*** trstr function ***/
	LF_LSP_STRSPN,	/*** strspn function ***/
	LF_LSP_STRCSPN,	/*** strcspn function ***/
	LF_LSP_STRLEN,	/*** strlen function ***/
	LF_LSP_NRSTR,	/*** nrtostr function ***/
	LF_LSP_LCHR,	/*** last character offset ***/
	LF_LSP_PLS,	/*** plus ***/
	LF_LSP_MNS,	/*** minus ***/
	LF_LSP_MOD,	/*** mod ***/
	LF_LSP_MUL,	/*** multiply ***/
	LF_LSP_DIV	/*** divide ***/
};
	
struct lfname_lsp_var {
	enum lfname_lsp_type	type;
	union {
		char	*str;
		int	num;
		char	macro;
	} val;
	enum lfname_lsp_type	rettype;
	union {
		char	*str;
		int	num;
	} ret_val;
	struct lfname_lsp_var	*param1;
	struct lfname_lsp_var	*param2;
	struct lfname_lsp_var	*param3;
};


static char *lfname_lsp_get_by_url(struct lfname_lsp_interp *);
static struct lfname_lsp_var *lfname_lsp_analyze(char **);
static void lfname_lsp_var_free(struct lfname_lsp_var *);

static char *_strfindnchr(str , chr , n)
char *str;
int chr;
int n;
{
        int cnt;
        char *p;

        for (p = str , cnt = 0 ; *p && cnt < n ; p++)
        {
                if (*p == chr) cnt ++;
        }
        if (cnt != n) return NULL;
        else return p-1;
}

static char *_strrfindnchr(str , chr , n)
char *str;
int chr;
int n;
{
        int cnt;
        char *p;

        for (p = str+strlen(str)-1 , cnt = 0 ; p >= str && cnt < n ; p--)
        {
                if (*p == chr) cnt ++;
        }
        if (cnt != n) return NULL;
        else return p+1;
}

#ifdef HAVE_REGEX
static char *lfname_re_sub(lfnamep, urlstr, nr)
lfname *lfnamep;
char *urlstr;
int nr;
{
	char pom[4096];

	pom[0] = '\0';
#ifdef HAVE_POSIX_REGEX
	{
		regmatch_t *pmatch = lfnamep->pmatch;
		if (nr >= 0 && nr <= lfnamep->preg.re_nsub)
		{
			strncpy(pom , urlstr+pmatch[nr].rm_so , pmatch[nr].rm_eo - pmatch[nr].rm_so);
			pom[pmatch[nr].rm_eo - pmatch[nr].rm_so] = '\0';
		}
	}
#endif
#if 0
#ifdef HAVE_V8_REGEX
	{
		char ssect[10];
		if (nr)
			sprintf(ssect , "\\%d" , nr);
		else
			strcpy(ssect , "&");
		regsub(lfnamep->preg , ssect , pom);
	}
#endif
#endif
#ifdef HAVE_GNU_REGEX
	if (nr >= 0 && nr < lfnamep->preg.re_nsub)
	{
		strncpy(pom , urlstr+lfnamep->pmatch.start[nr] , 
			lfnamep->pmatch.end[nr] - lfnamep->pmatch.start[nr]);
		pom[lfnamep->pmatch.end[nr] - lfnamep->pmatch.start[nr]] = '\0';
	}
#endif
	return new_string(pom);
}
#endif

/* $x - x-th match section	*/
/* %i - protocol id		*/
/* %p - password		*/
/* %u - user name		*/
/* %h - host name		*/
/* %m - domain name		*/
/* %r - port number		*/
/* %d - doc path		*/
/* %n - doc name		*/
/* %b - base name of document	*/
/* %e - extension		*/
/* %s - search string		*/
/* %-x - x-th dirname from end	*/
/* %x - x-th dirname from start */

char *lfname_get_by_url(urlp , urlstr , lfnamep)
url *urlp;
char *urlstr;
lfname *lfnamep;
{
	char *ps,*pd,*pp,*p1,*p2;
	char pom[4096];
	char pstr[4096];
	int nr;
	char *n,*d,*t,*e,*b,*m;
	char *retv = NULL;

	p1 = url_get_path(urlp);
	if (urlp->type == URLT_GOPHER)
	{
		if (urlp->p.gopher.selector[0] == '1')
			sprintf(pstr , "/%s/%s" ,
				urlp->p.gopher.selector , cfg.index_name);
		else
			sprintf(pstr , "/%s" , urlp->p.gopher.selector);

	}
	else if (tl_is_dirname(p1) ||
		((urlp->type == URLT_FTP || urlp->type == URLT_FTPS) &&
		 urlp->p.ftp.dir))
	{
		sprintf(pstr , "%s/%s" , p1 , cfg.index_name);
	}
	else strcpy(pstr , p1);

	t = get_abs_file_path(pstr);

	strcpy(pstr , t);

	p1 = strrchr(pstr , '/');

	d = p1 ? new_n_string(pstr , p1 - pstr) : new_string("");

	n = p1 ? new_string(p1+1) : new_string(pstr);

	e = new_string(get_extension(pstr));

	p1 = strrchr(n , '.');

	if (p1) b = new_n_string(n , p1 - n);
	else b = new_string(n);

	m = url_get_site(urlp);
	p1 = strchr(m , '.');
	if (p1) m = p1+1;

	pom[0] = '\0';

	if (lfnamep->transstr[0] == '(')
	{
		struct lfname_lsp_interp interp;
		char port[10];

		interp.urlstr = urlstr;
		interp.scheme = prottable[urlp->type].dirname;
		interp.passwd = url_get_pass(urlp , NULL) ? url_get_pass(urlp, NULL) : "";
		interp.user = url_get_user(urlp, NULL) ? url_get_user(urlp, NULL) : "";
		interp.host = url_get_site(urlp) ? url_get_site(urlp) : "";
		interp.domain = m;
		sprintf(port, "%d" , url_get_port(urlp));
		interp.port = port;
		interp.path = d;
		interp.name = n;
		interp.basename = b;
		interp.extension = e;
		interp.query = url_get_search_str(urlp) ? url_get_search_str(urlp) : "";
		interp.deflt = url_get_default_local_name(urlp);
		interp.orig = lfnamep;

		retv = lfname_lsp_get_by_url(&interp);

		_free(interp.deflt);
	}
	else for (ps = lfnamep->transstr, pd = pom; *ps ; ps++)
	{
		if (!*(ps+1)) 
		{
			*pd = *ps;
			pd++;
			*pd = '\0';
			continue;
		}
		switch (*ps)
		{
			case '\\':
				ps++;
				*pd = *ps;
				pd++;
				*pd = '\0';
			break;
#ifdef HAVE_REGEX
			case '$':
				ps++;
				nr = strtol(ps , &pp , 10);
				p1 = lfname_re_sub(lfnamep, urlstr, nr);
				strcpy(pd, p1);
				_free(p1);
				while (*pd) pd++;
				ps = pp-1;
			break;
#endif
			case '%':
				ps++;
				switch (*ps)
				{
					case 'i': strcpy(pstr, prottable[urlp->type].dirname);
					break;
					case 'p': strcpy(pstr, url_get_pass(urlp , NULL) ? url_get_pass(urlp, NULL) : "");
					break;
					case 'u': strcpy(pstr, url_get_user(urlp , NULL) ? url_get_user(urlp, NULL) : "");
					break;
					case 'h': strcpy(pstr, url_get_site(urlp) ? url_get_site(urlp) : "");
					break;
					case 'm': strcpy(pstr, m);
					break;
					case 'r': sprintf(pstr, "%d" , url_get_port(urlp));
					break;
					case 't': strcpy(pstr, t);
					break;
					case 'd': strcpy(pstr, d);
					break;
					case 'n': strcpy(pstr, n);
					break;
					case 'b': strcpy(pstr, b);
					break;
					case 'e': strcpy(pstr, e);
					break;
					case 's': strcpy(pstr, url_get_search_str(urlp) ? url_get_search_str(urlp) : "");
					break;
					case '-':
						nr = strtol(ps+1 , &pp , 10);
						p1 = _strrfindnchr(d , '/' , nr);
						p2 = _strrfindnchr(d , '/' , nr+1);
						if (!p1) pstr[0] = '\0';
						else if (p2)
						{
							strncpy(pstr , p2+1 , p1-1-p2);
							*(pstr+(p1-1-p2)) = '\0';
						}
						else pstr[0] = '\0';
						ps = pp-1;
					break;
					case '1':
					case '2':
					case '3':
					case '4':
					case '5':
					case '6':
					case '7':
					case '8':
					case '9':
						nr = strtol(ps , &pp , 10);
						p1 = _strfindnchr(d , '/' , nr);
						p2 = _strfindnchr(d , '/' , nr+1);
						if (!p1) pstr[0] = '\0';
						else if (p2)
						{
							strncpy(pstr , p1+1 , p2-1-p1);
							*(pstr+(p2-1-p1)) = '\0';
						}
						else strcpy(pstr , p1+1);
						ps = pp-1;
					break;
					default:
						pstr[0] = *(ps-1);
						pstr[1] = *ps;
						pstr[2] = '\0';
				}
				strcat(pd , pstr);
				while (*pd) pd++;
			break;
			default:
				*pd = *ps;
				pd++;
				*pd = '\0';
		}
		retv = new_string(pom);
	}
	free(e);
	free(n);
	free(t);
	free(d);
	return retv;
}

void lfname_free(lfnamep)
lfname *lfnamep;
{
#ifdef HAVE_REGEX
	if (lfnamep->type == LFNAME_REGEX)
	{
#ifdef HAVE_POSIX_REGEX
		regfree(&(lfnamep->preg));
		_free(lfnamep->pmatch);
#endif
#ifdef HAVE_V8_REGEX
	_free(lfnamep->preg);
#endif
#ifdef HAVE_GNU_REGEX
	regfree(&lfnamep->preg);
	_free(lfnamep->pmatch.start);
	_free(lfnamep->pmatch.end);
#endif
	}
#endif

	_free(lfnamep->matchstr);
	_free(lfnamep->transstr);
	_free(lfnamep);
}

lfname *lfname_new(type , mpt , str)
lfname_type type;
char *mpt;
char *str;
{
	lfname *rv;
	char *p;

	rv = _malloc(sizeof(lfname));
	rv->type = type;
	rv->matchstr = NULL;
	rv->transstr = NULL;
#ifdef HAVE_REGEX
	if (type == LFNAME_REGEX)
	{
#ifdef HAVE_POSIX_REGEX
		int ec;
		if ((ec = regcomp(&(rv->preg) , mpt , REG_EXTENDED)))
		{
			char pom[PATH_MAX];
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , mpt);
			regerror(ec , &(rv->preg) , pom , sizeof(pom));
			xprintf(0 , "%s\n" , pom);
			regfree(&(rv->preg));
			free(rv);
			return NULL;
		}
		rv->pmatch = _malloc((rv->preg.re_nsub + 1) * sizeof(regmatch_t));
#endif
#ifdef HAVE_V8_REGEX
		if (!(rv->preg = regcomp(mpt)))
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , mpt);
			free(rv->preg);
			free(rv);
			return NULL;
		}
#endif
#ifdef HAVE_BSD_REGEX
		if ((p = re_comp(mpt)))
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , mpt);
			xprintf(0, p);
			free(rv);
			return NULL;
		}
#endif
#ifdef HAVE_GNU_REGEX
		rv->preg.allocated = 0;
		rv->preg.buffer = NULL;
		rv->preg.fastmap = NULL;
		re_set_syntax(r_2phase_star);
		if ((p = re_compile_pattern(mpt, strlen(mpt) , &rv->preg)))
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , mpt);
			xprintf(0 , "%s\n", p);
			regfree(&(rv->preg));
			free(rv);
			return NULL;
		}
		rv->pmatch.start = _malloc((rv->preg.re_nsub + 1) * sizeof(*rv->pmatch.start));
		rv->pmatch.end = _malloc((rv->preg.re_nsub + 1) * sizeof(*rv->pmatch.end));
		rv->pmatch.num_regs  = rv->preg.re_nsub + 1;
		rv->preg.regs_allocated = REGS_FIXED;
#endif
	}
#endif
	if (str[0] == '(')
	{
		struct lfname_lsp_var *variant;
		p = str;
		if ((variant = lfname_lsp_analyze(&p)))
		{
			lfname_lsp_var_free(variant);
			if (*p)
			{
				xprintf(0, gettext("LSP analyze error: bad token at - %s\n"), p);
				lfname_free(rv);
				return NULL;
			}
			else
			{
				rv->transstr = new_string(str);
			}
		}
		else
		{
			lfname_free(rv);
			return NULL;
		}
	}
	else
		rv->transstr = new_string(str);
	rv->matchstr = new_string(mpt);
	return rv;
}

int lfname_match(lfnamep , urlstr)
lfname *lfnamep;
char *urlstr;
{
#ifdef HAVE_REGEX
	if (lfnamep->type == LFNAME_REGEX)
#ifdef HAVE_POSIX_REGEX
		return !regexec(&(lfnamep->preg) , urlstr , lfnamep->preg.re_nsub + 1 , lfnamep->pmatch , 0);
#endif
#ifdef HAVE_V8_REGEX
		return regexec(lfnamep->preg , urlstr);
#endif
#ifdef HAVE_BSD_REGEX
	{
		re_comp(lfnamep->matchstr);
		return re_exec(urlstr);
	}
#endif
#ifdef HAVE_GNU_REGEX
		return re_match(&(lfnamep->preg), urlstr, strlen(urlstr), 0 , &lfnamep->pmatch) >= 0;
#endif
	else
#endif
		return !fnmatch(lfnamep->matchstr , urlstr , FNM_PATHNAME);
}


int lfname_check_rule(str)
char *str;
{
	if (str[0] == '(')
	{
		char *p = str;
		struct lfname_lsp_var *variant;

		if ((variant = lfname_lsp_analyze(&p)))
		{
			lfname_lsp_var_free(variant);
			if (*p)
			{
				xprintf(0, gettext("LSP analyze error: bad token at - %s\n"), p);
				return FALSE;
			}
			else
				return TRUE;
		}
		else
			return FALSE;
	}
	return TRUE;
}

int lfname_check_pattern(type , str)
lfname_type type;
char *str;
{
#ifdef HAVE_REGEX
	if (type == LFNAME_REGEX)
	{
#ifdef HAVE_POSIX_REGEX
		int ec;
		char pom[PATH_MAX];
		regex_t preg;

		ec = regcomp(&preg , str , REG_EXTENDED);

		if (ec)
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , str);
			regerror(ec , &preg , pom , sizeof(pom));
			xprintf(0 , "%s\n" , pom);
		}
		regfree(&preg);
		return !ec;
#endif
#ifdef HAVE_V8_REGEX
		regexp *preg;

		preg = regcomp(str);

		if (!preg)
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , str);
		_free(preg);
		return preg != NULL;
#endif
#ifdef HAVE_BSD_REGEX
		char *p;

		p = re_comp(str);

		if (p)
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , str);
			xprintf(0 , p);
		}
		return p == NULL;
#endif
#ifdef HAVE_GNU_REGEX
		char *p;
		struct re_pattern_buffer preg;

		preg.allocated = 0;
		preg.buffer = NULL;
		preg.fastmap = NULL;
		
		if ((p = re_compile_pattern(str, strlen(str) , &preg)))
		{
			xprintf(0 , gettext("Error compiling regular expression : %s\n") , str);
			xprintf(0 , "%s\n" , p);
		}
		regfree(&preg);
		return p == NULL;
#endif
	}
	else
#endif
		return TRUE;
}

struct {
	enum lfname_lsp_type type;
	enum lfname_lsp_type rettype;
	char	*name;
	short	params;
	enum lfname_lsp_type p1type;
	enum lfname_lsp_type p2type;
	enum lfname_lsp_type p3type;
} lfname_lsp_ftbl[] =
{
	{LF_LSP_UNKNOWN, LF_LSP_UNKNOWN, NULL, 0,
		LF_LSP_UNKNOWN, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_STR, LF_LSP_STR, NULL , 0,
		LF_LSP_UNKNOWN, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_NUM, LF_LSP_NUM, NULL , 0,
		LF_LSP_UNKNOWN, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_MACRO, LF_LSP_STR, NULL, 0,
		LF_LSP_UNKNOWN, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_SUB, LF_LSP_STR, "sp ", 1,
		LF_LSP_NUM, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_SC, LF_LSP_STR, "sc " , 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_SS, LF_LSP_STR, "ss " , 3,
		LF_LSP_STR, LF_LSP_NUM, LF_LSP_NUM},
	{LF_LSP_HASH, LF_LSP_NUM, "hsh " , 2,
		LF_LSP_STR, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_MD5, LF_LSP_STR, "md5 " , 1,
		LF_LSP_STR, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_LOWER, LF_LSP_STR, "lo ", 1,
		LF_LSP_STR, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_UPPER, LF_LSP_STR, "up ", 1,
		LF_LSP_STR, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_UENC, LF_LSP_STR, "ue " , 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_DELCHR, LF_LSP_STR, "dc ", 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_TRCHR, LF_LSP_STR, "tc ", 3,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_STR},
	{LF_LSP_TRSTR, LF_LSP_STR, "ts ", 3,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_STR},
	{LF_LSP_STRSPN, LF_LSP_NUM, "spn ", 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_STRCSPN, LF_LSP_NUM, "cspn ", 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_STRLEN, LF_LSP_NUM, "sl ", 1,
		LF_LSP_STR, LF_LSP_UNKNOWN, LF_LSP_UNKNOWN},
	{LF_LSP_NRSTR, LF_LSP_STR, "ns ", 2,
		LF_LSP_STR, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_LCHR, LF_LSP_NUM, "lc ", 2,
		LF_LSP_STR, LF_LSP_STR, LF_LSP_UNKNOWN},
	{LF_LSP_PLS, LF_LSP_NUM , "+ ", 2,
		LF_LSP_NUM, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_MNS, LF_LSP_NUM , "- ", 2,
		LF_LSP_NUM, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_MOD, LF_LSP_NUM , "% ", 2,
		LF_LSP_NUM, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_MUL, LF_LSP_NUM , "* ", 2,
		LF_LSP_NUM, LF_LSP_NUM, LF_LSP_UNKNOWN},
	{LF_LSP_DIV, LF_LSP_NUM , "/ ", 2,
		LF_LSP_NUM, LF_LSP_NUM, LF_LSP_UNKNOWN},
};

static enum lfname_lsp_type lfname_lsp_token_type(pstr)
char **pstr;
{
	char *p = *pstr;
	enum lfname_lsp_type retv = LF_LSP_UNKNOWN;

	while (*p == ' ') p++;

	if (*p == '(')
	{
		int i;
		for (i = 0 ; i < NUM_ELEM(lfname_lsp_ftbl) ; i++)
		{
			if (lfname_lsp_ftbl[i].name &&
				!strncmp(p+1, lfname_lsp_ftbl[i].name, strlen(lfname_lsp_ftbl[i].name)))
			{
				retv = lfname_lsp_ftbl[i].type;
				p += 1 + strlen(lfname_lsp_ftbl[i].name);
				break;
			}
		}
	}
	else if (*p == '\"')
	{
		retv = LF_LSP_STR;
		p++;
	}
	else if (*p == '%')
	{
		retv = LF_LSP_MACRO;
		p++;
	}
	else if (isdigit(*p) || *p == '-')
		retv = LF_LSP_NUM;

	*pstr = p;

	return retv;
}

static struct lfname_lsp_var *lfname_lsp_var_new(type)
enum lfname_lsp_type type;
{
	struct lfname_lsp_var *retv = NULL;

	retv = _malloc(sizeof(struct lfname_lsp_var));
	retv->type = type;
	retv->val.str = NULL;
	retv->rettype = lfname_lsp_ftbl[type].rettype;
	retv->ret_val.str = NULL;
	retv->param1 = NULL;
	retv->param2 = NULL;
	retv->param3 = NULL;

	return retv;
}

static void lfname_lsp_var_ret_free(var)
struct lfname_lsp_var *var;
{
	if (!var) return;

	lfname_lsp_var_ret_free(var->param1);
	lfname_lsp_var_ret_free(var->param2);
	lfname_lsp_var_ret_free(var->param3);
	if (var->rettype == LF_LSP_STR)
		_free(var->ret_val.str);
}

static void lfname_lsp_var_free(var)
struct lfname_lsp_var *var;
{
	if (!var) return;

	lfname_lsp_var_free(var->param1);
	lfname_lsp_var_free(var->param2);
	lfname_lsp_var_free(var->param3);
	if (var->type == LF_LSP_STR)
		_free(var->val.str);
	_free(var);
}

static struct lfname_lsp_var *lfname_lsp_analyze(pstr)
char **pstr;
{
	struct lfname_lsp_var *retv = NULL;
	enum lfname_lsp_type type;
	char *p;

	type = lfname_lsp_token_type(pstr);

	switch (type)
	{
	    case LF_LSP_UNKNOWN:
		xprintf(0, gettext("LSP analyze error: bad token at - %s\n"), *pstr);
	    break;
	    case LF_LSP_NUM:
	    {
		int nval;

		errno = 0;
		nval = strtol(*pstr, &p, 0);

		if ((errno == ERANGE) || (*p != '\0' && *p != ')' && *p != ' '))
		{
			xprintf(0, gettext("LSP analyze error: bad numeric value at - %s\n"), *pstr);
			break;
		}
		retv = lfname_lsp_var_new(type);
		retv->val.num = nval;
		*pstr = p;
	    }
	    break;
	    case LF_LSP_MACRO:
	    {
		p = strchr("ipuhmrdnbesUo", **pstr);

		if (!p || !*p || (*(*pstr+1) != '\0' && *(*pstr+1) != ')' && *(*pstr+1) != ' '))
		{
			xprintf(0, gettext("LSP analyze error: bad macro at - %s\n"), *pstr-1);
			break;
		}
		*pstr += 1;
		retv = lfname_lsp_var_new(type);
		retv->val.macro = *p;
	    }
	    break;
	    case LF_LSP_STR:
	    {
		char *tmp = _malloc(strlen(*pstr)+1);
		char *tp;

		p = *pstr;
		tp = tmp;

		while (*p)
		{
			if (*p == '\"') break;
			if (*p == '\\') p++;
			*tp = *p;
			tp++;
			if (*p) p++;
		}
		*tp = '\0';

		if (*p != '\"')
		{
			xprintf(0, gettext("LSP analyze error: unterminated string at - %s\n"), *pstr-1);
			break;
		}
		retv = lfname_lsp_var_new(type);
		retv->val.str = new_string(tmp);
		_free(tmp);
		*pstr = p + 1;
	    }
	    break;
	    default:
	    {
		struct lfname_lsp_var *p1 = NULL;
		struct lfname_lsp_var *p2 = NULL;
		struct lfname_lsp_var *p3 = NULL;
		if (lfname_lsp_ftbl[type].params >= 1)
		{
			p = *pstr;
			p1 = lfname_lsp_analyze(pstr);
			if (!p1) break;
			if (p1->rettype != lfname_lsp_ftbl[type].p1type)
			{
				xprintf(0, gettext("LSP analyze error: bad parameter type at - %s\n"), p);
				lfname_lsp_var_free(p1);
				break;
			}
		}
		if (p1 && lfname_lsp_ftbl[type].params >= 2)
		{
			p = *pstr;
			p2 = lfname_lsp_analyze(pstr);
			if (!p2)
			{
				lfname_lsp_var_free(p1);
				break;
			}
			if (p2->rettype != lfname_lsp_ftbl[type].p2type)
			{
				xprintf(0, gettext("LSP analyze error: bad parameter type at - %s\n"), p);
				lfname_lsp_var_free(p1);
				lfname_lsp_var_free(p2);
				break;
			}
		}
		if (p2 && lfname_lsp_ftbl[type].params >= 3)
		{
			p = *pstr;
			p3 = lfname_lsp_analyze(pstr);
			if (!p3)
			{
				lfname_lsp_var_free(p1);
				lfname_lsp_var_free(p2);
				break;
			}
			if (p3->rettype != lfname_lsp_ftbl[type].p3type)
			{
				xprintf(0, gettext("LSP analyze error: bad parameter type at - %s\n"), p);
				lfname_lsp_var_free(p1);
				lfname_lsp_var_free(p2);
				lfname_lsp_var_free(p3);
				break;
			}
		}
		while (**pstr == ' ') (*pstr)++;
		if (**pstr != ')')
		{
			xprintf(0, gettext("LSP analyze error: bad token at - %s\n"), *pstr);
			if (p1) lfname_lsp_var_free(p1);
			if (p2) lfname_lsp_var_free(p2);
			if (p3) lfname_lsp_var_free(p3);
		}
		else
		{
			(*pstr) ++;
			retv = lfname_lsp_var_new(type);
			retv->param1 = p1;
			retv->param2 = p2;
			retv->param3 = p3;
		}
	    }
	    break;
	}

	return retv;
}

static int lfname_lsp_eval(interp, var)
struct lfname_lsp_interp *interp;
struct lfname_lsp_var *var;
{
	if (var->param1)
		lfname_lsp_eval(interp, var->param1);
	if (var->param2)
		lfname_lsp_eval(interp, var->param2);
	if (var->param3)
		lfname_lsp_eval(interp, var->param3);

	switch(var->type)
	{
	    case LF_LSP_UNKNOWN:
	    break;
	    case LF_LSP_STR:
		var->ret_val.str = new_string(var->val.str);
	    break;
	    case LF_LSP_NUM:
		var->ret_val.num = var->val.num;
	    break;
	    case LF_LSP_MACRO:
		switch (var->val.macro)
		{
		    case 'i':
			var->ret_val.str = new_string(interp->scheme);
		    break;
		    case 'p':
			var->ret_val.str = new_string(interp->passwd);
		    break;
		    case 'u':
			var->ret_val.str = new_string(interp->user);
		    break;
		    case 'h':
			var->ret_val.str = new_string(interp->host);
		    break;
		    case 'm':
			var->ret_val.str = new_string(interp->domain);
		    break;
		    case 'r':
			var->ret_val.str = new_string(interp->port);
		    break;
		    case 'd':
			var->ret_val.str = new_string(interp->path);
		    break;
		    case 'n':
			var->ret_val.str = new_string(interp->name);
		    break;
		    case 'b':
			var->ret_val.str = new_string(interp->basename);
		    break;
		    case 'e':
			var->ret_val.str = new_string(interp->extension);
		    break;
		    case 's':
			var->ret_val.str = new_string(interp->query);
		    break;
		    case 'U':
			var->ret_val.str = new_string(interp->urlstr);
		    break;
		    case 'o':
			var->ret_val.str = new_string(interp->deflt);
		    break;
		}
	    break;
	    case LF_LSP_SUB:
		var->ret_val.str = lfname_re_sub(interp->orig,
				interp->urlstr,
				var->param1->ret_val.num);
	    break;
	    case LF_LSP_SC:
	    {
		char *p;
		p = _malloc(strlen(var->param1->ret_val.str)+strlen(var->param2->ret_val.str)+1);
		strcpy(p, var->param1->ret_val.str);
		strcat(p, var->param2->ret_val.str);
		var->ret_val.str = p;
	    }
	    break;
	    case LF_LSP_SS:
	    {
		char *p;
		p = var->param1->ret_val.str;
		if (var->param2->ret_val.num > 0)
			p += (strlen(p) >= var->param2->ret_val.num) ? var->param2->ret_val.num : strlen(p);
		if (var->param3->ret_val.num > 0)
			var->ret_val.str = new_n_string(p, var->param3->ret_val.num);
		else
			var->ret_val.str = new_string(p);
	    }
	    break;
	    case LF_LSP_HASH:
		var->ret_val.num = str_hash_func(var->param2->ret_val.num,
					var->param1->ret_val.str);
	    break;
	    case LF_LSP_MD5:
		var->ret_val.str = _md5(var->param1->ret_val.str);
	    break;
	    case LF_LSP_LOWER:
		var->ret_val.str = lowerstr(var->param1->ret_val.str);
	    break;
	    case LF_LSP_UPPER:
		var->ret_val.str = upperstr(var->param1->ret_val.str);
	    break;
	    case LF_LSP_UENC:
		var->ret_val.str = url_encode_str(var->param1->ret_val.str,
					var->param2->ret_val.str);
	    break;
	    case LF_LSP_DELCHR:
		var->ret_val.str = tr_del_chr(var->param2->ret_val.str,
					var->param1->ret_val.str);
	    break;
	    case LF_LSP_TRCHR:
		var->ret_val.str = tr_chr_chr(var->param2->ret_val.str,
					var->param3->ret_val.str,
					var->param1->ret_val.str);
	    break;
	    case LF_LSP_TRSTR:
		var->ret_val.str = tr_str_str(var->param2->ret_val.str,
					var->param3->ret_val.str,
					var->param1->ret_val.str);
	    break;
	    case LF_LSP_STRSPN:
		var->ret_val.num = strspn(var->param1->ret_val.str,
					var->param2->ret_val.str);
	    break;
	    case LF_LSP_STRCSPN:
		var->ret_val.num = strcspn(var->param1->ret_val.str,
					var->param2->ret_val.str);
	    break;
	    case LF_LSP_STRLEN:
		var->ret_val.num = strlen(var->param1->ret_val.str);
	    break;
	    case LF_LSP_NRSTR:
	    {
		char pom[1024];
		sprintf(pom, var->param1->ret_val.str, var->param2->ret_val.num);
		var->ret_val.str = new_string(pom);
	    }
	    break;
	    case LF_LSP_LCHR:
	    {
		char *p;
		p = strrchr(var->param1->ret_val.str, *var->param2->ret_val.str);
		var->ret_val.num = p ? p - var->param1->ret_val.str : 0;
	    }
	    break;
	    case LF_LSP_PLS:
		var->ret_val.num = var->param1->ret_val.num + var->param2->ret_val.num;
	    break;
	    case LF_LSP_MNS:
		var->ret_val.num = var->param1->ret_val.num - var->param2->ret_val.num;
	    break;
	    case LF_LSP_MOD:
		var->ret_val.num = var->param1->ret_val.num % var->param2->ret_val.num;
	    break;
	    case LF_LSP_MUL:
		var->ret_val.num = var->param1->ret_val.num * var->param2->ret_val.num;
	    break;
	    case LF_LSP_DIV:
		var->ret_val.num = var->param1->ret_val.num / var->param2->ret_val.num;
	    break;
	}
	if (var->rettype == LF_LSP_STR && !var->ret_val.str)
		var->ret_val.str = new_string("");

	return 0;
}

static char *lfname_lsp_get_by_url(interp)
struct lfname_lsp_interp *interp;
{
	char *retv = NULL;
	struct lfname_lsp_var *variant;
	char *p;
	p = interp->orig->transstr;

	variant = lfname_lsp_analyze(&p);

	if (variant)
	{
		lfname_lsp_eval(interp, variant);
		if (variant->rettype == LF_LSP_NUM)
		{
			char nr[10];
			sprintf(nr, "%d", variant->ret_val.num);
			retv = new_string(nr);
		}
		else
		{
			retv = new_string(variant->ret_val.str);
		}
		lfname_lsp_var_free(variant);
	}

	return retv;
}
