/***************************************************************************

Mr. Do Wild Ride / Mr. Do Run Run memory map (preliminary)

0000-1fff ROM
2000-37ff RAM
3800-39ff Sprites
4000-9fff ROM
a000-a008 Shared with second CPU
b000-b3ff Video RAM
b400-b7ff Color RAM

write:
a800      Watchdog reset?
b800      Trigger NMI on second CPU (?)

SECOND CPU:
0000-3fff ROM
8000-87ff RAM

read:
e000-e008 data from first CPU
c003      bit 0-3 = joystick
          bit 4-7 = ?
c005      bit 0 = fire
          bit 1 = fire (again?!)
		  bit 2 = ?
		  bit 3 = START 1
		  bit 4-6 = ?
		  bit 4 = START 2
c081      coins per play

write:
e000-e008 data for first CPU
a000      sound port 1
a400      sound port 2
a800      sound port 3
ac00      sound port 4

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"



int docastle_shared0_r(int offset);
int docastle_shared1_r(int offset);
void docastle_shared0_w(int offset,int data);
void docastle_shared1_w(int offset,int data);
void docastle_nmitrigger(int offset,int data);

void dowild_vh_convert_color_prom(unsigned char *palette, unsigned char *colortable,const unsigned char *color_prom);
int docastle_vh_start(void);
void docastle_vh_stop(void);
void docastle_vh_screenrefresh(struct osd_bitmap *bitmap);

void docastle_sound1_w(int offset,int data);
void docastle_sound2_w(int offset,int data);
void docastle_sound3_w(int offset,int data);
void docastle_sound4_w(int offset,int data);
int docastle_sh_start(void);
void docastle_sh_stop(void);
void docastle_sh_update(void);



static struct MemoryReadAddress readmem[] =
{
	{ 0x2000, 0x37ff, MRA_RAM },
	{ 0x0000, 0x1fff, MRA_ROM },
	{ 0x4000, 0x9fff, MRA_ROM },
	{ 0xa000, 0xa008, docastle_shared0_r },
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress writemem[] =
{
	{ 0x2000, 0x37ff, MWA_RAM },
	{ 0x3800, 0x39ff, MWA_RAM, &spriteram, &spriteram_size },
	{ 0xb000, 0xb3ff, videoram_w, &videoram, &videoram_size },
	{ 0xb400, 0xb7ff, colorram_w, &colorram },
	{ 0xa000, 0xa008, docastle_shared1_w },
	{ 0xb800, 0xb800, docastle_nmitrigger },
	{ 0xa800, 0xa800, MWA_NOP },
	{ 0x0000, 0x1fff, MWA_ROM },
	{ 0x4000, 0x9fff, MWA_ROM },
	{ -1 }	/* end of table */
};


static struct MemoryReadAddress readmem2[] =
{
	{ 0x8000, 0x87ff, MRA_RAM },
	{ 0xc003, 0xc003, input_port_0_r },
	{ 0xc083, 0xc083, input_port_0_r },
	{ 0xc005, 0xc005, input_port_1_r },
	{ 0xc085, 0xc085, input_port_1_r },
	{ 0xc007, 0xc007, input_port_2_r },
	{ 0xc087, 0xc087, input_port_2_r },
	{ 0xc002, 0xc002, input_port_3_r },
	{ 0xc082, 0xc082, input_port_3_r },
	{ 0xc001, 0xc001, input_port_4_r },
	{ 0xc081, 0xc081, input_port_4_r },
	{ 0xe000, 0xe008, docastle_shared1_r },
	{ 0x0000, 0x3fff, MRA_ROM },
	{ -1 }	/* end of table */
};

static struct MemoryWriteAddress writemem2[] =
{
	{ 0x8000, 0x87ff, MWA_RAM },
	{ 0xe000, 0xe008, docastle_shared0_w },
	{ 0xa000, 0xa000, docastle_sound1_w },
	{ 0xa400, 0xa400, docastle_sound2_w },
	{ 0xa800, 0xa800, docastle_sound3_w },
	{ 0xac00, 0xac00, docastle_sound4_w },
	{ 0x0000, 0x3fff, MWA_ROM },
	{ -1 }	/* end of table */
};



static struct InputPort input_ports[] =
{
	{	/* IN0 */
		0xff,
		{ OSD_KEY_RIGHT, OSD_KEY_UP, OSD_KEY_LEFT, OSD_KEY_DOWN,
				OSD_KEY_Q, OSD_KEY_W, OSD_KEY_E, OSD_KEY_R },
		{ OSD_JOY_RIGHT, OSD_JOY_UP, OSD_JOY_LEFT, OSD_JOY_DOWN,
				0, 0, 0, 0 }
	},
	{	/* IN1 */
		0xff,
		{ OSD_KEY_LCONTROL, 0, 0, OSD_KEY_1,
				0, 0, 0, OSD_KEY_2 },
		{ OSD_JOY_FIRE, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* IN2 */
		0xff,
		{ OSD_KEY_T, OSD_KEY_C, OSD_KEY_5, OSD_KEY_Z, OSD_KEY_4, OSD_KEY_3, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSWA */
		0xdf,
		{ 0, 0, OSD_KEY_F1, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* COIN */
		0xff,
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{ -1 }	/* end of table */
};

static struct TrakPort trak_ports[] =
{
        { -1 }
};


static struct KEYSet dowild_keys[] =
{
	{ 0, 1, "MOVE UP" },
	{ 0, 2, "MOVE LEFT"  },
	{ 0, 0, "MOVE RIGHT" },
	{ 0, 3, "MOVE DOWN" },
	{ 1, 0, "ACCELERATE" },
	{ -1 }
};

static struct KEYSet dorunrun_keys[] =
{
	{ 0, 1, "MOVE UP" },
	{ 0, 2, "MOVE LEFT"  },
	{ 0, 0, "MOVE RIGHT" },
	{ 0, 3, "MOVE DOWN" },
	{ 1, 0, "FIRE" },
	{ -1 }
};

static struct KEYSet kickridr_keys[] =
{
	{ 0, 1, "MOVE UP" },
	{ 0, 2, "MOVE LEFT"  },
	{ 0, 0, "MOVE RIGHT" },
	{ 0, 3, "MOVE DOWN" },
	{ 1, 0, "KICK" },
	{ -1 }
};


static struct DSW dsw[] =
{
	{ 3, 0x80, "LIVES", { "5", "3" }, 1 },
	{ 3, 0x03, "DIFFICULTY", { "HARDEST", "HARD", "MEDIUM", "EASY" }, 1 },
	{ 3, 0x10, "EXTRA", { "HARD", "EASY" }, 1 },
	{ 3, 0x40, "SPECIAL", { "NOT GIVEN", "GIVEN" }, 1 },
	{ 3, 0x08, "SW4", { "ON", "OFF" }, 1 },
	{ -1 }
};



static struct GfxLayout charlayout =
{
	8,8,    /* 8*8 characters */
	512,    /* 512 characters */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 }, /* the bitplanes are packed in one nibble */
	{ 0, 4, 8, 12, 16, 20, 24, 28 },
	{ 0*32, 1*32, 2*32, 3*32, 4*32, 5*32, 6*32, 7*32 },
	32*8   /* every char takes 32 consecutive bytes */
};
static struct GfxLayout spritelayout =
{
	16,16,  /* 16*16 sprites */
	256,    /* 256 sprites */
	4,      /* 4 bits per pixel */
	{ 0, 1, 2, 3 }, /* the bitplanes are packed in one nibble */
	{ 0, 4, 8, 12, 16, 20, 24, 28,
			32, 36, 40, 44, 48, 52, 56, 60 },
	{ 0*64, 1*64, 2*64, 3*64, 4*64, 5*64, 6*64, 7*64,
			8*64, 9*64, 10*64, 11*64, 12*64, 13*64, 14*64, 15*64 },
	128*8  /* every sprite takes 128 consecutive bytes */
};



static struct GfxDecodeInfo gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,       0, 64 },
	{ 1, 0x4000, &spritelayout, 64*16, 32 },
	{ -1 } /* end of array */
};



static unsigned char dowild_color_prom[] =
{
	0x00,0xB6,0x92,0x6F,0x4A,0x42,0x00,0x00,0x80,0xD8,0x90,0x48,0x4C,0xFF,0xE1,0x77,
	0x80,0xF9,0xF5,0xED,0x4A,0xFF,0x81,0xD3,0x80,0x1C,0x14,0x0C,0x4A,0xFF,0x81,0x77,
	0x80,0xD9,0xB5,0x91,0x4A,0xFF,0x81,0xD3,0x80,0x1B,0x13,0x0B,0x4A,0xFF,0x81,0x77,
	0xC1,0x1E,0x16,0x0E,0x4A,0xFF,0x49,0xD3,0x00,0xFC,0xE0,0x1C,0xB4,0x0F,0xF2,0x00,
	0x00,0xA4,0x8C,0x60,0xFF,0x1C,0xF0,0x00,0x00,0x1C,0xF0,0xF8,0x82,0xA2,0x65,0x00,
	0xFF,0xFC,0xF4,0xE0,0x17,0x60,0x1C,0x00,0xFF,0xFC,0xF4,0xE0,0x17,0x60,0x1C,0x77,
	0xFF,0xFC,0xF4,0xE0,0x17,0x60,0x1C,0x00,0xE0,0x14,0x00,0x92,0xFC,0x13,0x8C,0x00,
	0x00,0xFC,0xE0,0x1B,0xF4,0x0F,0xFF,0xFC,0x90,0x18,0xE0,0xFC,0x6E,0xFF,0x00,0x00,
	0x00,0x92,0xF4,0x60,0x10,0xFC,0xEC,0x00,0x00,0xE0,0x80,0xB6,0x17,0xEC,0xFF,0xB6,
	0x00,0xA0,0x6E,0x14,0xFF,0xD3,0x13,0xE0,0x00,0xA0,0x6E,0x14,0xFF,0x77,0x13,0xE0,
	0x00,0xE4,0xC0,0xF8,0x60,0x0F,0xFF,0x00,0x4A,0x92,0xB6,0xFF,0xFC,0xF0,0xE0,0x00,
	0x1F,0xFC,0xF0,0xE2,0xE0,0x6E,0xFD,0x00,0x00,0xE0,0x80,0xB6,0x17,0xEC,0x6E,0x1F,
	0x00,0xE8,0xE0,0xF8,0xFC,0x0F,0xFF,0x88,0x61,0x40,0x4A,0x6E,0xB6,0xFF,0xF4,0xF0,
	0x00,0x46,0x92,0x6E,0x49,0xFC,0xB4,0x1B,0x00,0xE0,0x00,0xFC,0x0F,0x00,0x00,0xE2,
	0x00,0x20,0x40,0x60,0x80,0xA0,0xC0,0xE0,0x00,0x04,0x08,0x0C,0x10,0x14,0x18,0x1C,
	0x00,0x00,0x01,0x01,0x02,0x02,0x03,0x03,0x00,0x24,0x49,0x6D,0x92,0xB6,0xDB,0xFF
};

static unsigned char dorunrun_color_prom[] =
{
	0x00,0xFF,0xFC,0xE8,0xE0,0xC0,0xA0,0xFC,0x00,0x14,0x10,0x0C,0xFC,0xC0,0xFF,0x00,
	0x00,0x26,0x27,0x4B,0xF9,0xFF,0xE0,0xEA,0x00,0x26,0x27,0x4B,0xE0,0xFC,0x9C,0xFC,
	0x00,0xEB,0xFC,0x13,0x0F,0x0B,0xFF,0x6E,0x00,0x42,0xFF,0xFC,0xF4,0x1A,0x17,0x0F,
	0x00,0x42,0xFF,0xFC,0xF4,0xF8,0xD4,0xB0,0x00,0x42,0xFF,0xFC,0xF4,0xF6,0xD2,0xAE,
	0x00,0x42,0xFF,0xFC,0xF4,0x9C,0x78,0x54,0x00,0x42,0xFF,0xFC,0xF4,0x9E,0x7A,0x56,
	0x00,0x42,0xFF,0xFC,0xF4,0xD8,0xB4,0x90,0x00,0x42,0xFF,0xFC,0xF4,0xF7,0xD3,0xAF,
	0x00,0x42,0xFF,0xFC,0xF4,0x59,0x35,0x11,0x00,0x42,0xFF,0xFC,0xF4,0xF2,0xCE,0xAA,
	0x00,0x42,0xFF,0xFC,0xF4,0xD9,0xB5,0x91,0x00,0x42,0xFF,0xFC,0xF4,0xA1,0x81,0x61,
	0x00,0x42,0x18,0xF8,0xE4,0xB6,0x92,0x6E,0x00,0x42,0xFC,0xDC,0x9C,0xB6,0x92,0x6E,
	0x00,0x42,0xFC,0xE0,0x14,0xB6,0x92,0x6E,0x00,0x42,0xDE,0x19,0x15,0xB6,0x92,0x6E,
	0x00,0xA0,0x0F,0xFF,0xE0,0xEC,0xFC,0x0F,0x1C,0x14,0x0C,0xF4,0xD0,0xAC,0x88,0x00,
	0xE4,0xF8,0x88,0xB4,0x0B,0x7F,0x6E,0x17,0xFD,0xF8,0xB0,0xE4,0xC0,0x80,0x40,0x00,
	0x40,0x60,0x80,0xA0,0xF7,0xEF,0xFF,0x00,0x00,0x13,0x52,0xE0,0xFF,0x63,0x1C,0x9C,
	0x00,0xC1,0x52,0xFC,0xF7,0xEF,0xFB,0x17,0xE4,0xF2,0xF0,0xFC,0x7F,0x1F,0x13,0x00,
	0x00,0x20,0x40,0x60,0x80,0xA0,0xC0,0xE0,0x00,0x04,0x08,0x0C,0x10,0x14,0x18,0x1C,
	0x00,0x00,0x01,0x01,0x02,0x02,0x03,0x03,0x00,0x24,0x49,0x6D,0x92,0xB6,0xDB,0xFF
};

static unsigned char kickridr_color_prom[] =
{
	0x00,0x80,0x60,0x40,0xE0,0xC0,0xA0,0xFC,0xE0,0xC0,0xA0,0xF0,0x6E,0x00,0xFF,0x00,
	0x96,0x14,0xFF,0x08,0x40,0xAC,0x1B,0x00,0x0F,0x07,0x03,0xF0,0x02,0x00,0xFF,0x00,
	0xD0,0x84,0x64,0x44,0x00,0x00,0xA4,0x00,0xF8,0xF0,0x0A,0xF4,0x00,0xFF,0x00,0x00,
	0x92,0x14,0xFF,0x08,0x40,0xAC,0x0F,0xEA,0x72,0x14,0xFF,0x08,0x40,0xAC,0x1B,0x00,
	0x0F,0x07,0xDB,0x4A,0x00,0xF4,0xFF,0x00,0x89,0x81,0xDA,0x4A,0x00,0xF4,0xFF,0x00,
	0xE0,0xFC,0xA0,0x00,0xFF,0x6E,0x00,0x00,0x09,0x05,0x01,0x00,0xFF,0x6E,0xF0,0x00,
	0x04,0x14,0x10,0x08,0x04,0x40,0x00,0x00,0xE0,0xFF,0x92,0x80,0x01,0xB6,0x40,0x20,
	0xFF,0xE0,0x92,0x80,0xF0,0xB6,0x40,0x20,0xF0,0xD0,0x00,0x00,0x00,0x00,0x00,0x00,
	0x0F,0xE2,0xFF,0xFC,0x1B,0x1F,0xFF,0x00,0xFF,0xF0,0xA0,0xE8,0xFC,0x00,0x6E,0x00,
	0xDA,0x6E,0x00,0xFC,0x4A,0xFF,0xFF,0x00,0x16,0xB6,0x92,0x1C,0xFF,0xE0,0xFC,0x00,
	0x0F,0x4A,0xE0,0xA0,0xF0,0x00,0xFF,0x00,0xC0,0xA0,0x80,0x60,0x00,0xDA,0xFF,0x00,
	0xE0,0xA0,0xDB,0x4A,0x00,0xF0,0xFF,0x00,0xB6,0x92,0x6E,0x00,0x00,0xF4,0xFF,0x00,
	0xF0,0xD0,0xDB,0xFF,0xFF,0x0C,0x92,0x6E,0xFC,0xFF,0x00,0x02,0x01,0xB6,0x92,0x6E,
	0xE0,0xFF,0xAC,0x80,0x1B,0x00,0x80,0x40,0xFF,0xE0,0xAC,0x80,0xF4,0x00,0x80,0x40,
	0x00,0x20,0x40,0x60,0x80,0xA0,0xC0,0xE0,0x00,0x04,0x08,0x0C,0x10,0x14,0x18,0x1C,
	0x00,0x00,0x01,0x01,0x02,0x02,0x03,0x03,0x00,0x24,0x49,0x6D,0x92,0xB6,0xDB,0xFF
};



static struct MachineDriver machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_Z80,
			4000000,	/* 4 Mhz ? */
			0,
			readmem,writemem,0,0,
			interrupt,1
		},
		{
			CPU_Z80,
			4000000,	/* 4 Mhz ??? */
			2,	/* memory region #2 */
			readmem2, writemem2,0,0,
			interrupt,8
		}
	},
	60,
	100,	/* 100 CPU slices per frame - an high value to ensure proper */
			/* synchronization of the CPUs */
	0,

	/* video hardware */
	32*8, 32*8, { 1*8, 31*8-1, 4*8, 28*8-1 },
	gfxdecodeinfo,
	256, 96*16,
	dowild_vh_convert_color_prom,

	VIDEO_TYPE_RASTER|VIDEO_SUPPORTS_DIRTY,
	0,
	docastle_vh_start,
	docastle_vh_stop,
	docastle_vh_screenrefresh,

	/* sound hardware */
	0,
	0,
	docastle_sh_start,
	docastle_sh_stop,
	docastle_sh_update
};



/***************************************************************************

  Game driver(s)

***************************************************************************/

ROM_START( dowild_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "W1",  0x0000, 0x2000, 0xaeac2454 )
	ROM_LOAD( "W3",  0x4000, 0x2000, 0x54fca40a )
	ROM_LOAD( "W4",  0x6000, 0x2000, 0xb549e115 )
	ROM_LOAD( "W2",  0x8000, 0x2000, 0x443cb69e )

	ROM_REGION(0xc000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "W5",  0x0000, 0x4000, 0x2df6441e )
	ROM_LOAD( "W6",  0x4000, 0x2000, 0x2b17a47b )
	ROM_LOAD( "W7",  0x6000, 0x2000, 0xa68f1277 )
	ROM_LOAD( "W8",  0x8000, 0x2000, 0xff188768 )
	ROM_LOAD( "W9",  0xa000, 0x2000, 0xfec01f0c )

	ROM_REGION(0x10000)	/* 64k for the second CPU */
	ROM_LOAD( "W10", 0x0000, 0x4000, 0x431c3ca4 )
ROM_END


ROM_START( dorunrun_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "2764.P1",  0x0000, 0x2000, 0x8ce02038 )
	ROM_LOAD( "2764.L1",  0x4000, 0x2000, 0x8a8056e4 )
	ROM_LOAD( "2764.K1",  0x6000, 0x2000, 0xafdb2121 )
	ROM_LOAD( "2764.N1",  0x8000, 0x2000, 0xffaac0aa )

	ROM_REGION(0xc000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "27128.A3", 0x0000, 0x4000, 0x62269746 )
	ROM_LOAD( "2764.M4",  0x4000, 0x2000, 0xd51d97c3 )
	ROM_LOAD( "2764.L4",  0x6000, 0x2000, 0x423b4513 )
	ROM_LOAD( "2764.J4",  0x8000, 0x2000, 0xe53d66a5 )
	ROM_LOAD( "2764.H4",  0xa000, 0x2000, 0xf652c7f0 )

	ROM_REGION(0x10000)	/* 64k for the second CPU */
	ROM_LOAD( "27128.P7", 0x0000, 0x4000, 0x0f9be91f )
ROM_END

ROM_START( kickridr_rom )
	ROM_REGION(0x10000)	/* 64k for code */
	ROM_LOAD( "K1",  0x0000, 0x2000, 0xbc77e7f3 )
	ROM_LOAD( "K3",  0x4000, 0x2000, 0x299a68c4 )
	ROM_LOAD( "K4",  0x6000, 0x2000, 0xe4e2c6a0 )
	ROM_LOAD( "K2",  0x8000, 0x2000, 0x1325e773 )

	ROM_REGION(0xc000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "K5",  0x0000, 0x4000, 0x197f8237 )
	ROM_LOAD( "K6",  0x4000, 0x2000, 0xdc19cfbf )
	ROM_LOAD( "K7",  0x6000, 0x2000, 0x2f8ac3fe )
	ROM_LOAD( "K8",  0x8000, 0x2000, 0x88eecb70 )
	ROM_LOAD( "K9",  0xa000, 0x2000, 0x50dc5584 )

	ROM_REGION(0x10000)	/* 64k for the second CPU */
	ROM_LOAD( "K10", 0x0000, 0x4000, 0xd77bd167 )
ROM_END



static int dowild_hiload(void)
{
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x2020],"\x01\x00\x00",3) == 0 &&
			memcmp(&RAM[0x2068],"\x01\x00\x00",3) == 0)
	{
		void *f;


		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x2020],10*8);
			osd_fclose(f);
		}

		return 1;
	}
	else return 0;	/* we can't load the hi scores yet */
}



static void dowild_hisave(void)
{
	void *f;
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x2020],10*8);
		osd_fclose(f);
	}
}



static int dorunrun_hiload(void)
{
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	/* check if the hi score table has already been initialized */
	if (memcmp(&RAM[0x2010],"\x00\x10\x00",3) == 0 &&
			memcmp(&RAM[0x2198],"\x00\x10\x00",3) == 0)
	{
		void *f;


		if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,0)) != 0)
		{
			osd_fread(f,&RAM[0x2010],50*8);
			osd_fclose(f);
		}

		return 1;
	}
	else return 0;	/* we can't load the hi scores yet */
}



static void dorunrun_hisave(void)
{
	void *f;
	/* get RAM pointer (this game is multiCPU, we can't assume the global */
	/* RAM pointer is pointing to the right place) */
	unsigned char *RAM = Machine->memory_region[0];


	if ((f = osd_fopen(Machine->gamedrv->name,0,OSD_FILETYPE_HIGHSCORE,1)) != 0)
	{
		osd_fwrite(f,&RAM[0x2010],50*8);
		osd_fclose(f);
	}
}



struct GameDriver dowild_driver =
{
	"Mr. Do! Wild Ride",
	"dowild",
	"MIRKO BUFFONI\nNICOLA SALMORIA\nGARY WALTON\nSIMON WALLS",
	&machine_driver,

	dowild_rom,
	0, 0,
	0,

	input_ports, 0, trak_ports, dsw, dowild_keys,

	dowild_color_prom, 0, 0,
	ORIENTATION_DEFAULT,

	dowild_hiload, dowild_hisave
};

struct GameDriver dorunrun_driver =
{
	"Mr. Do! Run Run",
	"dorunrun",
	"MIRKO BUFFONI\nNICOLA SALMORIA\nGARY WALTON\nSIMON WALLS",
	&machine_driver,

	dorunrun_rom,
	0, 0,
	0,

	input_ports, 0, trak_ports, dsw, dorunrun_keys,

	dorunrun_color_prom, 0, 0,
	ORIENTATION_DEFAULT,

	dorunrun_hiload, dorunrun_hisave
};

struct GameDriver kickridr_driver =
{
	"Kick Rider",
	"kickridr",
	"MIRKO BUFFONI\nNICOLA SALMORIA\nGARY WALTON\nSIMON WALLS",
	&machine_driver,

	kickridr_rom,
	0, 0,
	0,

	input_ports, 0, trak_ports, dsw, kickridr_keys,

	kickridr_color_prom, 0, 0,
	ORIENTATION_DEFAULT,

	dowild_hiload, dowild_hisave
};
