/*
 * $Id: uzip2mem.c $
 *
 * Author: Tomi Ollila <too@iki.fi>
 *
 * 	Copyright (c) 1997 Tomi Ollila
 * 	    All rights reserved
 *
 * Created: Thu Jul 31 16:21:43 1997 too
 * Last modified: Sun Nov  9 13:10:20 1997 too
 *
 * HISTORY 
 * $Log: $
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <fcntl.h>

#include "trace.h"

#include "uzip2mem.h"

#include "uzipsupp.h"

#include "zlib.h"

#define LOCAL_HDR_SIG_1 '\003'
#define LOCAL_HDR_SIG_2 '\004'

#define CENTRAL_HDR_SIG_1 '\001'
#define CENTRAL_HDR_SIG_2 '\002'

#define END_CENTRAL_SIG_1 '\005'
#define END_CENTRAL_SIG_2 '\006'

/* stuff grabbed from unzip-5.2 sources */

#      define L_COMPRESSION_METHOD              4

#      define L_CRC32                           10
#      define L_COMPRESSED_SIZE                 14
#      define L_UNCOMPRESSED_SIZE               18
#      define L_FILENAME_LENGTH                 22
#      define L_EXTRA_FIELD_LENGTH              24


struct EX
{
  long csize;
  long crc32;
  short compr;
  short exlen;
};

#ifndef O_BINARY
#define O_BINARY 0
#endif

int unstore_file(struct NeedData * nd, char * out, int fsize);
int inflate_file(struct NeedData * nd, char * out, int csize, int fsize);

void free_uzip2mem(struct FE * fehead)
{
  struct FE * curr;

  while (fehead)
  {
    curr = fehead;
    fehead = fehead->next;
    free(curr);
  }
}

static int uzip2mem2(const char * filename,
		     struct NeedData * nd, struct FE ** fetail);

struct FE * uzip2mem(const char * filename)
{
  int rv;
  struct NeedData * nd;
  struct FE * fehead = NULL;

  if ((nd = createNeedData(32768)) == NULL)
    return NULL; /* failed */

  rv = uzip2mem2(filename, nd, &fehead);

  deleteNeedData(nd);
  
  if (rv) {
    free_uzip2mem(fehead);
    return NULL; /* failed */
  }

  return fehead;
}
  
int uzip2mem2(const char * filename, struct NeedData * nd, struct FE ** fetail)
{
  int fd;
  
  if ((fd = open(filename, O_RDONLY|O_BINARY)) < 0) {
    if (0) perrf("Can not open %s", filename);
    return FAIL;
  }

  initNeedData(nd, fd); /* currently `deleteNeedData closes fd ... */
  
  while (1)
  {
    struct FE fe;
    struct FE * fbuf;
    struct EX ex;
    int i;
    
    char * p = needData(nd, 30);

    if (!p)
      return FAIL;
    
    if (!(p[0] == 'P' && p[1] == 'K')) {
      errf("Garbage in input file %s\n", filename);
      return FAIL;
    }

#if 0    
    printf("%d %d\n", p[2], p[3]);
#endif    
    
    if ((p[2] == CENTRAL_HDR_SIG_1 && p[3] == CENTRAL_HDR_SIG_2) ||
	(p[2] == END_CENTRAL_SIG_1 && p[3] == END_CENTRAL_SIG_2))
      return SUCC; /* SUCCESSFUL EXIT */

    if (p[2] != LOCAL_HDR_SIG_1 || p[3] != LOCAL_HDR_SIG_2) {
      
      errf("Garbage in input file %s\n", filename);
      return FAIL;
    }
    p+= 4;

    fe.next = NULL;
    fe.fsize = makelong((unsigned char *)&p[L_UNCOMPRESSED_SIZE]);
    fe.dataoff = makeword((unsigned char *)&p[L_FILENAME_LENGTH]); 
    ex.csize = makelong((unsigned char *)&p[L_COMPRESSED_SIZE]);
    ex.crc32 = makelong((unsigned char *)&p[L_CRC32]);
    ex.compr = makeword((unsigned char *)&p[L_COMPRESSION_METHOD]);
    ex.exlen = makeword((unsigned char *)&p[L_EXTRA_FIELD_LENGTH]);

    fbuf = (struct FE *)
      malloc (sizeof (struct FE) + fe.dataoff /* + 1 */ + fe.fsize);

    if (fbuf == NULL)
    {
      errf("Out Of Memory!!");
      return FAIL;
    }
    
    if ((p = needData(nd, fe.dataoff + ex.exlen)) == NULL)
      return FAIL;

    *fbuf = fe;
    /* link this node into FE list */
    *fetail = fbuf;
    fetail = &fbuf->next;

#if 1    
    for (i = 0; i < fe.dataoff; i++)
      fbuf->filename[i] = isupper(p[i])? tolower(p[i]): p[i];
#else    
    memcpy(fbuf->filename, p, fe.dataoff);
#endif    
    fbuf->filename[fbuf->dataoff++] = '\0';

#ifdef TRACING
    printf("\ncompression method: %d\n", ex.compr);
    printf("crc32: %lx\n", ex.crc32);
    printf("compressed size: %ld\n", ex.csize);
    printf("uncompressed size: %ld\n", fe.fsize);
    printf("filename length: %d\n", fe.dataoff); /* NOTE: fbuf->dataoff - 1 */
    printf("extra field length: %d\n", ex.exlen);
    printf("file name: %s\n\n", fbuf->filename);
#endif

    switch (ex.compr) {
    case 0:
      if (unstore_file(nd, fbuf->filename + fbuf->dataoff, fe.fsize))
	return FAIL;
      break;
    case 8:
      if (inflate_file(nd, fbuf->filename + fbuf->dataoff, ex.csize, fe.fsize))
	return FAIL;
      break;
    default:
      errf("This zip dearchiver supports only "
	   "`Stored' and `Deflated' -methods");
      return FAIL;
    }
  }
  /* NOT REACHED, see SUCCESSFUL EXIT above */
}

int unstore_file(struct NeedData * nd, char * out, int fsize)
{
  while (fsize > 0)
  {
    char * p;
    int n = fsize > 4096? 4096: fsize;
    
    if (!(p = needData(nd, n)))
      return FAIL;
      
    memcpy(out, p, n);
    out += n;
    fsize -= n;
  }
  return SUCC;
}

extern char * z_errmsg[]; /* defined in zutil.h (why not somewhere else ?) */

#define CHECK_ERR(err, msg) { \
    if (err != Z_OK) { \
        errf("%s error: %s", msg, z_errmsg[2 - err]); \
        return FAIL; \
    } \
}

int inflate_file(struct NeedData * nd, char * out, int csize, int fsize)
{
  int err;
  int crem;
  z_stream d_stream; /* decompression stream */

  d_stream.zalloc = (alloc_func)0;
  d_stream.zfree = (free_func)0;
  d_stream.opaque = (voidpf)0;

  err = inflateInit2(&d_stream, -MAX_WBITS);
  CHECK_ERR(err, "inflateInit2");

  d_stream.avail_out = fsize;
  d_stream.next_out = (Bytef *)out;

  crem = csize;
  do {

    if (crem > 4096)  {
      d_stream.avail_in = 4096;
      d_stream.next_in = (Bytef *)needData(nd, 4096);
      crem -= 4096;
    }
    else  {
      d_stream.avail_in = crem;
      d_stream.next_in = (Bytef *)needData(nd, crem);
      crem =  0;
    }
    if (d_stream.next_in == NULL)
      break;
    
    err = inflate(&d_stream, Z_NO_FLUSH);
    if (err == Z_STREAM_END) break;
    CHECK_ERR(err, "inflate");
  } while (d_stream.total_out < fsize && d_stream.total_in < csize);
  
  err = inflateEnd(&d_stream);
  CHECK_ERR(err, "inflateEnd");

  return d_stream.total_out < fsize; /* SUCC or FAIL */
}
  
#ifdef TESTMAIN

void main(int argc, char ** argv)
{
  struct FE * felist;
  
  char * name = "test.zip";
  
  if (argv[1] != NULL)
    name = argv[1];

  printf("name `%s'\n", name);
  
  felist = uzip2mem(name);

  printf("\n");
  while(felist)
  {
    printf("file name: %s\n", felist->filename);
    printf("file size: %ld\n\n", felist->fsize);
#if 1
    fflush(stdout);
    write(1, felist->filename + felist->dataoff, felist->fsize);
#endif    
    felist = felist->next;
  }
  
  exit(0);
}

#endif
