/*
 * $Id: zipfiles.c $
 *
 * Author: Tomi Ollila <too@iki.fi>
 *
 * 	Copyright (c) 1997 Tomi Ollila
 * 	    All rights reserved
 *
 * Created: Wed Oct 29 18:24:44 1997 too
 * Last modified: Sun Nov  9 12:50:03 1997 too
 *
 * HISTORY 
 * $Log: $
 */

#include <stdio.h>	/* for SEEK_SET & friends */
#include <stdlib.h>
#include <string.h>

#include "trace.h"

#include "zipfiles.h"

#include "uzip2mem.h"
#include "uzipsupp.h"

#define USE_INLINE_FUNCTIONS
#include "list.h"

struct ZipArch
{
  struct Node archnode;
  struct FE * fehead;
  int refcount;
  int delexp;
  char zipfilename[1];
};

struct List ZipList = { 0 };

void initZip(void)
{
      newList(&ZipList);
}  

static struct ZipArch * find_ziparch(const char * name)
{
  Scan_List_BEGIN(&ZipList, struct ZipArch *, ziparch, 0)
    if (strcasecmp(name, ziparch->zipfilename) == 0)
      return ziparch;
  Scan_List_END;

  return NULL;
}

static struct ZipArch * create_ziparch(const char * name)
{
  int namelen = strlen(name);
  struct ZipArch *
    ziparch = (struct ZipArch *)malloc(sizeof *ziparch + namelen /* + 1 */);

  if (ziparch == NULL) {
    errf("Out Of Memory\n");
    return NULL;
  }
    
  ziparch->fehead = uzip2mem(name);

  if (ziparch->fehead == NULL) {
    free(ziparch);
    return NULL;
  }
  ziparch->refcount = 0;
  ziparch->delexp = 0;
    
  memcpy(ziparch->zipfilename, name, namelen + 1);

  addTail(&ZipList, &ziparch->archnode);

  return ziparch;
}

struct ZipArch * openZip(const char * name)
{
  struct ZipArch * ziparch = find_ziparch(name);

  TRACE(TRCF_FUNCENTRY, ("entering openZip(name == %s)\n", name));
  
  if (ziparch)
    return ziparch;

  ziparch = create_ziparch(name);

  if (ziparch)
    return ziparch;

  return NULL;
}

struct ZipFp * zipopen(const char * filename, struct ZipArch * ziparch)
{
  char *p;
  struct FE * felist = ziparch->fehead;
  
  TRACE(TRCF_FUNCENTRY, ("entering zipopen(filename == %s)\n", filename));

  while (felist)
  {
    if ( (p=strrchr(felist->filename, '/')) != NULL)
      p++;
    else
      p = felist->filename;
    
    /* if (strcmp(filename, p) == 0) */ /* all chars lowercase ! */
    if (strcasecmp(filename, p) == 0) /* all chars lowercase ! */
    {
      struct ZipFp * zipfp = (struct ZipFp *)malloc(sizeof *zipfp);

      if (zipfp == NULL) {
        errf("Out Of Memory\n");
        return NULL;
      }
      ziparch->refcount++;
      ziparch->delexp = 0;
      zipfp->ziparch = ziparch;
      zipfp->fenode = (void *)felist;
      zipfp->filepos = 0;
      return zipfp;
    }
    else
      felist = felist->next;
  }
  return NULL;
}

int zipread(struct ZipFp * zp, char * buffer, int length)
{
  struct FE * fe = (struct FE *)zp->fenode;
  int rlen = fe->fsize - zp->filepos;
  int len = (length > rlen)? rlen: length;

  TRACE(TRCF_FUNCENTRY,
	("entering zipread(zipfp == %p length == %d)\n", zp, length));
  
  memcpy(buffer, fe->filename + fe->dataoff + zp->filepos, len);

  zp->filepos+= len;

  TRACE(TRCF_FUNCEXIT,
	("exiting zipread(), len == %d)\n", len));

  return len;
}

#if 0
/* ARGSUSED */
int zipwrite(struct ZipFp * zp, char * buffer, int length)
{
  TRACE(TRCF_FUNCENTRY, ("entering zipwrite(length == %d)\n", length));

  return -1;
}
#endif

int zipseek(struct ZipFp * zp, int offset, int whence)
{
  struct FE * fe = (struct FE *)zp->fenode;

  TRACE(TRCF_FUNCENTRY,
	("entering zipseek(offset == %d, whence == %d)\n", offset, whence));
  
  switch (whence)
  {
  case SEEK_SET:	zp->filepos = offset;			break;
  case SEEK_CUR:	zp->filepos += offset;			break;
  case SEEK_END:	zp->filepos = fe->fsize + offset;	break;
  }
  if	  (zp->filepos < 0)		zp->filepos = 0;
  else if (zp->filepos > fe->fsize)	zp->filepos = fe->fsize;

  return zp->filepos;
}

static void delete_ziparch(struct ZipArch * ziparch)
{
  remNode(&ziparch->archnode);
  free_uzip2mem(ziparch->fehead);
  free(ziparch);
}

void expungeZip()
{
  TRACE(TRCF_FUNCENTRY, ("entering expungeZip()\n"));

  Scan_List_BEGIN(&ZipList, struct ZipArch *, ziparch, 2)
    if (ziparch->refcount <= 0) /* should not have negative values... */
      delete_ziparch(ziparch);
    else
      ziparch->delexp = 1;
  Scan_List_END;
}

void zipclose(struct ZipFp * zp)
{
  struct ZipArch * ziparch = zp->ziparch;

  TRACE(TRCF_FUNCENTRY, ("entering zipclose(zipfp == %p)\n", zp));
  
  ziparch->refcount--;

  if (ziparch->refcount <= 0 && ziparch->delexp)
  {
    delete_ziparch(ziparch);
  }
  free(zp);
}
