/****************************************************************************
 ** ui.h extension file, included from the uic-generated form implementation.
 **
 ** If you wish to add, delete or rename functions or slots use
 ** Qt Designer which will update this file, preserving your code. Create an
 ** init() function in place of a constructor, and a destroy() function in
 ** place of a destructor.
 *****************************************************************************/

void DatabaseWizard::init()
{
#ifndef NDEBUG
  std::cerr << "Initialise" << std::endl;
#endif

  // Set default values.
  databaseUserEdit->setText( getenv( "USER" ) );
  // portSpinBox->setValue( MYSQL_PORT );
  
  setNextEnabled( connectPage       , false );
  setNextEnabled( createDatabasePage, false );
  setNextEnabled( createUserPage    , false );
  setFinishEnabled( importRecipesPage, true );

  const char *home = getenv( "HOME" );
#ifndef NDEBUG
  std::cerr << "HOME = " << home << std::endl;
#endif
  if ( home != NULL ) {
    socketEdit->setText( QString( home ) +
                         "/.kde/share/apps/anymeal/db/mysql.sock" );
  };
  serverWidgetStack->raiseWidget( serverTypeCombo->currentItem() );
}


void DatabaseWizard::connectButtonEnable()
{
  serverUser.reset();
  serverRoot.reset();
  daemon.reset();

  connectButton->setEnabled
    ( !( serverTypeCombo->currentItem() == 0 ?
         serverEdit->text() :
         socketEdit->text() ).isEmpty() &&
      !rootUserEdit->text().isEmpty() && 
      ( !rootPassWordCheckBox->isChecked() ||
        !rootPassWordEdit->text().isEmpty() ) );

  setNextEnabled( connectPage, false );
  createDatabaseEnable();
}


void DatabaseWizard::connectToServer()
{
  try {

    assert( serverTypeCombo->count() == 2 );
    assert( serverTypeCombo->text( 0 ) == i18n( "Network" ) );
    assert( serverTypeCombo->text( 1 ) == i18n( "Embedded" ) );

    assert( !( serverTypeCombo->currentItem() == 0 ?
               serverEdit->text() :
               socketEdit->text() ).isEmpty() );
    assert( !rootUserEdit->text().isEmpty() );
    assert( !rootPassWordCheckBox->isChecked() ||
            !rootPassWordEdit->text().isEmpty() );

    switch ( serverTypeCombo->currentItem() ) {
    case 0: {
      DisplayWaitCursor w;
      serverRoot = MySQLServerPtr
        ( new MySQLNetwork( rootUserEdit->text(), 
                            rootPassWordCheckBox->isChecked() ?
                            (const char *)rootPassWordEdit->text() : "",
                            serverEdit->text(), MYSQL_PORT ) );
      break;}
    case 1: {
      daemon = MySQLDaemonPtr( new MySQLDaemon
                               ( createDatabase( folderEdit->text() ),
                                 socketEdit->text() ) );
      DisplayWaitCursor w;
      serverRoot = MySQLServerPtr
        ( new MySQLSocket( rootUserEdit->text(), 
                           rootPassWordCheckBox->isChecked() ?
                           (const char *)rootPassWordEdit->text() : "",
                           socketEdit->text(), daemon ) );
      break;}
    default:
      assert( false );
    };

    // Lock repetition of this step.
    connectButton->setEnabled( false );
    // Allow next step.
    setNextEnabled( connectPage, true );

  } catch ( std::exception &e ) {
    
#ifndef NDEBUG
    std::cerr << "Error connecting." << std::endl;
#endif
    KMessageBox::error( this, e.what() );

  }; 
}


void DatabaseWizard::createDatabaseEnable()
{
  createDatabaseButton->setEnabled( !databaseEdit->text().isEmpty() );
  setNextEnabled( createDatabasePage, false );
  createUserEnable();
}


void DatabaseWizard::createDatabase()
{
  assert( serverRoot );
  assert( !databaseEdit->text().isEmpty() );

  // Update initial value for allowed clients to sensible value.
  clientEdit->setText( ( serverEdit->text() == "localhost" ||
                         serverTypeCombo->currentItem() == 1 ) ?
                       "localhost" : "%" );

  try {

    std::ostringstream createQuery;
    createQuery << "CREATE DATABASE " << databaseEdit->text()
                << " DEFAULT CHARACTER SET utf8;";
    
    bool exist = false;

    // Try to set up a UTF-8 database.
    if ( mysql_query( serverRoot->getConnection(),
                      createQuery.str().c_str() ) != 0 ) {
      if ( mysql_errno( serverRoot->getConnection() ) == ER_DB_CREATE_EXISTS )
        exist = true;
      else {
        // Probably an older version of MySQL. Try initialising without
        // UTF-8 character-set.
        std::ostringstream createQuery;
        createQuery << "CREATE DATABASE " << databaseEdit->text() << ';';
        if ( mysql_query( serverRoot->getConnection(),
                          createQuery.str().c_str() ) != 0 ) {
          if ( mysql_errno( serverRoot->getConnection() ) == ER_DB_CREATE_EXISTS )
            exist = true;
          else
            ERRORMACRO( false,
                        Error, , i18n( "Error creating database: %1" ).
                        arg( mysql_error( serverRoot->getConnection() ) ) );
        };
          
        KMessageBox::sorry( this,
                            i18n( "Could not configure database to use "
                                  "UTF-8 character set. "
                                  "If you want a UTF-8-aware database, you "
                                  "should upgrade to MySQL 4.1 or higher." ) );
      };
    };

    if ( exist )
      ERRORMACRO( KMessageBox::warningContinueCancel
                  ( this,
                    i18n( "Database '%1' seems to be existing already. "
                          "Do you want to use it?" ).
                    arg( databaseEdit->text() ) ) == KMessageBox::Continue,
                  Error, ,
                  i18n( "Database '%1' is existing already." ).
                  arg( databaseEdit->text() ) );
    
    try {

      // Try to connect to database (using administrator-account).
      CookBook cookBook
        ( DatabasePtr( new MySQLDatabase
                       ( serverRoot, databaseEdit->text() ) ),
          findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ),
          findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ),
          exist );

      // If database is new, the tables have to be created.
      if ( !exist ) {
        DisplayWaitCursor w;
        std::string query( "<?xml version='1.0' encoding='UTF-8'?><create/>" );
        cookBook.getXMLLayer()->translate( query );
      };
      
    } catch ( CookBookDeprecated &e ) {

      // The database is not up to date.
      int version = e.getVersion();
      
      if ( KMessageBox::warningContinueCancel
           ( this,
             i18n( "Do you want to update the database "
                   "from version %1 to version %2? "
                   "It is recommended to do a backup "
                   "before." ).
             arg( versionText( version ) ).
             arg( versionText( DATABASEVERSION ) ),
             i18n( "Update Database" ) ) == KMessageBox::Continue ) {
        
        DisplayWaitCursor w;
        
        // View as cookbook without performing version-check and update.
        CookBook cookBook
          ( DatabasePtr( new MySQLDatabase
                         ( serverRoot, databaseEdit->text() ) ),
            findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ),
            findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ),
            false );
        
        cookBook.update();
        
      } else
        throw e;

    };

    // Lock repetition of this step.
    createDatabaseButton->setEnabled( false );
    // Allow next step.
    setNextEnabled( createDatabasePage, true );

  } catch ( std::exception &e ) {

    KMessageBox::error( this, e.what() );
    
  };

}


void DatabaseWizard::createUserEnable()
{
  // Name of client-host may be empty!
  // Changing it requires redoing the grant-command though.
  createUserButton->setEnabled
    ( !databaseUserEdit->text().isEmpty() &&
      ( !databasePassWordCheckBox->isChecked() ||
        ( !databasePassWordEdit->text().isEmpty() &&
          databasePassWordEdit->text() ==
          databasePassWordConfirmEdit->text() ) ) );
  setNextEnabled( createUserPage, false );
}


void DatabaseWizard::createUser()
{
  assert( serverRoot );

  try {
    
    assert( !databaseUserEdit->text().isEmpty() );
    assert( !databasePassWordCheckBox->isChecked() ||
            !databasePassWordEdit->text().isEmpty() );

    DisplayWaitCursor w;

    std::ostringstream grantQuery;
    grantQuery << "GRANT ALL ON " << databaseEdit->text() << ".* TO "
               << databaseUserEdit->text();
    if ( !clientEdit->text().isEmpty() )
      grantQuery << "@'" << clientEdit->text() << '\'';
    if ( databasePassWordCheckBox->isChecked() ) {
      assert( !databasePassWordEdit->text().isEmpty() );
      grantQuery << " IDENTIFIED BY '" << databasePassWordEdit->text() << "'";
    };
    grantQuery << ';';

    ERRORMACRO( !mysql_query( serverRoot->getConnection(),
                              grantQuery.str().c_str() ), Error, ,
                i18n( "Error setting up database account: %1" ).
                arg( mysql_error( serverRoot->getConnection() ) ) );

    // Try to connect in case user already existed!
    switch ( serverTypeCombo->currentItem() ) {
    case 0:
      serverUser = MySQLServerPtr
        ( new MySQLNetwork( databaseUserEdit->text(), 
                            databasePassWordCheckBox->isChecked() ?
                            (const char *)databasePassWordEdit->text() : "",
                            serverEdit->text(), MYSQL_PORT ) );
      break;
    case 1:
      assert( daemon );
      serverUser = MySQLServerPtr
        ( new MySQLSocket( databaseUserEdit->text(), 
                           databasePassWordCheckBox->isChecked() ?
                           (const char *)databasePassWordEdit->text() : "",
                           socketEdit->text(), daemon ) );
      break;
    default:
      assert( false );
    };

    // Check access to database.
    MySQLDatabase( serverUser, databaseEdit->text() );

    // Lock repetition of this step.
    createUserButton->setEnabled( false );
    // Allow next step and finalisation.
    setNextEnabled( createUserPage, true );

  } catch ( std::exception &e ) {
    
    KMessageBox::error( this, e.what() );

  };
}


void DatabaseWizard::importRecipes()
{
  assert( serverUser );

  try {
    
    ImportMealMasterDialog importMealMasterDialog( this );
    importMealMasterDialog.databaseWizard = true;
    importMealMasterDialog.cookBook = CookBookPtr
      ( new CookBook
        ( DatabasePtr( new MySQLDatabase
                       ( serverUser, databaseEdit->text() ) ),
          findAnyMealFile( "appdata", "scripts/mysqlIn.xsl" ),
          findAnyMealFile( "appdata", "scripts/mysqlOut.xsl" ) ) );
    importMealMasterDialog.exec();

  } catch ( std::exception &e ) {
    
    KMessageBox::error( this, e.what() );

  };
}


void DatabaseWizard::selectSocket()
{
  QString fileName =
    KFileDialog::getOpenFileName( ":dbfolder",
                                  i18n( "*.sock|Socket (*.sock)\n"
                                        "*|All Files(*)" ),
                                  this,
                                  i18n( "Select database socket" ) );
  if ( fileName != QString::null ) {
    socketEdit->setText( fileName );
  };
}

const std::string &DatabaseWizard::createDatabase( const std::string &dir )
{
  if ( testFile( dir + "/mysql" ) != EISDIR ) {
    ERRORMACRO( KMessageBox::questionYesNo
                ( this, i18n( "Directory '%1' does not exist. Should the "
                              "database be created?" ).arg( dir ),
                  i18n( "Database does not exist" ) ) == KMessageBox::Yes,
                Error, ,
                "Database '" << dir << "' does not exist." );
    std::string info = MySQLDaemon::create( dir, dir + "/mysql.sock" );
    KMessageBox::information( this, info, i18n( "Installation log" ) );
  };
  return dir;
}


void DatabaseWizard::socketChanged( const QString &_s )
{
  std::string socket( (const char *)_s );
  unsigned int pos = socket.find_last_of( "/" );
  if ( pos != std::string::npos )
    folderEdit->setText( std::string( socket, 0, pos ).c_str() );
}
