/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/times.h>

#include "gdis.h"

#define DEBUG 0

extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];

gint selected_label = -1;
gint current_colour[3];
GtkWidget *label_list;

/***************************************/
/* apply PBC to the atoms in the model */
/***************************************/
/* TODO - flag for delete repetitions yes/no */
#define DEBUG_PBC_ATOMS 0
void pbc_constrain_atoms(struct model_pak *data)
{
gint i, j;
gfloat vec[3], r2;

g_return_if_fail(data != NULL);

#ifdef TIMER
start_timer("pbc_constrain_atoms");
#endif
#if DEBUG_PBC_ATOMS
printf("[A] num atoms: %d\n", data->num_atoms);
printf("[A] num shels: %d\n", data->num_shells);
#endif

/* NEW - fractional & cartesian coords are now separate */
/* as cartesian are no longer converted to fractional */

/* requires bond & mol calc to have been done */
/* NOTES - non-orthogonal axes transformations are */
/* all taken care of in coord update routines, so */
/* that adding std xlat vec's to x,y,z should work */
/* TODO - if PBC transform is off - revert back to initial coords? */
/* molecule PBC translation - avoid breaking bonds */
/* NOTE - use original coords for PBC translation */
/* as the code will not work for rotated coords */
/* this unfortunately makes it non-reversible */
for (i=data->num_atoms ; i-- ; )
  {
/* don't do periodic images */
  if (!(data->atoms+i)->orig)
    {
    (data->atoms+i)->status |= DELETED;
    continue;
    }
/* get coords */
/* NEW - should always be untransformed (ie scaled fractional) */
  vec[0] = (data->atoms+i)->x;
  vec[1] = (data->atoms+i)->y;
  vec[2] = (data->atoms+i)->z;
/* constrain */
  for (j=data->periodic ; j-- ; )
    {
/* calc number of moves to make within pbc */
    while (vec[j] < 0.0)
      vec[j] += 1.0;
    while (vec[j] > 1.0)
      vec[j] -= 1.0;
    }
/* apply the translation */
  (data->atoms+i)->x = vec[0];
  (data->atoms+i)->y = vec[1];
  (data->atoms+i)->z = vec[2];
  }

/* NEW - do shells separately */
for (i=data->num_shells ; i-- ; )
  {
/* don't do periodic images */
  if (!(data->shells+i)->orig)
    {
    (data->shells+i)->status |= DELETED;
    continue;
    }
/* get coords */
/* NEW - should always be untransformed (ie scaled fractional) */
  vec[0] = (data->shells+i)->x;
  vec[1] = (data->shells+i)->y;
  vec[2] = (data->shells+i)->z;
/* constrain */
  for (j=data->periodic ; j-- ; )
    {
/* calc number of moves to make within pbc */
    while (vec[j] < 0.0)
      vec[j] += 1.0;
    while (vec[j] > 1.0)
      vec[j] -= 1.0;
    }
/* apply the translation */
  (data->shells+i)->x = vec[0];
  (data->shells+i)->y = vec[1];
  (data->shells+i)->z = vec[2];
  }
mem_shrink(data);

#if DEBUG_PBC_ATOMS
printf("[B] num atoms: %d\n", data->num_atoms);
printf("[B] num shels: %d\n", data->num_shells);
#endif

/* NEW - delete repetitions */
for (i=0 ; i<data->num_atoms-1 ; i++)
  {
  if ((data->atoms+i)->status & DELETED)
    continue;
  for (j=i+1 ; j<data->num_atoms ; j++)
    {
    if ((data->atoms+j)->status & DELETED)
      continue;
/* cope with atoms that are closer to a periodic image of another atom */
    vec[0] = fabs((data->atoms+i)->x  - (data->atoms+j)->x);
    if (vec[0] > 0.5)
      vec[0] -= 1.0;
    vec[1] = fabs((data->atoms+i)->y  - (data->atoms+j)->y);
    if (vec[1] > 0.5)
      vec[1] -= 1.0;
    vec[2] = fabs((data->atoms+i)->z  - (data->atoms+j)->z);
    if (vec[2] > 0.5)
      vec[2] -= 1.0;
    r2 = VEC3MAGSQ(vec);

    if (r2 < FRACTION_TOLERANCE)
      {
/* only delete if same atom type & no site occ. factor */
      if ((data->atoms+i)->atom_code == (data->atoms+j)->atom_code)
        if ((data->atoms+i)->sof > 0.9)
          if ((data->atoms+j)->sof > 0.9)
            (data->atoms+j)->status |= DELETED;
      }
    }
  }

/* delete repetitions (shells) */
for (i=0 ; i<data->num_shells-1 ; i++)
  {
  if ((data->shells+i)->status & DELETED)
    continue;
  for (j=i+1 ; j<data->num_shells ; j++)
    {
/* cope with atoms that are 1 repeat length apart */
    vec[0] = fabs((data->shells+i)->x  - (data->shells+j)->x);
    if (vec[0] > 0.5)
      vec[0] -= 1.0;
    vec[1] = fabs((data->shells+i)->y  - (data->shells+j)->y);
    if (vec[1] > 0.5)
      vec[1] -= 1.0;
    vec[2] = fabs((data->shells+i)->z  - (data->shells+j)->z);
    if (vec[2] > 0.5)
      vec[2] -= 1.0;

    r2 = VEC3MAGSQ(vec);

/* FIXME - what effect does ignoring SOF of atom have here? */
    if (r2 < FRACTION_TOLERANCE)
      if ((data->shells+i)->atom_code == (data->shells+j)->atom_code)
        (data->shells+j)->status |= DELETED;
    }
  }

mem_shrink(data);

#if DEBUG_PBC_ATOMS
printf("[C] num atoms: %d\n", data->num_atoms);
printf("[C] num shels: %d\n", data->num_shells);
#endif

#ifdef TIMER
stop_timer("pbc_constrain_atoms");
#endif
}

/**************************************/
/* apply PBC to the mols in the model */
/**************************************/
/* TODO - flag for delete repetitions yes/no */
#define DEBUG_CONFINE_MOLS 0
void pbc_constrain_mols(struct model_pak *data)
{
gint i, j, k, m, mov[3];
gfloat vec[3];

g_return_if_fail(data != NULL);

#ifdef TIMER
start_timer("pbc_constrain_mols");
#endif

/* NEW - all orig coords are fractional */

for (i=data->num_mols ; i-- ; )
  {
/* get coords (should be scaled untransformed fractional) */
  ARR3SET(vec, (data->mols+i)->centroid);

/* constrain */
  mov[0] = mov[1] = mov[2] = 0;
  for (j=data->periodic ; j-- ; )
    {
/* calc number of moves to make within pbc */
    while (vec[j] < 0.0)
      {
      vec[j] += 1.0;
      mov[j]++;
      }
    while (vec[j] > 1.0)
      {
      vec[j] -= 1.0;
      mov[j]--;
      }
    }

/* add the translation */
  for (j=3 ; j-- ; )
    (data->mols+i)->centroid[j] += (gfloat) mov[j];

#if DEBUG_CONFINE_MOLS
P3VEC("new: ",(data->mols+i)->centroid);
#endif

/* apply to all atoms in this molecule */
  for (j=(data->mols+i)->num_atoms ; j-- ; )
    {
    k = *((data->mols+i)->atom+j);
    (data->atoms+k)->x += (gfloat) mov[0];
    (data->atoms+k)->y += (gfloat) mov[1];
    (data->atoms+k)->z += (gfloat) mov[2];
/* also to any attached shell */
    if ((data->atoms+k)->has_shell)
      {
      m = (data->atoms+k)->idx_shell;
      (data->shells+m)->x += (gfloat) mov[0];
      (data->shells+m)->y += (gfloat) mov[1];
      (data->shells+m)->z += (gfloat) mov[2];
      }
    }
  }
#ifdef TIMER
stop_timer("pbc_constrain_mols");
#endif
}

/*******************************************/
/* draw a box - contents are the selection */
/*******************************************/
void update_box(gint x, gint y, struct model_pak *data, gint call_type)
{
switch (call_type)
  {
  case START:
/* NEW - don't clean the selection (allows multiple box selections) */
/* setup the box selection object */
    data->box_on = TRUE;
    data->box_select[0].px = x;
    data->box_select[0].py = y;
    data->box_select[1].px = x;
    data->box_select[1].py = y;
    break;
  case UPDATE:
    data->box_select[1].px = x;
    data->box_select[1].py = y;
    break;
  }
}

/****************************************/
/* add a particular atom to a selection */
/****************************************/
#define DEBUG_ADD_SELECT 0
gint add_select(struct model_pak *data, gint atom)
{
gint i;

/* valid range? */
if (atom < 0 || atom > data->num_atoms)
  return(1);

/* already in the selection? */
for (i=data->select_size ; i-- ; )
  if (*(data->select_list+i) == atom)
    return(2);
/* check allocation */
if (data->select_size >= data->select_limit)
  mem_grow(data, SELECT, DEFAULT_SS);
/* add the atom to the list */
*(data->select_list+data->select_size) = atom;
data->select_size++;

#if DEBUG_ADD_SELECT
printf("list:");
for (i=0 ; i<data->select_size ; i++)
  printf(" %d", *(data->select_list+i));
printf("\n");
#endif
return(0);
}

/******************************************/
/* remove an atom from the selection list */
/******************************************/
gint del_select(struct model_pak *data, gint atom)
{
gint i, j;

/* valid range? */
if (atom < 0 || atom > data->num_atoms)
  return(1);

/* in the selection? */
for (i=data->select_size ; i-- ; )
  if (*(data->select_list+i) == atom)
    goto found;
return(2);

/* shuffle the list to overwrite the deleted atom */
found:;
j = i+1;
while(j<data->select_size)
  {
  *(data->select_list+i) = *(data->select_list+j);
  i++;
  j++;
  }
data->select_size--;

#if DEBUG_ADD_SELECT
printf("list:");
for (i=0 ; i<data->select_size ; i++)
  printf(" %d", *(data->select_list+i));
printf("\n");
#endif
return(0);
}

/*************************************************/
/* find & highlight atoms within (box) selection */
/*************************************************/
void select_box()
{
gint i, j, atom, mol;
gint x1, y1, x2, y2, tmp;
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);

x1 = data->box_select[0].px;
y1 = data->box_select[0].py;
x2 = data->box_select[1].px;
y2 = data->box_select[1].py;

/* ensure x1 < x2 and y1 < y2 */
if (x1 > x2)
  {
  tmp = x1;
  x1 = x2;
  x2 = tmp;
  }
if (y1 > y2)
  {
  tmp = y1;
  y1 = y2;
  y2 = tmp;
  }

/* get active subwin */
tmp = -1;
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (sysenv.displayed[i] == sysenv.active)
    tmp = i;
/* subtract off subwin centre */
x1 -= sysenv.subcenx[tmp];
y1 -= sysenv.subceny[tmp];
x2 -= sysenv.subcenx[tmp];
y2 -= sysenv.subceny[tmp];

/* account for translation */
x1 -= data->offset[0];
y1 -= data->offset[1];
x2 -= data->offset[0];
y2 -= data->offset[1];

/* compare and select if... */
for (i=0 ; i<data->num_atoms ; i++)
  {
/* ... not deleted */
  if ((data->atoms+i)->status & DELETED)
    continue;
/* ... and within box */
  if ((data->atoms+i)->px > x1 && (data->atoms+i)->px < x2)
    if ((data->atoms+i)->py > y1 && (data->atoms+i)->py < y2)
      {
      if (sysenv.select_mode == ATOM)
        {
/* add atom to the selection */
        add_select(data, i);
        (data->atoms+i)->status |= SELECT_HL;
        }
      else
        {
/* add atom's assoc mol to the selection */
        mol = (data->atoms+i)->molecule;
        for (j=(data->mols+mol)->num_atoms ; j-- ; )
          {
          atom = *((data->mols+mol)->atom+j);
          add_select(data, atom);
          (data->atoms+atom)->status |= SELECT_HL;
          }
        }
      }
  }

/* done */
redraw_canvas(SINGLE);
}

/****************************************************/
/* create a general purpose colour selection dialog */
/****************************************************/
void colsel_dialog(GtkWidget *w, gpointer *obj)
{
gint id;
GtkWidget *csd;
GtkWidget *colsel;

/* check */
g_return_if_fail(obj != NULL);

/* create colour selection dialog */
csd = gtk_color_selection_dialog_new("Select color");
gtk_signal_connect(GTK_OBJECT(csd), "destroy",
                   GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer) csd);
gtk_signal_connect_object 
       (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->cancel_button),
       "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), (gpointer) csd);

/* TODO - move selection cursor to whatever the current colour is */
colsel = GTK_COLOR_SELECTION_DIALOG(csd)->colorsel;

/* callback routine depends on who called us */
id = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(obj), "id"));
switch(id)
  {
  case MORPH_COLOUR:
/* set title */
/* apply colour change event */
    gtk_object_set_data(GTK_OBJECT(colsel), "id", (gpointer) MORPH_COLOUR);
    gtk_object_set_data(GTK_OBJECT(colsel), "ptr", (gpointer) 
                        gtk_object_get_data(GTK_OBJECT(obj), "ptr"));
    gtk_signal_connect 
       (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->ok_button),
       "clicked", GTK_SIGNAL_FUNC(event_render_modify), (gpointer) colsel);
/*
    gtk_signal_connect(GTK_OBJECT(colsel), "color_changed",
                       GTK_SIGNAL_FUNC(event_render_modify), (gpointer) colsel);
*/
    break;
  case ELEMENT_COLOUR:
/* apply colour change event */
    gtk_object_set_data(GTK_OBJECT(colsel), "wid", (gpointer) obj);
    gtk_object_set_data(GTK_OBJECT(colsel), "ptr", (gpointer) 
                        gtk_object_get_data(GTK_OBJECT(obj), "ptr"));
    gtk_signal_connect 
       (GTK_OBJECT(GTK_COLOR_SELECTION_DIALOG(csd)->ok_button),
       "clicked", GTK_SIGNAL_FUNC(elem_colour_change), (gpointer) colsel);
    break;
  }

gtk_widget_show(csd);
}

/*******************************************/
/* scan and delete invalid geometry labels */
/*******************************************/
#define DEBUG_CHECK_GEOM 0
void check_geom(struct model_pak *data)
{
gint i, j, n, flag, items;
struct geom_pak *ptr;

g_return_if_fail(data != NULL);

#if DEBUG_CHECK_GEOM
printf("Num old labels: %d\n",data->num_geom);
printf("Num atoms: %d\n",data->num_atoms);
#endif

n=0;
while (n<data->num_geom)
  {
  flag=0;
/* measurement type */
  items=2;
#if DEBUG_CHECK_GEOM
printf("label atoms: ");
#endif
  switch((data->geom+n)->type)
    {
    case DIHEDRAL:
      items++;
    case ANGLE:
      items++;
    case BOND:
    case DIST:
      ptr = data->geom + n;
      for (i=0 ; i<items ; i++)
        {
#if DEBUG_CHECK_GEOM
printf(" %d",*(ptr->list+i));
#endif
        if (*(ptr->list+i) >= data->num_atoms)
          {
          flag++;
/* FIXME - redundant code (see delete geom label) */
/* delete it */
          g_free(ptr->list);
          g_free(ptr->text);
/* FIXME - this shuffling'll cause problems with checking */
/* subsequent labels for validity */
          for (j=n ; j<data->num_geom-1 ; j++) 
            {
            (data->geom+j)->type = (data->geom+j+1)->type;
            (data->geom+j)->list = (data->geom+j+1)->list;
            (data->geom+j)->text = (data->geom+j+1)->text;
            }
/* update */
          data->num_geom--;
          break;
          }
        }
      break;
    }
  if (!flag)
    n++;
#if DEBUG_CHECK_GEOM
printf("\n");
#endif
  }
/* done */
#if DEBUG_CHECK_GEOM
printf("Num new labels: %d\n",data->num_geom);
#endif
}

/*********************************/
/* redo the geometry label clist */
/*********************************/
void update_geom_info()
{
gint i,j,k,l,n,flag;
gfloat val;
gchar *info[3];
struct model_pak *data;

/* checks */
if (!sysenv.num_models)
  return;

/* NEW - still update values even if no dialog */
flag=0;
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].type == GEOMETRY && sysenv.dialog[i].active)
    flag++;

/* delete all existing entries */
if (flag)
  {
  gtk_clist_freeze((GtkCList *) label_list);
  gtk_clist_clear((GtkCList *) label_list);
  }

/* check */
data = model_ptr(sysenv.active,RECALL);
if (!data)
  return;

for (n=0 ; n<data->num_geom ; n++)
  {
/* measurement type */
  switch((data->geom+n)->type)
    {
    case BOND:
      info[0] = g_strdup("Bond");
/* convert # into atom labels */
      info[1] = (gchar *) g_malloc((2*ELEM_LABEL_SIZE+3)*sizeof(gchar));
      i = *((data->geom+n)->list);
      j = *((data->geom+n)->list+1);
/* TODO - else delete */
      if (i < data->num_atoms && j < data->num_atoms)
        {
        g_snprintf(info[1],8,"%2s : %-2s",(data->atoms+i)->element
                                      ,(data->atoms+j)->element);
/* recalc */
        val = calc_dist(data, i, j);
        g_free((data->geom+n)->text);
        (data->geom+n)->text = g_strdup_printf("%7.3f",val);
        info[2] = (data->geom+n)->text;
        }
      break;
    case DIST:
      info[0] = g_strdup("Dist");
      info[1] = (gchar *) g_malloc((2*ELEM_LABEL_SIZE+1)*sizeof(gchar));
      i = *((data->geom+n)->list);
      j = *((data->geom+n)->list+1);
/* TODO - else delete */
      if (i < data->num_atoms && j < data->num_atoms)
        {
        g_snprintf(info[1],8,"%2s : %-2s",(data->atoms+i)->element
                                      ,(data->atoms+j)->element);
/* recalc */
        val = calc_dist(data, i, j);
        g_free((data->geom+n)->text);
        (data->geom+n)->text = g_strdup_printf("%7.3f",val);
        info[2] = (data->geom+n)->text;
        }
      break;
    case ANGLE:
      info[0] = g_strdup("Angle");
/* convert # into atom labels */
      info[1] = (gchar *) g_malloc((3*ELEM_LABEL_SIZE+2)*sizeof(gchar));
      i = *((data->geom+n)->list);
      j = *((data->geom+n)->list+1);
      k = *((data->geom+n)->list+2);
      g_snprintf(info[1],13,"%2s : %-2s : %-2s",(data->atoms+i)->element
                                              ,(data->atoms+j)->element
                                              ,(data->atoms+k)->element);
/* recalc */
      val = calc_angle(data, i, j, k);
      g_free((data->geom+n)->text);
      (data->geom+n)->text = g_strdup_printf("%7.3f",val);
      info[2] = (data->geom+n)->text;
      break;
    case DIHEDRAL:
      info[0] = g_strdup("Torsion");
/* convert # into atom labels */
      info[1] = (gchar *) g_malloc((4*ELEM_LABEL_SIZE+4)*sizeof(gchar));
      i = *((data->geom+n)->list);
      j = *((data->geom+n)->list+1);
      k = *((data->geom+n)->list+2);
      l = *((data->geom+n)->list+3);
      g_snprintf(info[1],18,"%2s : %-2s : %-2s : %-2s",
                             (data->atoms+i)->element,
                             (data->atoms+j)->element,
                             (data->atoms+k)->element,
                             (data->atoms+l)->element);
/* recalc */
      val = calc_dihedral(data, i, j, k, l);
      g_free((data->geom+n)->text);
      (data->geom+n)->text = g_strdup_printf("%7.3f",val);
      info[2] = (data->geom+n)->text;
      break;
    default:
      info[0] = g_strdup("Unknown");
      info[1] = g_strdup("?");
      info[2] = NULL;
    }
/* add to the list */
  if (flag)
    gtk_clist_append((GtkCList *) label_list, &info[0]);
/* free */
  g_free(info[0]);
  g_free(info[1]);
  } 

/* update */
if (flag)
  gtk_clist_thaw((GtkCList *) label_list);
}

/***************************/
/* geometry label creation */
/***************************/
gint geom_label(gint type, struct model_pak *data, gint *atoms, gchar *label)
{
gint n;

/* checks */
if (!data->geom_limit)
  return(1);

/* do we need more memory? */
while (data->num_geom >= data->geom_limit)
  mem_grow(data, GEOM, DEFAULT_NL);
n = data->num_geom;

(data->geom+n)->type = type;
switch(type)
  {
  case BOND:
  case DIST:
/* allocate */
    (data->geom+n)->list = (gint *) g_malloc(2*sizeof(gint));
/* assign */ 
    *((data->geom+n)->list) = *(atoms);
    *((data->geom+n)->list+1) = *(atoms+1);
    (data->geom+n)->text = strdup(label);
    break;
  case ANGLE:
/* allocate */
    (data->geom+n)->list = (gint *) g_malloc(3*sizeof(gint));
/* assign */ 
    ARR3SET((data->geom+n)->list, atoms);
    (data->geom+n)->text = strdup(label);
    break;
  case DIHEDRAL:
/* allocate */
    (data->geom+n)->list = (gint *) g_malloc(4*sizeof(gint));
/* assign */ 
    *((data->geom+n)->list) = *(atoms);
    *((data->geom+n)->list+1) = *(atoms+1);
    *((data->geom+n)->list+2) = *(atoms+2);
    *((data->geom+n)->list+3) = *(atoms+3);
    (data->geom+n)->text = strdup(label);
    break;
  default:
    printf("geom_label() error: unknown type.\n");
    return(1);
  }

/* creation successful */
data->num_geom++;
return(0);
}

/***************************/
/* geometry label deletion */
/***************************/
void delete_geom_label(gint mode)
{
gint i, n, flag;
struct model_pak *data;

/* checks */
if (!sysenv.num_models)
  return;

flag=0;
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].type == GEOMETRY && sysenv.dialog[i].active)
    flag++;

if (!flag)
  return;

data = model_ptr(sysenv.active, RECALL);
if (data == NULL)
  return;

/* perform the requested deletion */
switch(mode)
  {
  case SINGLE:
/* valid selection? */
    if (selected_label >= 0)
      {
/* delete it */
      g_free((data->geom+selected_label)->list);
      g_free((data->geom+selected_label)->text);
      for (n=selected_label ; n<data->num_geom-1 ; n++) 
        {
        (data->geom+n)->type = (data->geom+n+1)->type;
        (data->geom+n)->list = (data->geom+n+1)->list;
        (data->geom+n)->text = (data->geom+n+1)->text;
        }
/* update */
      data->num_geom--;
      if (selected_label == data->num_geom)
        selected_label--;
      }
    break;
  case ALL:
/* delete all labels */
    for (n=0 ; n<data->num_geom ; n++)
      {
      g_free((data->geom+n)->list);
      g_free((data->geom+n)->text);
      }
/* update */
    data->num_geom = 0;
    selected_label = -1;
    break;
  }

/* update & redraw */
update_geom_info();
redraw_canvas(SINGLE);
}

/*************/
/* find bond */
/*************/
GSList *seek_bond(gint subwin, gint x, gint y, struct model_pak *data)
{
gfloat dx, dy, dr2;
GSList *bond;

/* subtract off subwin centre */
x -= sysenv.subcenx[subwin];
y -= sysenv.subceny[subwin];

bond = data->bonds;
while(bond != NULL)
  {
  dx = ((struct bond_pak *) bond->data)->px - x + data->offset[0]; 
  dy = ((struct bond_pak *) bond->data)->py - y + data->offset[1]; 
  dr2 = dx*dx + dy*dy;
/* FIXME - too arbitrary? */
  if (dr2 < 36)
    return(bond);
  bond = g_slist_next(bond);
  }
return(NULL);
}

/****************************************/
/* NEW - routines for actual geom calcs */
/****************************************/
gfloat calc_dist(struct model_pak *data, gint i, gint j)
{
gfloat r[3];

r[0] = (data->atoms+i)->rx - (data->atoms+j)->rx;
r[1] = (data->atoms+i)->ry - (data->atoms+j)->ry;
r[2] = (data->atoms+i)->rz - (data->atoms+j)->rz;

return(VEC3MAG(r));
}

gfloat calc_angle(struct model_pak *data, gint i, gint j, gint k)
{
gfloat a[3], b[3];

/* TODO - check bonding order to ensure i-j-k is valid */

/* compute vectors (from center atom) */
a[0] = (data->atoms+i)->rx - (data->atoms+j)->rx;
a[1] = (data->atoms+i)->ry - (data->atoms+j)->ry;
a[2] = (data->atoms+i)->rz - (data->atoms+j)->rz;
b[0] = (data->atoms+k)->rx - (data->atoms+j)->rx;
b[1] = (data->atoms+k)->ry - (data->atoms+j)->ry;
b[2] = (data->atoms+k)->rz - (data->atoms+j)->rz;
return(180.0*via(a,b,3)/PI); 
}

gfloat calc_dihedral(struct model_pak *data, gint i, gint j, gint k, gint l)
{
gfloat len;
gfloat a[3], b[3], n[3];

/* TODO - check bonding order to ensure i-j-k-l is valid (improper???) */

/* compute end atom vectors (from 1-2 axis) */
a[0] = (data->atoms+i)->rx - (data->atoms+j)->rx;
a[1] = (data->atoms+i)->ry - (data->atoms+j)->ry;
a[2] = (data->atoms+i)->rz - (data->atoms+j)->rz;
b[0] = (data->atoms+l)->rx - (data->atoms+k)->rx;
b[1] = (data->atoms+l)->ry - (data->atoms+k)->ry;
b[2] = (data->atoms+l)->rz - (data->atoms+k)->rz;
/* compute normal to the plane in which the dihedral is to be calc'd */
n[0] = (data->atoms+j)->rx - (data->atoms+k)->rx;
n[1] = (data->atoms+j)->ry - (data->atoms+k)->ry;
n[2] = (data->atoms+j)->rz - (data->atoms+k)->rz;
/* normalize */
len = VEC3MAG(n);
n[0] /= len;
n[1] /= len;
n[2] /= len;
/* n[0..2] is the normal of the plane */
/* project 1-0 onto the normal (ie dist to plane) */
len = a[0]*n[0] + a[1]*n[1] + a[2]*n[2];
/* subtract vector to plane from 1-0 to get projection on plane */
a[0] -= len*n[0];
a[1] -= len*n[1];
a[2] -= len*n[2];
/* project 2-3 onto the normal */
len = b[0]*n[0] + b[1]*n[1] + b[2]*n[2];
/* subtract vector to plane from 2-3 to get projection on plane */
b[0] -= len*n[0];
b[1] -= len*n[1];
b[2] -= len*n[2];
/* compute angle between projected vectors */
return(180.0*via(a,b,3)/PI); 
}

/*****************/
/* info on bonds */
/*****************/
void info_bond(gint subwin, gint x, gint y, struct model_pak *data)
{
gint i,j,atom[2];
gfloat r;
gchar txt[80];
GSList *bond;

/* seek */
bond = seek_bond(subwin, x, y, data);
if (!bond)
  return;

i = ((struct bond_pak *) bond->data)->atom1;
j = ((struct bond_pak *) bond->data)->atom2;

/* separation */
r = calc_dist(data, i, j);
/* create geometry label */
atom[0] = i;
atom[1] = j;
g_snprintf(txt,8,"%7.3f",r);
geom_label(BOND, data, atom, txt);
update_geom_info();
/* redraw */
redraw_canvas(SINGLE);
}

/*************************/
/* bond overlay modifier */
/*************************/
void bond_overlay(struct model_pak *data)
{
gint i, j, m;
gint count;
GSList *bond;

/* search normal bonds */
bond = data->bonds;
while(bond != NULL)
  {
/* retrieve constituents */
  i = ((struct bond_pak *) bond->data)->atom1;
  j = ((struct bond_pak *) bond->data)->atom2;
/* search user bonds... */
  for (m=0 ; m<data->num_ubonds ; m++)
    {
    count=0;
    if (i == (data->ubonds+m)->atom1)
      count++;
    if (i == (data->ubonds+m)->atom2)
      count++;
    if (j == (data->ubonds+m)->atom1)
      count++;
    if (j == (data->ubonds+m)->atom2)
      count++;
/* overlay the type */
    if (count > 1)
      {
      ((struct bond_pak *) bond->data)->type = (data->ubonds+m)->type; 
      break;
      }
    }
  bond = g_slist_next(bond);
  }
}

/* must be an existing (normal) bond */
/* this makes selection (one click) and drawing easier */
void user_bond(gint subwin, gint x, gint y, struct model_pak *data, gint type)
{
gint i, j, n;
gint count, flag;
GSList *bond;

/* valid bond? */
bond = seek_bond(subwin, x, y, data);
if (!bond)
  return;
/* retrieve constituents */
i = ((struct bond_pak *) bond->data)->atom1;
j = ((struct bond_pak *) bond->data)->atom2;

/* search... */
flag=0;
for (n=0 ; n<data->num_ubonds ; n++)
  {
  count=0;
  if (i == (data->ubonds+n)->atom1)
    count++;
  if (i == (data->ubonds+n)->atom2)
    count++;
  if (j == (data->ubonds+n)->atom1)
    count++;
  if (j == (data->ubonds+n)->atom2)
    count++;
/* ...and make the change */
  if (count > 1)
    {
    (data->ubonds+n)->type = type; 
    flag=1;
    break;
    }
  }

/* not found - create */
if (!flag)
  {
  n = data->num_ubonds;
  if (++data->num_ubonds >= data->ubond_limit)
    mem_grow(data, UBOND, 10);
  (data->ubonds+n)->atom1 = i;
  (data->ubonds+n)->atom2 = j;
  (data->ubonds+n)->type = type;
  }

/* done */
redraw_canvas(SINGLE);
}

/*****************/
/* info on dists */
/*****************/
void info_dist(gint x, gint y, struct model_pak *data)
{
static gint calls=0, first=0, last=0;
gint i, atom[2];
gfloat r;
gchar txt[80];

/* (kludgy) fix for picking only one atom, switching modes */
/* and then changing back, and having the first new atom */
/* really the last atom of the previous time */
/* clean up call (mode switching) */
if (x < 0)
  {
  calls=first=last=0;
  return;
  }

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;
else
  calls++;

/* are we looking at the 1st or the 2nd call? */
switch (calls)
  {
  case 1:
/* clear any previous highlighting */
/* should be ok on very 1st call - assuming no highlighting */
/* is allowed to persist after a mode change */
    (data->atoms+first)->status &= ~SQUARE_HL;
/* save */
    first = i;
/* highlight selected atom */
    (data->atoms+first)->status |= SQUARE_HL;
    redraw_canvas(SINGLE);
    break;
  case 2:
/* save */
    last = i;
/* remove highlight (we now have a label) */
    (data->atoms+first)->status &= ~SQUARE_HL;
/* we now have two atoms we can calculate a distance for */
    r = calc_dist(data, first, last);
/* create geometry label */
    atom[0] = first;
    atom[1] = last;
    g_snprintf(txt,8,"%7.3f",r);
    geom_label(DIST, data, atom, txt);
    update_geom_info();
/* redraw */
    redraw_canvas(SINGLE);
/* reset counter */
    calls=0;
    break;
  default:
    printf("Error: bad state in info_dist()\n");
    calls=0;
  }
}

/******************/
/* info on angles */
/******************/
void info_angle(gint x, gint y, struct model_pak *data)
{
static gint calls=0, atoms[3]={0,0,0};
gint i;
gfloat angle;
gchar txt[40];

/* (kludgy) fix for picking only one atom, switching modes */
/* and then changing back, and having the first new atom */
/* really the last atom of the previous time */
/* clean up call (mode switching) */
if (x < 0)
  {
  calls=atoms[0]=atoms[1]=atoms[2]=0;
  return;
  }

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;
else
  calls++;

/* are we looking at the 1st or the 2nd call? */
switch (calls)
  {
  case 1:
/* clear any previous highlighting */
/* should be ok on very 1st call - assuming no highlighting */
/* is allowed to persist after a mode change */
    (data->atoms+atoms[0])->status &= ~SQUARE_HL;
/* save */
    atoms[0] = i;
/* highlight selected atom */
    (data->atoms+i)->status |= SQUARE_HL;
    break;
  case 2:
/* save */
    atoms[1] = i;
/* highlight selected atom */
    (data->atoms+i)->status |= SQUARE_HL;
/*
    g_snprintf(txt,12,"Selected %-2s", (data->atoms+i)->element);
*/
    break;
  case 3:
/* we now have three atoms we can calculate an angle for */
    atoms[2] = i;
/* remove highlights (we now have a label) */
    (data->atoms+atoms[0])->status &= ~SQUARE_HL;
    (data->atoms+atoms[1])->status &= ~SQUARE_HL;
/* compute vectors (from center atom) */
    angle = calc_angle(data, atoms[0], atoms[1], atoms[2]);
/* NEW - unified geometry label creation */
/* TODO - if atoms aren't bonded - draw a yellow line between them? */
/* eg a distance label with blank text as the value */
    g_snprintf(txt,8,"%7.3f",fabs(angle));
    geom_label(ANGLE, data, atoms, txt);
    update_geom_info();
/* reset counter */
    calls=0;
    break;
  default:
    printf("Error: bad state in info_angles()\n");
    calls=0;
    return;
  }
/* update */
redraw_canvas(SINGLE);
}

/****************************/
/* info on torsional angles */
/****************************/
void info_torsion(gint x, gint y, struct model_pak *data)
{
static gint calls=0, atoms[4]={0,0,0,0};
gint i;
gfloat angle;
gchar txt[40];

/* (kludgy) fix for picking only one atom, switching modes */
/* and then changing back, and having the first new atom */
/* really the last atom of the previous time */
/* clean up call (mode switching) */
if (x < 0)
  {
  calls=atoms[0]=atoms[1]=atoms[2]=atoms[3]=0;
  return;
  }

/* attempt to match point with an atom */
i = seek_atom(x, y, data);

/* no match - leave */
if (i < 0)
  return;
else
  calls++;

/* what stage are we at? */
switch (calls)
  {
  case 1:
/* FIXME - do we need to clear any previous highlighting? */
  case 2:
  case 3:
/* save */
    atoms[calls-1] = i;
/* highlight selected atom */
    (data->atoms+i)->status |= SQUARE_HL;
    break;
  case 4:
/* we now have four atoms we can calculate a dihedral for */
    atoms[3] = i;
/* remove highlights (we now have a label) */
    for(i=0 ; i<3 ; i++)
      (data->atoms+atoms[i])->status &= ~SQUARE_HL;
    angle = calc_dihedral(data, atoms[0], atoms[1], atoms[2], atoms[3]);
    g_snprintf(txt,8,"%7.3f",fabs(angle));
    geom_label(DIHEDRAL, data, atoms, txt);
    update_geom_info();
/* reset counter */
    calls=0;
    break;
  default:
    printf("Error: bad state in info_angles()\n");
    calls=0;
    return;
  }
/* update */
redraw_canvas(SINGLE);
}

/**************************/
/* callbacks for the list */
/**************************/
void select_label(GtkWidget *clist, gint row, gint column,
                         GdkEventButton *event, gpointer data)
{
selected_label = row;
}
void unselect_label(GtkWidget *clist, gint row, gint column,
                         GdkEventButton *event, gpointer data)
{
selected_label = -1;
}

/**************************/
/* NEW - geometric labels */
/**************************/
void geom_info()
{
gint id;
GtkWidget *frame, *vbox, *hbox, *button;
GtkWidget *scr_win;
gchar *titles[3] = {"  Type  ", " Constituent atoms ", " Value "};
struct dialog_pak *geom_dialog;

/* already got a dialog? */
if (!sysenv.num_models)
  return;

/* request a new dialog */
if ((id = request_dialog(sysenv.active, GEOMETRY)) < 0)
  return;
geom_dialog = &sysenv.dialog[id];

/* create new dialog */
geom_dialog->win = gtk_dialog_new();
gtk_window_set_title(GTK_WINDOW (geom_dialog->win), "Geometry panel");
gtk_window_set_default_size(GTK_WINDOW(geom_dialog->win), 250, 400);
gtk_signal_connect(GTK_OBJECT(geom_dialog->win), "destroy",
                   GTK_SIGNAL_FUNC(event_close_dialog), (gpointer) id);

/* Frame 1 */
frame = gtk_frame_new ("Measurement type");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(geom_dialog->win)->vbox),frame,FALSE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);

/* hbox for the modes */
hbox = gtk_hbox_new(TRUE, 0);
gtk_container_add(GTK_CONTAINER(frame), hbox);
gtk_container_set_border_width (GTK_CONTAINER(hbox), 5);

/* button 1 */
button = gtk_button_new_with_label ("Bond");
gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_mode),
                         (gpointer) BOND_INFO);

/* button 2 */
button = gtk_button_new_with_label (" Distance ");
gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_mode),
                         (gpointer) DIST_INFO);

/* button 3 */
button = gtk_button_new_with_label ("Angle");
gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_mode),
                         (gpointer) ANGLE_INFO);
/* button 4 */
button = gtk_button_new_with_label (" Dihedral ");
gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 0);
gtk_signal_connect_object(GTK_OBJECT (button), "clicked",
                          GTK_SIGNAL_FUNC (switch_mode),
                         (gpointer) DIHEDRAL_INFO);

/* Frame 2 */
frame = gtk_frame_new ("Label list");
gtk_box_pack_start(GTK_BOX(GTK_DIALOG(geom_dialog->win)->vbox),frame,TRUE,TRUE,0); 
gtk_container_set_border_width (GTK_CONTAINER(frame), 5);
/* create a vbox in the frame */
vbox = gtk_vbox_new (FALSE, 0);
gtk_container_add (GTK_CONTAINER(frame), vbox);
gtk_container_set_border_width (GTK_CONTAINER(GTK_BOX(vbox)), 3);
/* scrolled model pane */
scr_win = gtk_scrolled_window_new (NULL, NULL);
gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scr_win),
                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
gtk_box_pack_start(GTK_BOX(vbox), scr_win, TRUE, TRUE, 0);
/* Create the CList */
label_list = gtk_clist_new_with_titles(3, titles);
/* events */
gtk_signal_connect(GTK_OBJECT(label_list), "select_row",
                   GTK_SIGNAL_FUNC(select_label), NULL);
gtk_signal_connect(GTK_OBJECT(label_list), "unselect_row",
                   GTK_SIGNAL_FUNC(unselect_label), NULL);

/* set some properties of the CList */
gtk_clist_set_shadow_type (GTK_CLIST(label_list), GTK_SHADOW_OUT);
gtk_container_add(GTK_CONTAINER(scr_win), label_list);

/* Delete buttons */
hbox = gtk_hbox_new(TRUE, 0);
gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
/* col 1 - only one */
button = gtk_button_new_with_label ("  Delete  ");
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (delete_geom_label),
                           (gpointer) SINGLE);
/* col 2 - all labels */
button = gtk_button_new_with_label (" Clear all ");
gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC (delete_geom_label),
                           (gpointer) ALL);

/* terminating button */
button = gtk_button_new_with_label ("  Close  ");
gtk_box_pack_start (GTK_BOX (GTK_DIALOG (geom_dialog->win)->action_area),
                                         button, FALSE, FALSE, 0);
gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
                           GTK_SIGNAL_FUNC(close_dialog),
                           (gpointer) id);

/* done */
gtk_widget_show_all(geom_dialog->win);
/* refresh labels */
update_geom_info();
}

