/*
Copyright (C) 2000 by Sean David Fleming

sean@power.curtin.edu.au

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

The GNU GPL can also be found at http://www.gnu.org
*/

#include "config.h"
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#include "gdis.h"

/* the main pak structures */
extern struct sysenv_pak sysenv;
extern struct elem_pak elements[];
/* main model database */
struct model_pak *models[MAX_MODELS];

/*****************************/
/* default model orientation */
/*****************************/
/*
void init_rotmat(struct model_pak *data)
*/
void init_rotmat(gfloat *mat)
{
VEC3SET(&mat[0], 1.0, 0.0, 0.0);
VEC3SET(&mat[3], 0.0, 0.0,-1.0);
VEC3SET(&mat[6], 0.0, 1.0, 0.0);
/*
VEC3SET(&mat[0], 1.0, 0.0, 0.0);
VEC3SET(&mat[3], 0.0, 1.0, 0.0);
VEC3SET(&mat[6], 0.0, 0.0, 1.0);
*/
}

/***********************************/
/* unified initialization/cleaning */
/***********************************/
void template_model(struct model_pak *data)
{
/* NB: all the important defines should go here */
/* ie those values that might cause a core dump due to */
/* code that requires they at least have a value assigned  */
data->num_ghosts = 0;
data->num_atoms = 0;
data->num_orig = 0;
data->num_asym = 0;
data->num_bonds = 0;
data->num_ubonds = 0;
data->num_mols = 0;
data->num_shells = 0;
data->num_geom = 0;
data->num_images = 0;
data->num_frames = 0;
data->cur_frame = 0;
data->select_size = 0;
data->atom_info = -1;
data->shell_info = -1;
data->has_sof = FALSE;
data->sof_colourize = FALSE;
data->construct_pbc = FALSE;
/* linked list init */
data->bonds = NULL;
data->planes = NULL;
/* TODO - need for these will be phased out by linked lists */
data->atom_limit = DEFAULT_NA;
data->shell_limit = DEFAULT_NA;
data->ubond_limit = DEFAULT_NA;
data->geom_limit = DEFAULT_NL;
data->select_limit = DEFAULT_SS;

data->atoms = g_malloc(DEFAULT_NA*sizeof(struct atom_pak));
data->shells = g_malloc(DEFAULT_NA*sizeof(struct shell_pak));
data->ubonds = g_malloc(DEFAULT_NA*sizeof(struct bond_pak));
data->mols = g_malloc(sizeof(struct mol_pak));
(data->mols)->atom = g_malloc(sizeof(gint));
data->geom = g_malloc(DEFAULT_NL*sizeof(struct geom_pak));
data->select_list = g_malloc(DEFAULT_SS*sizeof(struct geom_pak));
/* vdW surface calc */
data->csurf.touch = g_malloc(sizeof(gint));
data->csurf.code = g_malloc(sizeof(gint));
data->csurf.h = g_malloc(sizeof(gfloat));
data->csurf.pa = g_malloc(sizeof(gfloat));
data->csurf.pb = g_malloc(sizeof(gfloat));
data->csurf.pc = g_malloc(sizeof(gfloat));
data->csurf.px = g_malloc(sizeof(gint));
data->csurf.py = g_malloc(sizeof(gint));
data->csurf.rx = g_malloc(sizeof(gfloat));
data->csurf.ry = g_malloc(sizeof(gfloat));
data->csurf.rz = g_malloc(sizeof(gfloat));
data->csurf.n[0] = g_malloc(sizeof(gfloat));
data->csurf.n[1] = g_malloc(sizeof(gfloat));
data->csurf.n[2] = g_malloc(sizeof(gfloat));
/* NEW - mysurf calc */
data->mysurf[SOLSURF].num_points = 0;
data->mysurf[SOLSURF].max_points = 0;
data->mysurf[MOLSURF].num_points = 0;
data->mysurf[MOLSURF].max_points = 0;
data->solsurf_on = FALSE;
data->molsurf_on = FALSE;
/* init flags */
data->periodic = 0;
data->fractional = FALSE;
data->axes_type = CARTESIAN;
data->axes_on = FALSE;
data->cell_on = FALSE;
data->mesh_on = FALSE;
data->box_on = FALSE;
data->csurf_on = FALSE;
data->asym_on = FALSE;
data->atom_labels = FALSE;
data->show_shells = FALSE;
/* can we eliminate da? I don't like it. */
data->da = 0.0;
data->scale = 1.0;
/* do NOT make these 0 */
data->mesh.px = 10;
data->mesh.py = 10;
/* init the model orientation */
init_rotmat(data->rotmat);
/* init translation */
data->offset[0] = 0;
data->offset[1] = 0;
data->offset[2] = 0;
/* this is now necessary due to the new latvec code */
/* ie standard cartesian axes */
data->pbc[0] = 1.0;
data->pbc[1] = 1.0;
data->pbc[2] = 1.0;
data->pbc[3] = 0.5*PI;
data->pbc[4] = 0.5*PI;
data->pbc[5] = 0.5*PI;
/* space group info */
data->sginfo.spacenum = 0;
data->sginfo.lattice = 0;
data->sginfo.pointgroup = 0;
data->sginfo.cellchoice = 0;
data->sginfo.order = 0;
data->sginfo.spacename = g_strdup("None");
data->sginfo.latticename = g_strdup("None");
/* NEW - store the raw SgInfo data */
data->sginfo.raw = g_malloc(1);
/* symmetry info */
data->symmetry.num_symops = 0;
data->symmetry.symops = g_malloc(sizeof(struct symop_pak));
data->symmetry.num_items = 1;
data->symmetry.items = g_malloc(2*sizeof(gchar *));
*(data->symmetry.items) = g_strdup("none");
*((data->symmetry.items)+1) = NULL;
data->symmetry.pg_entry = NULL;
data->symmetry.summary = NULL;
data->symmetry.pg_name = g_strdup("unknown");
/* periodic images */
data->image_limit[0] = 0.0;
data->image_limit[1] = 1.0;
data->image_limit[2] = 0.0;
data->image_limit[3] = 1.0;
data->image_limit[4] = 0.0;
data->image_limit[5] = 1.0;
/* existence */
data->region_empty[REGION1A] = FALSE;
data->region_empty[REGION2A] = TRUE;
data->region_empty[REGION1B] = TRUE;
data->region_empty[REGION2B] = TRUE;
/* lighting */
data->region[REGION1A] = TRUE;
data->region[REGION2A] = FALSE;
data->region[REGION1B] = FALSE;
data->region[REGION2B] = FALSE;
/* morphology */
data->type = LENGTH;
data->num_vertices = 0;
data->num_facets = 0;
data->num_planes = 0;

/* TODO - array->ptr switchover */
strcpy(data->filename,"none");
data->basename = g_strdup("none");
/* model tree widget */
data->tree_label=NULL;
/* widget - name that appears in the model tree */
data->tree_name = NULL;
/* default potential library */
strcpy(data->gulplib,LIBRARY);

/* gulp template init */
data->gulp.maxcyc=0;
data->gulp.num_potentials=0;
data->gulp.num_elem=0;
data->gulp.num_species=0;
data->gulp.energy=0.0;
data->gulp.no_esurf=FALSE;
data->gulp.esurf=0.0;
data->gulp.esurf_units=g_strdup(" ");
data->gulp.no_eatt=FALSE;
data->gulp.eatt=0.0;
data->gulp.eatt_units=g_strdup(" ");
data->gulp.sbulkenergy=0.0;
data->gulp.sdipole=999999999.0;
data->gulp.run=E_SINGLE;
data->gulp.free = FALSE;
data->gulp.zsisa = FALSE;
data->gulp.compare = FALSE;
data->gulp.method = -1;
data->gulp.optimiser = -1;
data->gulp.ensemble = -1;
data->gulp.temp = 0.0;
VEC3SET(data->gulp.super, 1, 1, 1);
data->gulp.coulomb = NOBUILD;
data->gulp.libfile = g_strdup(LIBRARY);
data->gulp.srcfile = NULL;
data->gulp.minimiser = NULL;
data->gulp.extra = NULL;
data->gulp.temp_file = g_strdup("none");
data->gulp.out_file = g_strdup("none");
data->gulp.dump_file = g_strdup("none");
data->gulp.lib_file = g_strdup("gdis.lib");
data->gulp.trj_file = g_strdup("none");
/* surface creation init */
data->surface.miller[0] = 1;
data->surface.miller[1] = 0;
data->surface.miller[2] = 0;
data->surface.shift = 0.0;
data->surface.dspacing = 0.0;
data->surface.region[0] = 1;
data->surface.region[1] = 1;
/* no non-default element data for the model (yet) */
data->num_elem = 0;
data->elements = NULL;
/* unique element list */
data->num_unique = 0;
/* file pointer */
data->animation = FALSE;
data->afp = NULL;
data->delay = 50;
}

/*************************************************/
/*  model display shuffle (usually after delete) */
/*************************************************/
void display_shuffle()
{
gint i;

/* adjust the displayed models */
for (i=0 ; i<sysenv.num_displayed ; i++)
  if (sysenv.displayed[i] > sysenv.num_models-1)
    sysenv.displayed[i] = -1;

/* adjust (if necessary) the currrently active model */
if (sysenv.num_models && sysenv.active >= sysenv.num_models)
  sysenv.active = sysenv.num_models-1;
}

/************************/
/* free model list data */
/************************/
/* TODO - more data will be lists */
void wipe_bond_list(struct model_pak *data)
{
GSList *tmp;

tmp = data->bonds;
while(tmp != NULL)
  {
  g_free(tmp->data);
  tmp = g_slist_next(tmp);
  }
g_slist_free(data->bonds);
data->bonds = NULL;
data->num_bonds = 0;
}
void wipe_plane_list(struct model_pak *data)
{
GSList *tmp;

tmp = data->planes;
while(tmp != NULL)
  {
  g_free(tmp->data);
  tmp = g_slist_next(tmp);
  }
g_slist_free(data->planes);
data->planes = NULL;
data->num_planes = 0;
}

/***************************/
/* delete the active model */
/***************************/
#define DEBUG_DELETE_MODEL 0
void delete_model(gint model)
{
gint i;
gchar txt[80];
struct model_pak *data;

/* exit if nothing is loaded */
if (!sysenv.num_models)
  return;

#if DEBUG_DELETE_MODEL
printf("delete_model() starting...\n");
#endif

/* force select on main tree item - avoids gtk-crits that occur */
/* when an unselect signal is sent to a (defunct) sub tree item */
tree_select(model);

#if DEBUG_DELETE_MODEL
printf("freeing assoc. dialogs...\n");
#endif
/* destroy any open dialogs assoc with the model */
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].model == model || sysenv.num_models == 1)
    {
    switch(sysenv.dialog[i].type)
      {
/* exceptions - only delete if no more models left */
      case POVRAY:
        if (sysenv.num_models > 1)
          break;
      default:
        close_dialog(i);
      }
    }
#if DEBUG_DELETE_MODEL
printf("shuffling dialog numbers...\n");
#endif
/* shuffle the dialog model numbers to account */
/* for the deletion of sysenv.active */
for (i=0 ; i<MAX_DIALOGS ; i++)
  if (sysenv.dialog[i].model > model)
    sysenv.dialog[i].model--;

#if DEBUG_DELETE_MODEL
printf("shuffling model numbers...\n");
#endif
/* shuffle the data->number to account */
/* for the deletion of sysenv.active */
for (i=0 ; i<sysenv.num_models ; i++)
  {
  if (i <= model)
    continue;
  data = model_ptr(i, RECALL);
  g_return_if_fail(data != NULL);
  data->number--; 
  }

/* FIXME - occasional problems here */
del_tree_item(model);

/* free the model's pointers */
data = model_ptr(model, RECALL);
g_return_if_fail(data != NULL);

#if DEBUG_DELETE_MODEL
printf("freeing string data...\n");
#endif
g_free(data->basename);
g_free(data->sginfo.spacename);
g_free(data->sginfo.latticename);
g_free(data->symmetry.pg_name);

#if DEBUG_DELETE_MODEL
printf("freeing coord data...\n");
#endif
g_free(data->atoms);
g_free(data->shells);
g_free(data->ubonds);
g_free(data->mols);

#if DEBUG_DELETE_MODEL
printf("freeing lists...\n");
#endif
wipe_bond_list(data);
wipe_plane_list(data);

#if DEBUG_DELETE_MODEL
printf("freeing mysurf data...\n");
#endif
if (data->mysurf[SOLSURF].max_points)
  {
  g_free(data->mysurf[SOLSURF].touch);
  g_free(data->mysurf[SOLSURF].px);
  g_free(data->mysurf[SOLSURF].py);
  g_free(data->mysurf[SOLSURF].x);
  g_free(data->mysurf[SOLSURF].y);
  g_free(data->mysurf[SOLSURF].z);
  }
if (data->mysurf[MOLSURF].max_points)
  {
  g_free(data->mysurf[MOLSURF].touch);
  g_free(data->mysurf[MOLSURF].px);
  g_free(data->mysurf[MOLSURF].py);
  g_free(data->mysurf[MOLSURF].x);
  g_free(data->mysurf[MOLSURF].y);
  g_free(data->mysurf[MOLSURF].z);
  }

#if DEBUG_DELETE_MODEL
printf("freeing symmetry data...\n");
#endif
g_free(data->symmetry.symops);
g_strfreev(data->symmetry.items);

#if DEBUG_DELETE_MODEL
printf("freeing measurements...\n");
#endif
for (i=0 ; i<data->num_geom ; i++)
  {
  g_free((data->geom+i)->list);
  g_free((data->geom+i)->text);
  }
g_free(data->geom);

#if DEBUG_DELETE_MODEL
printf("freeing morphology...\n");
printf("%d : %d : %d\n",data->num_planes, data->num_facets, data->num_vertices);
#endif
if ((data->num_planes))
  g_free(data->planes);
if ((data->num_vertices))
  {
  g_free(data->vertices->adj);
  g_free(data->vertices);
  }

free_gulp_data(data);

#if DEBUG_DELETE_MODEL
printf("freeing element data...\n");
#endif
if (data->num_elem)
  g_free(data->elements);

#if DEBUG_DELETE_MODEL
printf("closing animation stream...\n");
#endif
if (data->afp)
  fclose(data->afp);

#if DEBUG_DELETE_MODEL
printf("releasing slot...\n");
#endif
model_ptr(model, RELEASE);
g_snprintf(txt,17,"Deleted model %-2d",model);
show_text(txt);

/* adjust the displayed models */
display_shuffle();
/* highlight the new active model */
tree_select(sysenv.active);
/* redraw */
redraw_canvas(ALL);
#if DEBUG_DELETE_MODEL
printf("delete_model() completed.\n");
#endif
}

/**************************/
/* convenient signal hook */
/**************************/
void delete_active()
{
delete_model(sysenv.active);
}

/*******************************************/
/* get pointer to a requested model's data */
/*******************************************/
#define DEBUG_PTR 0
struct model_pak *model_ptr(gint model, gint mode)
{
struct model_pak *ptr=NULL;

/* how were we called? */
switch (mode)
  {
  case ASSIGN:
/* allocate a new pointer */
    ptr = g_malloc(sizeof(struct model_pak));
/* FIXME - append is inefficient */
    sysenv.mal = g_slist_append(sysenv.mal, ptr);
#if DEBUG_PTR
printf("Allocated: address=%p\n",ptr);
#endif
/* the model is new, so do some setup */
    sysenv.active = model;
    sysenv.num_models++;
/* record the model number (needed before calling template_model() */
    ptr->number = model;
/* standard settings/allocations */
    template_model(ptr);
    reset_view(NULL, ROTATION);
    break;

  case RECALL:
/* FIXME - there are an awful lot of calls to this; check for routines */
/* that pass the model number, when they could pass the model ptr instead */
#if DEBUG_PTR
printf("Request for model %d [%d,%d], ptr %p\n", model, 0, sysenv.num_models, ptr);
#endif
/* check for out of bounds request */
    if (model >= 0 && model < sysenv.num_models)
      ptr = (struct model_pak *) g_slist_nth_data(sysenv.mal, model);
    break;

/* delete a pointer */
  case RELEASE:
#if DEBUG_PTR
printf("Requested release of model=%d\n",model);
#endif
    if (model >= 0 && model < sysenv.num_models)
      {
/* get the model pointer and free it */
      ptr = (struct model_pak *) g_slist_nth_data(sysenv.mal, model);
      g_free(ptr);
/* remove it from the list */
      sysenv.mal = g_slist_remove(sysenv.mal, ptr);
      sysenv.num_models--;
      }
    break;
  }

return(ptr);
}

/*****************************/
/* coordinate memory testing */
/*****************************/
gint mem_check(struct model_pak *data, gint type)
{
gint num, lim;

switch(type)
  {
  case ATOM:
    num = data->num_atoms;
    lim = data->atom_limit;
    break;
  case SHELL:
    num = data->num_shells;
    lim = data->shell_limit;
    break;
  default:
    printf("mem_check(): unknown type requested.\n");
    return(0);
  }
/* need more; return non zero */
/*
if (num == lim)
*/
if (num == lim)
  return(1);
/* we're in trouble if this happens */ 
if (num > lim)
  {
  printf("Serious memory under allocation. (%d:%d)\n",num,lim);
  exit(0);
  }
return(0);
}

/****************************/
/* memory allocation expand */
/****************************/
#define DEBUG_MEM_GROW 0
gint mem_grow(struct model_pak *data, gint type, gint amount)
{
/* Don't have to do a case for MOLS - as the routine used */
/* enables allocation to be done perfectly by counting how */
/* many molecules there are BEFORE allocating - NOT possible */
/* (at least not efficiently possible) with atoms and bonds */

#if DEBUG_MEM_GROW
printf("Alloc'ing more memory...\n");
printf("for: ");
#endif

switch(type)
  {
  case ATOM:
    data->atom_limit += amount;
#if DEBUG_MEM_GROW
printf("atoms [%d]\n",data->atom_limit);
#endif
    data->atoms = g_renew(struct atom_pak, data->atoms, data->atom_limit);
    break;
  case BOND:
printf("Error: should not be called.\n");
    break;
  case UBOND:
    data->ubond_limit += amount;
#if DEBUG_MEM_GROW
printf("user bonds [%d]\n",data->ubond_limit);
#endif
    data->ubonds = g_renew(struct bond_pak, data->ubonds, data->ubond_limit);
    break;
  case SHELL:
    data->shell_limit += amount;
#if DEBUG_MEM_GROW
printf("shells [%d]\n",data->shell_limit);
#endif
    data->shells = g_renew(struct shell_pak, data->shells, data->shell_limit);
    break;
  case GEOM:
    data->geom_limit += amount;
#if DEBUG_MEM_GROW
printf("geometry labels [%d]\n",data->geom_limit);
#endif
    data->geom = g_renew(struct geom_pak, data->geom, data->geom_limit);
    break;
  case SELECT:
    data->select_limit += amount;
#if DEBUG_MEM_GROW
printf("selection size [%d]\n",data->select_limit);
#endif
    data->select_list = g_renew(gint, data->select_list, data->select_limit);
    break;

  default:
    printf("Unknown storage type requested.\n");
    return(1);
  }
#if DEBUG_MEM_GROW
printf(" [ok]\n");
#endif
return(0);
}

/****************************/
/* memory allocation shrink */
/****************************/
#define DEBUG_MEM_SHRINK 0
void mem_shrink(struct model_pak *data)
{
gint n, src, dest;

/* seek & overwrite any deleted atoms */
n=dest=src=0;
while (src < data->num_atoms)
 {
 if ((data->atoms+src)->status & DELETED)
   {
/* first time only */
   if (!dest)
     dest=src;
/* seek next undeleted atom */
   while (src < data->num_atoms)
     {
/* if we found something before termination, copy src to dest */
     if (!((data->atoms+src)->status & DELETED))
       {
       memcpy(data->atoms+dest,data->atoms+src,sizeof(struct atom_pak));
       (data->atoms+src)->status |= DELETED;
       dest++;
       n++;
       break;
       }
     src++;
     }
   }
 else
   {
/* fill space arising from src going beyond dest */
   if (src > dest) 
     {
     memcpy(data->atoms+dest,data->atoms+src,sizeof(struct atom_pak));
     (data->atoms+src)->status |= DELETED;
     }
   dest++;
   n++;
   }
/* next */
 src++;
 }
#if DEBUG_MEM_SHRINK
printf("[core] Old size: %d, new size: %d (%d)\n", data->num_atoms, n, dest);
#endif
/* shrink cores to new size, and adjust limits accordingly */
data->atoms = g_renew(struct atom_pak, data->atoms, n);
data->atom_limit = n;
data->num_atoms = n;

/* seek & overwrite any deleted shells */
n=dest=src=0;
while (src < data->num_shells)
 {
 if ((data->shells+src)->status & DELETED)
   {
/* first time only */
   if (!dest)
     dest=src;
/* seek next undeleted shell */
   while (src < data->num_shells)
     {
/* if we found something before termination, copy src to dest */
     if (!((data->shells+src)->status & DELETED))
       {
       memcpy(data->shells+dest,data->shells+src,sizeof(struct shell_pak));
       (data->shells+src)->status |= DELETED;
       dest++;
       n++;
       break;
       }
     src++;
     }
   }
 else
   {
/* fill space arising from src going beyond dest */
   if (src > dest) 
     {
     memcpy(data->shells+dest,data->shells+src,sizeof(struct shell_pak));
     (data->shells+src)->status |= DELETED;
     }
   dest++;
   n++;
   }
/* next */
 src++;
 }

#if DEBUG_MEM_SHRINK
printf("[shell] Old size: %d, new size: %d (%d)\n", data->num_shells, n, dest);
#endif

/* shrink shells to new size, and adjust limits accordingly */
data->shells = g_renew(struct shell_pak, data->shells, n);
data->shell_limit = n;
data->num_shells = n;


/* NEW - consequences!!! */
/* update labels - may ref non existent atoms */
check_geom(data);
}

/****************************************************************/
/* overlay a model that tracks the rotation of the active model */
/****************************************************************/
void add_ghost(struct model_pak *data, gint model)
{
struct model_pak *ptr;

/* CURRENT - only add the next model (if exists) as a ghost */
if (model < 0 || model >= sysenv.num_models)
  return;

if (data->num_ghosts)
  data->ghosts = g_renew(gint, data->ghosts, data->num_ghosts+1);
else
  data->ghosts = g_malloc(sizeof(gint));

*(data->ghosts+data->num_ghosts) = model;
data->num_ghosts++;

/* init rotation */
ptr = model_ptr(model, RECALL);
ARR3SET(&ptr->rotmat[0], &data->rotmat[0]);
ARR3SET(&ptr->rotmat[3], &data->rotmat[3]);
ARR3SET(&ptr->rotmat[6], &data->rotmat[6]);

init_objs(REDO_COORDS, data);
redraw_canvas(ALL);
}

/* TODO - improve */
void overlay_models_dialog()
{
struct model_pak *data;

data = model_ptr(sysenv.active, RECALL);
if (!data)
  return;
if (data->number+1 >= sysenv.num_models)
  return;

add_ghost(data, data->number+1);
}
