/* $Id: netpolling.cc,v 1.11 2002/03/10 18:37:10 bergo Exp $ */

/*

    GPS - Graphical Process Statistics
    Copyright (C) 1999-2002 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "transient.h"
#include "systype.h"

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <gtk/gtk.h>
#include <errno.h>
#include <pthread.h>

#include "polling.h"
#include "netpolling.h"
#include "gps.h"
#include "tstring.h"
#include "importglobals.h"

/* thread control */
int already_in=0,should_continue=0,final_result=0,what_im_doing=0;

void *thread_network_blocking_stuff(void *p) {
  NetworkActor *na;
  struct hostent *he;
  struct sockaddr_in sa;
  int i;
  char x;
  time_t there,here;

  // mu ha ha ha ha ha

  #ifdef ZZ_LINUX
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE,NULL);
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
  #endif

  na=(NetworkActor *)p;
  if ((already_in)||(p==NULL)) {
    should_continue=0;
    pthread_exit(NULL);
  }
  already_in++;

  there=time(NULL);

  /* real work starts */
  na->thesocket=socket(AF_INET,SOCK_STREAM,0);

  what_im_doing=0;
  he=gethostbyname(na->privateb);
  if (he==NULL) {
    goto hopen_will_exit_with_error;
  }

  sa.sin_family=he->h_addrtype;
  sa.sin_port=htons(na->thedoor);
  memcpy(&sa.sin_addr,he->h_addr_list[0],he->h_length);

  what_im_doing=1;
  i=connect(na->thesocket,(struct sockaddr *)&sa,sizeof(sa));
  if (i!=0) {
    goto hopen_will_exit_with_error;
  }

  if (read(na->thesocket,&x,1)<0)
    goto hopen_will_exit_with_error;

  /* real work ends */

  here=time(NULL);
  if (((here-there)>nettimeout)||(net_halt_request)) {
    close(na->thesocket);
    pthread_exit(NULL);
    return NULL;
  }

  /* all fine */
  final_result=0;
  already_in--;
  should_continue=0;
  pthread_exit(NULL);
  return NULL;

  hopen_will_exit_with_error:
  close(na->thesocket);

  here=time(NULL);
  if (((here-there)>nettimeout)||(net_halt_request)) {
    pthread_exit(NULL);
    return NULL;
  }

  final_result=-1;
  should_continue=0;
  already_in--;
  pthread_exit(NULL);
  return NULL;
}

/* the real work is done by thread_network_blocking_stuff in
   a separate thread */
int NetworkActor::hopen(char *hostname,int port) {
  pthread_t tid;
  int r;
  char buffer[512];
  time_t there,here;
  int lwid;

  is_special=0;

  if (gotgui) {
    snprintf(buffer,512,"Looking up host %s...",hostname);
    buffer[511]=0;
    gtk_label_set_text(GTK_LABEL(status_label[1]),buffer);
    while(gtk_events_pending()) gtk_main_iteration();
  }

  there=time(NULL);

  strcpy(privateb,hostname);
  thedoor=port;
  while(already_in) gtk_main_iteration();
  should_continue=1;
  what_im_doing=0;
  pthread_create(&tid,NULL,&thread_network_blocking_stuff,(void *)this);

  while(should_continue) {
    if (gotgui) {
      walk_wheel();

      if (what_im_doing!=lwid) {
	lwid=what_im_doing;
	switch(what_im_doing) {
	case 0:
	  snprintf(buffer,512,"Looking up host %s...",hostname);
	  buffer[511]=0;
	  break;
	case 1:
	  snprintf(buffer,512,"Opening connection to %s...",hostname);
	  buffer[511]=0;
	  break;
	}
	gtk_label_set_text(GTK_LABEL(status_label[1]),buffer);
      }

      usleep(46234);
      while (gtk_events_pending())
	gtk_main_iteration();
      here=time(NULL);

      // timeout or stop button clicked
      if (((here-there)>nettimeout)||(net_halt_request)) {

	#ifdef ZZ_LINUX
	pthread_cancel(tid);
	pthread_join(tid,NULL);
	#endif

	#ifdef ZZ_BSD
	pthread_detach(tid);
        #endif

	already_in=0;
	if (net_halt_request)
	  strcpy(buffer,"Failed! User abort.");
	else
	  strcpy(buffer,"Failed! Connection timed out.");
	gtk_label_set_text(GTK_LABEL(status_label[1]),buffer);
	while(gtk_events_pending()) gtk_main_iteration();
	usleep(400000);
	return(-1);
      }
    }
  }

  r=final_result;
  pthread_join(tid,NULL);
 
  return r;
}

int NetworkActor::hreadln(char *buffer,int max) {
  int i,v;
  char ph;
  if (ok<0) return -1;
  memset(buffer,0,max);
  for(i=0;i<max;i++) {
    v=read(thesocket,&buffer[i],1);
    if (v<=0) break;
    if (buffer[i]=='\n') {
      if (!is_special)
	if (strstr(buffer,"<special>")!=NULL)
	  is_special=1;
      return 0;
    }
  }
  if (strlen(buffer)>0) {
    do v=read(thesocket,&ph,1); while ((v>=0)&&(ph!='\n'));
    if (buffer[0]!='*')
      return 0;
  }
  return -1;
}

int NetworkActor::hclose() {
  char y[4];
  if (ok<0) return 0;
  strcpy(y,"q");
  write(thesocket,y,1);
  close(thesocket);
  is_special=0;
  return 0;
}

/* broken connection detection made here */
void NetworkActor::read_dead_output() {
  char x;
  if (ok<0) return;
  fcntl(thesocket,F_SETFL,O_NONBLOCK);
  while(read(thesocket,&x,1)>0) ; // read remaining data
  if (errno!=EAGAIN) { 
    ok=-1; close(thesocket); return;
  }
  fcntl(thesocket,F_SETFL,0);
}

int NetworkActor::isSpecial() {
  return((!ok)&&(is_special));
}

void NetworkActor::request_refresh() {
  char y[4];
  if (ok<0) return;
  strcpy(y," ");
  write(thesocket,y,strlen(y)); // ask for refresh
}

void NetworkActor::request_usage_header() {
  char y[4];
  if (!is_special) {
    request_refresh();
    return;
  }
  if (ok<0) return;
  strcpy(y,"h");
  write(thesocket,y,strlen(y)); // ask for new header
}

void NetworkActor::request_detailed_refresh() {
  char y[4];
  if (!is_special) {
    request_refresh();
    return;
  }
  if (ok<0) return;
  snprintf(y,4,"%c",12);
  y[3]=0;
  write(thesocket,y,strlen(y)); // ask for detailed refresh
}

/* **************************************************


     NetworkListPoller

        
   ************************************************** */


NetworkListPoller::NetworkListPoller(char *hname,int port) {
  strcpy(hostname,hname);
  tcpport=port;
  ok=-1;
  ok=hopen(hostname,port);
  process_list=NULL;
}

void NetworkListPoller::poll() {
  ProcessListItem *pli;
  char buffer[MAX_NET_LINE],b2[MAX_NET_LINE],b3[MAX_NET_LINE];
  int nprocesses,i;
  char *xp;
  TString *t;

  reset();

  if (ok<0)
    return;

  read_dead_output(); /* flush lost top output */

  if (ok<0)
    return; /* read_dead_output may have closed a refused connection */

  request_refresh();  /* request new top poll (with Space) */

  if (hreadln(buffer,1024)<0) return;
  if (hreadln(buffer,1024)<0) return;

  t=new TString(MAX_NET_LINE);
  t->set(buffer);
  nprocesses=atoi(t->token(" \n\t"));

  while(1) {
    if (hreadln(buffer,1024)<0) { delete t; return; }
    t->set(buffer);
    if (t->token(" \n\t")==NULL)
      break;
  }

  if (hreadln(buffer,1024)<0) { delete t; return; } /* header */

  for(i=0;i<nprocesses;i++) {
    pli=new ProcessListItem();
    pli->setField(MACHINE,hostname);

    if (hreadln(buffer,MAX_NET_LINE)<0) {
      delete pli;
      break;
    }

    t->set(buffer);
    xp=t->token(" \n\t");
    if (xp==NULL) {
      delete pli;
      break;
    }

    /* PID */
    strcpy(b2,xp);
    pli->setField(PID,b2);

    /* OWNER */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(OWNER,b2);

    /* PRIORITY */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(PRIORITY,b2);

    /* NICE */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(NICE,b2);

    /* SIZE */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(SIZE,b2);

    /* RSS */
    strcpy(b2,t->token(" \n\t"));
    pli->setField(RSS,b2);

    /* STATE */
    strcpy(b2,t->token(" \n\t"));
    if (strcmp(b2,"run")==0)
      pli->setField(STATE,"R");
    if (strcmp(b2,"sleep")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"stop")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"idl")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"WAIT")==0)
      pli->setField(STATE,"S");
    if (strcmp(b2,"zomb")==0)
      pli->setField(STATE,"Z");
   
    /* CPU TIME? */
    strcpy(b2,t->token(" \n\t"));

    /* CPU (WCPU in top) */
    strcpy(b2,t->token(" %\n\t"));
    snprintf(b3,MAX_NET_LINE,"%.1f",atof(b2));
    b3[MAX_NET_LINE-1]=0;
    pli->setField(CPU,b3);    

    strcpy(b2,t->token(" \n\t"));
    strcpy(b2,t->token(" \n\t"));

    if (strlen(b2)>255) b2[255]=0;
    pli->setField(NAME,b2);
    pli->setField(LONGNAME,b2);

    /* special pollers provide lots of info more, especially
       start time and cmdline, and may provide only cpu and
       memory, being much faster for history purposes */
    if (isSpecial()) {
      t->advance("!");
      strcpy(b2,t->token("!"));
      pli->setField(START,b2);
      t->advance("\010");
      strcpy(b2,t->token("\010\n"));
      if (strlen(b2)>511) b2[511]=0;
      pli->setField(LONGNAME,b2);
    } else
      pli->setField(START,"<not available>");

    process_list=g_list_append(process_list,(gpointer)pli);
  }
  read_dead_output();
  delete t;
}

void NetworkListPoller::terminate() {
  if (ok>=0)
    hclose();
}

void NetworkListPoller::reset() {
  GList *pt;
  if (process_list!=NULL) {
    for(pt=process_list;pt!=NULL;pt=g_list_next(pt))
      delete((ProcessListItem *)(pt->data));
    g_list_free(process_list);
    process_list=NULL;
  }
}

int NetworkListPoller::isOk() {
  if (ok==0)
    return 1;
  else
    return 0;
}

/* ************************************************************


     NetworkDetailsPoller
     (Remote details)


   ************************************************************ */

NetworkDetailsPoller::NetworkDetailsPoller(char *hname,int port) {
  item=NULL;
  tcpport=port;
  strcpy(hostname,hname);
  ok=-1;
  ok=hopen(hostname,port);
  netbuffer=(char *)g_malloc0(128<<10);
}

NetworkDetailsPoller::~NetworkDetailsPoller() {
  g_free(netbuffer);
}

/* reads a set of detailed information into the netbuffer */
void NetworkDetailsPoller::poll2buffer() {
  char buffer[MAX_NET_LINE];
  int nprocesses,i;
  TString *t;

  netbuffer[0]=0;

  if (item!=NULL) {
    delete item;
    item=NULL;
  } /* reset */

  if (ok<0)
    return; /* no retry on details */

  read_dead_output(); /* flush lost top output */

  if (ok<0)
    return;

  i=isSpecial();
  request_detailed_refresh();  /* request new top poll */

  if (hreadln(buffer,1024)<0) return;
  strcat(netbuffer,buffer);
  if (hreadln(buffer,1024)<0) return;
  strcat(netbuffer,buffer);

  if (isSpecial()!=i) {
    poll2buffer();
    return;
  }

  t=new TString(MAX_NET_LINE);
  t->set(buffer);
  nprocesses=atoi(t->token(" \n\t"));

  while(1) {
    if (hreadln(buffer,1024)<0) { delete t; return; }
    strcat(netbuffer,buffer);
    t->set(buffer);
    if (t->token(" \n\t")==NULL) break;
  }

  if (hreadln(buffer,1024)<0) return;
  strcat(netbuffer,buffer);

  for(i=0;i<nprocesses;i++) {
    if (hreadln(buffer,MAX_NET_LINE)<0) {
      delete t;
      return;
    }
    strcat(netbuffer,buffer);
  }
  read_dead_output();
  delete t;
}

void NetworkDetailsPoller::queryBuffer(int whichpid) {
  TString *t;
  int nprocesses,i;
  char buffer[MAX_NET_LINE],b2[MAX_NET_LINE],b3[MAX_NET_LINE];
  char *xp;

  if (item!=NULL) {
    delete item;
    item=NULL;
  } /* reset */

  if (!strlen(netbuffer))
    return;

  nb_gotop();
  nb_readln(buffer,1024);
  nb_readln(buffer,1024);

  t=new TString(MAX_NET_LINE);
  t->set(buffer);
  nprocesses=atoi(t->token(" \n\t"));

  while(1) {
    nb_readln(buffer,1024);
    t->set(buffer);
    if (t->token(" \n\t")==NULL) break;
  }

  nb_readln(buffer,1024);

  for(i=0;i<nprocesses;i++) {
    nb_readln(buffer,MAX_NET_LINE);
    t->set(buffer);
    xp=t->token(" \n\t");
    if (xp==NULL)
      break;

    /* PID */
    strcpy(b2,xp);

    if (atoi(b2)!=whichpid)
      continue;
    
    item=new ProcessItem();
    item->setField(PID,b2);
    item->setField(MACHINE,hostname);
    
    /* OWNER */
    strcpy(b2,t->token(" \n\t"));
    item->setField(OWNER,b2);
    
    /* PRIORITY */
    strcpy(b2,t->token(" \n\t"));
    item->setField(PRIORITY,b2);
    
    /* NICE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(NICE,b2);

    /* SIZE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(SIZE,b2);

    /* RSS */
    strcpy(b2,t->token(" \n\t"));
    item->setField(RSS,b2);

    /* STATE */
    strcpy(b2,t->token(" \n\t"));
    if (strcmp(b2,"run")==0)
      item->setField(STATE,"R");
    if (strcmp(b2,"sleep")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"stop")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"idl")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"WAIT")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"zomb")==0)
      item->setField(STATE,"Z");
   
    /* CPU TIME? */
    strcpy(b2,t->token(" \n\t"));
    
    /* CPU (WCPU in top) */
    strcpy(b2,t->token(" %\t\n"));
    snprintf(b3,MAX_NET_LINE,"%.1f",atof(b2));
    b3[MAX_NET_LINE-1]=0;
    item->setField(CPU,b3);    
    
    strcpy(b2,t->token(" \n\t"));
    
    strcpy(b2,t->token(" \n\t"));

    item->setField(NAME,b2);

    /* not parsed ? */
    if (isSpecial()) {

      t->advance("!");
      strcpy(b2,t->token("!"));
      item->setField(START,b2);

      t->advance("\010");
      strcpy(b2,t->token("\010\n"));
      if (strlen(b2)>511) b2[511]=0;
      item->setField(LONGNAME,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(PPID,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(PGID,b2);
      strcpy(b2,t->token(" \t\n"));

      item->setField(SID,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(TPGID,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(MINFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CMINFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(MAJFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CMAJFLT,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(USRJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CUSRJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(KRNJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CKRNJIFFIES,b2);
    } else {
      item->setField(START,"<not available>");
      item->setField(LONGNAME,"<not available>");
      item->setField(PPID,"<not available>");
      item->setField(PGID,"<not available>");
      item->setField(SID,"<not available>");
      item->setField(TPGID,"<not available>");
      
      item->setField(MINFLT,"<not available>");
      item->setField(CMINFLT,"<not available>");
      item->setField(MAJFLT,"<not available>");
      item->setField(CMAJFLT,"<not available>");
      
      item->setField(USRJIFFIES,"<not available>");
      item->setField(KRNJIFFIES,"<not available>");
      item->setField(CUSRJIFFIES,"<not available>");
      item->setField(CKRNJIFFIES,"<not available>");
    }

    /* not yet */
    item->setField(TTY,"<not available>");
    item->setField(FLAGS,"<not available>");
    item->setField(SIGPENDING,"<not available>");
    item->setField(SIGBLOCKED,"<not available>");
    item->setField(SIGIGNORED,"<not available>");
    item->setField(SIGCAUGHT,"<not available>");
    item->setField(RSSLIM,"<not available>");

    break;
  }
  delete t;
}

int NetworkDetailsPoller::nb_readln(char *dest,int lim) {
  char *pd;
  int c;
  pd=dest;
  memset(dest,0,lim-1);
  for(c=1;((nb_pos[0]!='\n')&&(c<lim));c++)
    *(pd++) = *(nb_pos++);
  *pd=0;
  nb_pos++;
  return(strlen(dest));
}

void NetworkDetailsPoller::nb_gotop() {
  nb_pos=netbuffer;
}

/* will make item NULL if not found */  
void NetworkDetailsPoller::poll(int whichpid) {
  char buffer[MAX_NET_LINE],b2[MAX_NET_LINE],b3[MAX_NET_LINE];
  int nprocesses,i;
  char *xp;
  TString *t;

  if (item!=NULL) {
    delete item;
    item=NULL;
  } /* reset */

  if (ok<0)
    return; /* no retry on details */

  read_dead_output(); /* flush lost top output */

  if (ok<0)
    return;

  i=isSpecial();
  request_detailed_refresh();  /* request new top poll */

  if (hreadln(buffer,1024)<0) return;
  if (hreadln(buffer,1024)<0) return;

  if (isSpecial()!=i) {
    poll(whichpid);
    return;
  }

  t=new TString(MAX_NET_LINE);
  t->set(buffer);
  nprocesses=atoi(t->token(" \n\t"));

  while(1) {
    if (hreadln(buffer,1024)<0) { delete t; return; }
    t->set(buffer);
    if (t->token(" \n\t")==NULL) break;
  }

  if (hreadln(buffer,1024)<0) return;

  for(i=0;i<nprocesses;i++) {
    if (hreadln(buffer,MAX_NET_LINE)<0) {
      delete t;
      return;
    }

    t->set(buffer);
    xp=t->token(" \n\t");
    if (xp==NULL)
      break;

    /* PID */
    strcpy(b2,xp);

    if (atoi(b2)!=whichpid)
      continue;
    
    item=new ProcessItem();
    item->setField(PID,b2);
    item->setField(MACHINE,hostname);
    
    /* OWNER */
    strcpy(b2,t->token(" \n\t"));
    item->setField(OWNER,b2);
    
    /* PRIORITY */
    strcpy(b2,t->token(" \n\t"));
    item->setField(PRIORITY,b2);
    
    /* NICE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(NICE,b2);

    /* SIZE */
    strcpy(b2,t->token(" \n\t"));
    item->setField(SIZE,b2);

    /* RSS */
    strcpy(b2,t->token(" \n\t"));
    item->setField(RSS,b2);

    /* STATE */
    strcpy(b2,t->token(" \n\t"));
    if (strcmp(b2,"run")==0)
      item->setField(STATE,"R");
    if (strcmp(b2,"sleep")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"stop")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"idl")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"WAIT")==0)
      item->setField(STATE,"S");
    if (strcmp(b2,"zomb")==0)
      item->setField(STATE,"Z");
   
    /* CPU TIME? */
    strcpy(b2,t->token(" \n\t"));
    
    /* CPU (WCPU in top) */
    strcpy(b2,t->token(" %\t\n"));
    snprintf(b3,MAX_NET_LINE,"%.1f",atof(b2));
    b3[MAX_NET_LINE-1]=0;
    item->setField(CPU,b3);    
    
    strcpy(b2,t->token(" \n\t"));
    
    strcpy(b2,t->token(" \n\t"));

    item->setField(NAME,b2);

    /* not parsed ? */
    if (isSpecial()) {

      t->advance("!");
      strcpy(b2,t->token("!"));
      item->setField(START,b2);

      t->advance("\010");
      strcpy(b2,t->token("\010\n"));
      if (strlen(b2)>511) b2[511]=0;
      item->setField(LONGNAME,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(PPID,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(PGID,b2);
      strcpy(b2,t->token(" \t\n"));

      item->setField(SID,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(TPGID,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(MINFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CMINFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(MAJFLT,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CMAJFLT,b2);

      strcpy(b2,t->token(" \t\n"));
      item->setField(USRJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CUSRJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(KRNJIFFIES,b2);
      strcpy(b2,t->token(" \t\n"));
      item->setField(CKRNJIFFIES,b2);
    } else {
      item->setField(START,"<not available>");
      item->setField(LONGNAME,"<not available>");
      item->setField(PPID,"<not available>");
      item->setField(PGID,"<not available>");
      item->setField(SID,"<not available>");
      item->setField(TPGID,"<not available>");
      
      item->setField(MINFLT,"<not available>");
      item->setField(CMINFLT,"<not available>");
      item->setField(MAJFLT,"<not available>");
      item->setField(CMAJFLT,"<not available>");
      
      item->setField(USRJIFFIES,"<not available>");
      item->setField(KRNJIFFIES,"<not available>");
      item->setField(CUSRJIFFIES,"<not available>");
      item->setField(CKRNJIFFIES,"<not available>");
    }

    /* not yet */
    item->setField(TTY,"<not available>");
    item->setField(FLAGS,"<not available>");
    item->setField(SIGPENDING,"<not available>");
    item->setField(SIGBLOCKED,"<not available>");
    item->setField(SIGIGNORED,"<not available>");
    item->setField(SIGCAUGHT,"<not available>");
    item->setField(RSSLIM,"<not available>");

    break;
  }
  read_dead_output();
  delete t;
}

void NetworkDetailsPoller::terminate() {
  if (ok>=0)
    hclose();
}

int NetworkDetailsPoller::isOk() {
  if (ok==0)
    return 1;
  else
    return 0;
}

int NetworkDetailsPoller::isHost(char *s) {
  if (strcmp(s,hostname)==0)
    return 1;
  else
    return 0;
}

/* ****************************************************


   NetworkSystemInfoProvider


   **************************************************** */

NetworkSystemInfoProvider::NetworkSystemInfoProvider(char *hname,int port) 
 : SystemInfoProvider() {
  int i;

  strcpy(hostname,hname);
  ok=-1;
  tcpport=port;
  ok=hopen(hostname,port);

  cpu_usage=0.0;
  memory_total=0;
  memory_used=0;
  memory_free=0;
  swap_total=0;
  swap_free=0;
  swap_used=0;

  for(i=0;i<256;i++)
    mem_shift[i]=0;
  mem_shift['K']=10;
  mem_shift['M']=20;
  mem_shift['G']=30;
  mem_shift['T']=40;
}

void NetworkSystemInfoProvider::update() {
  char buffer[1024],b2[512];
  int i,j;
  char *x;
  long m[6];
  TString *t;

  if (ok<0) return;

  read_dead_output(); /* flush lost top output */
  
  if (ok<0) return;
  
  request_usage_header();  /* request new top poll (with Space) */  
  
  if (hreadln(buffer,1024)<0) return;
  if (hreadln(buffer,1024)<0) return;
  if (hreadln(buffer,1024)<0) return;

  t=new TString(MAX_NET_LINE);
  t->set(buffer);

  /* CPU = 100%-(idle) */

  for(i=0;i<8;i++) { x=t->token(" \n\t"); if (x==NULL) { delete t; return; } }

  x=t->token(" %\t\n"); if (x==NULL) { delete t; return; }
  cpu_usage=(100.0-atof(x))/100.0;
  SMP_faker();

  /* MEMORY */
  
  if (hreadln(buffer,1024)<0) {
    memory_total=1;
    memory_used=0;
    memory_free=1;
    swap_total=1;
    swap_free=1;
    swap_used=0;
    delete t;
    return;
  }

  t->set(buffer);

  for(i=0;i<2;i++) { x=t->token(" \n\t"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[0]=atol(b2)<<mem_shift[x[j]];

  for(i=0;i<2;i++) { x=t->token(" \t\n"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[1]=atol(b2)<<mem_shift[x[j]];

  for(i=0;i<2;i++) { x=t->token(" \t\n"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[2]=atol(b2)<<mem_shift[x[j]];

  for (i=0;i<2;i++) { x=t->token(" \t\n"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[3]=atol(b2)<<mem_shift[x[j]];

  for(i=0;i<3;i++) { x=t->token(" \t\n"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[4]=atol(b2)<<mem_shift[x[j]];

  for(i=0;i<2;i++) { x=t->token(" \t\n"); if (x==NULL) { delete t; return; } }

  memset(b2,0,512);
  memcpy(b2,x,strlen(x)-1);
  j=strlen(x)-1;
  m[5]=atol(b2)<<mem_shift[x[j]];

  memory_total=m[0]+m[1];
  memory_used=m[0]-m[3];
  memory_free=memory_total-memory_used;

  swap_free=m[5];
  swap_used=m[4];
  swap_total=m[4]+m[5];

  read_dead_output(); /* flush lost top output */
  delete t;
}

void NetworkSystemInfoProvider::terminate() {
  if (ok>=0)
    hclose();
}

char *NetworkSystemInfoProvider::getHostname() {
  return(hostname);
}
