/***************************************************************************
 *
 *  $Id: kzencommandthread.h,v 1.13 2005/07/03 17:01:48 muszilla Exp $
 *
 *  Copyright (C) 2005 by Andreas Mussgiller
 *  muszilla@users.sourceforge.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the
 *  Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 ***************************************************************************/

#ifndef KZENCOMMANDTHREAD_H
#define KZENCOMMANDTHREAD_H

#include <queue>

#include <libnjb.h>

#include <qthread.h>
#include <qptrqueue.h> 
#include <qptrlist.h> 
#include <qdatetime.h> 
#include <qevent.h> 

class KURL;

class KZenCommand;
class KZenExplorer;
class KZenTrack;
class KZenPlayList;
class KZenArtist;
class KZenAlbum;

/**
@author Andreas Mussgiller
*/
class KZenCommandThread : public QThread
{
 public:
  
  KZenCommandThread(KZenExplorer *);
  ~KZenCommandThread();

  void                        setNJB(njb_t *);

  void                        stop() { fRun = false; }
  void                        run();

  void                        processCommand(KZenCommand *);

  void                        updateUsage();
  void                        addTrack(KZenTrack *, const KURL &, KZenPlayList *);
  void                        saveTrack(KZenTrack *, const KURL &);
  void                        newPlayList(const QString &);
  void                        newPlayList(KZenPlayList *);
  void                        newSmartPlayList(KZenPlayList *);
  void                        updateTrack(KZenTrack *);
  void                        updateTrackInline(KZenTrack *);
  void                        updatePlayList(KZenPlayList *);
  void                        renamePlayList(KZenPlayList *, const QString &);
  void                        destroyTrack(KZenTrack *);
  void                        deletePlayList(KZenPlayList *);
  void                        getOwnerString();
  void                        setOwnerString(const QString &);
  void                        getClock();
  void                        setClock(const QDateTime &);
  void                        playTrack(KZenTrack *);
  void                        queueTrack(KZenTrack *);
  void                        stopPlay();
  void                        updatePlayProgress();

  void                        lock() { fLock = true; }
  void                        unlock() { fLock = false; }

  bool                        cancelRequest() { return fCancelRequest; }
  void                        setCancelRequest(bool r) { fCancelRequest = r; }

  void                        setPercentage(int percentage);

  static int                  progress(u_int64_t sent,
				       u_int64_t total,
				       const char* buf,
				       unsigned len,
				       void *data);

  bool                        isPlaying() { return fPlaying; }

 protected:

  bool                        fRun;
  bool                        fLock;
  KZenExplorer               *fExplorer;
  QPtrQueue<KZenCommand>      fCommandQueue;

  static u_int64_t            fBytesSent;
  static u_int64_t            fBytesTotal;
  int                         fCurrentPercentage;

  bool                        fCancelRequest;

  std::queue<int>             fTrackQueue;
  bool                        fPlaying;
};

class KZenCommand
{
 public:
  
  enum CommandType {
    CmdUpdateUsage = 0,
    CmdAddTrack,
    CmdSaveTrack,
    CmdNewPlayList,
    CmdNewSmartPlayList,
    CmdUpdateTrack,
    CmdUpdateTrackInline,
    CmdUpdatePlayList,
    CmdRenamePlayList,
    CmdDestroyTrack,
    CmdDeletePlayList,
    CmdGetOwnerString,
    CmdSetOwnerString,
    CmdGetClock,
    CmdSetClock,
    CmdPlayTrack,
    CmdQueueTrack,
    CmdStopPlay,
    CmdUpdatePlayProgress,
    CmdUnknown
  };

  KZenCommand(CommandType);
  KZenCommand(CommandType, const QString &);
  KZenCommand(CommandType, const QDateTime &);
  KZenCommand(CommandType, KZenTrack *);
  KZenCommand(CommandType, KZenTrack *, KZenPlayList *);
  KZenCommand(CommandType, KZenTrack *, const KURL &);
  KZenCommand(CommandType, KZenTrack *, const KURL &, KZenPlayList *);
  KZenCommand(CommandType, KZenPlayList *);
  KZenCommand(CommandType, KZenPlayList *, const QString &);
  ~KZenCommand();
  
  CommandType                 getCommandType() { return fCmdType; }
  KZenTrack *                 getTrack() { return fTrack; }
  KZenPlayList *              getPlayList() { return fPlayList; }
  const QString &             getName() { return fName; }
  const QDateTime &           getDateTime() { return fDateTime; }
  const KURL &                getURL() { return fURL; }

 protected:

  CommandType                 fCmdType;
  KZenTrack                  *fTrack;
  KZenPlayList               *fPlayList;
  QString                     fName;
  QDateTime                   fDateTime;
  KURL                        fURL;
};

class KZenProgressEvent : public QCustomEvent
{
 public:
 
  KZenProgressEvent(int progress)
    :QCustomEvent(12345) { fProgress = progress; }

  int                         getProgress() { return fProgress; }

 protected:

  int                         fProgress;
};

class KZenItemEvent : public QCustomEvent
{
 public:
 
  KZenItemEvent(KZenTrack * track)
    :QCustomEvent(12346) { fTrack = track; }

  KZenTrack *                 getTrack() { return fTrack; }

 protected:

  KZenTrack                  *fTrack;
};

#endif
