/***************************************************************************
                          asyncfile.cpp  -  description
                             -------------------
    begin                : Sat Sep 1 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "mthread.h"
#include <unistd.h>
#include <iostream.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#include "tstring.h"
#include "asyncfile.h"

class MAFThread : public MThread {
public:
	MAFThread();
	~MAFThread();
	void OnRequestCompleted(AsyncFileRequest*);
	bool HaveRequests(MAsyncFile* pFrom);
	bool HaveRequests(MAsyncFile* pFrom, int nRequestMask);
	static int GetConflictingRequestMask(AFRType type);
	static bool CheckAndCreatePath(const CString& path);
	void run();
	void SendStopRequest();
	//
	MMutex                       m_mtxRequest;
	MWaitCondition               m_waitRequest;
	std::deque<AsyncFileRequest> m_queueRequest;
private:
	bool _eof(int hFile)
	{
		off_t pos = lseek(hFile,0,SEEK_CUR);
		if (pos == lseek(hFile,0,SEEK_END))
			return true;
		lseek(hFile,pos,SEEK_SET);
		return false;
	}
	long _tell(int hFile)
	{
		return lseek(hFile,0,SEEK_CUR);
	}
};

class MAFContext
{
public:
	MMutex mutex;
	std::set<MAsyncFile*> setAsyncFiles;
	MAFThread* pAFThreadRead;
	MAFThread* pAFThreadWrite;
	int nReadClients;
	int nWriteClients;
	//
	MAFContext(){
		pAFThreadRead = NULL;
		pAFThreadWrite = NULL;
	}
	~MAFContext(){
	}
	void WaitForThreadsStop()
	{
		ASSERT(0==nReadClients);
		ASSERT(0==nWriteClients);
		if (pAFThreadRead && 0==nReadClients)
		{
			TRACE("waiting for async read thread...");
			pAFThreadRead->SendStopRequest();
			//pAFThreadRead->wait();
			delete pAFThreadRead;
		}
		if (pAFThreadWrite && 0==nWriteClients)
		{
			TRACE("waiting for async write thread...");
			pAFThreadWrite->SendStopRequest();
			//pAFThreadWrite->wait();
			delete pAFThreadWrite;
		}
	}
};

static MAFContext s_context;

MAFThread::MAFThread()
{
}

MAFThread::~MAFThread()
{
}

void MAFThread::OnRequestCompleted(AsyncFileRequest* pRequest)
{
	MLock lock(s_context.mutex);
	// here we have to notify the MAsyncFile object about completion
	MAsyncFile* pAF = (MAsyncFile*) pRequest->dwAsyncFileID;
	if (s_context.setAsyncFiles.end()==s_context.setAsyncFiles.find(pAF))
	{
		return;// file object has gone
	}
	if (pRequest->type == AFR_FREE)
	{
		// or its about to go
		pAF->m_pThread = NULL;
		if (pAF->m_nMode == AFM_READ)
			--s_context.nReadClients;
		else
			--s_context.nWriteClients;
		delete pAF;
		s_context.setAsyncFiles.erase(pAF);
		return;
	}	
	m_mtxRequest.lock();
	pAF->m_pRequest->type = AFR_NONE;
	pAF->m_pRequest->type_compl = pRequest->type;
	pAF->m_pRequest->hFile = pRequest->hFile;
	pAF->m_pRequest->nErrNo = pRequest->nErrNo;
	pAF->m_pRequest->bEoF = pRequest->bEoF;
	pAF->m_pRequest->nReturn = pRequest->nReturn;
	pAF->m_pRequest->nFilePos = pRequest->nFilePos;
	pAF->m_pRequest->nFileSize = pRequest->nFileSize;
	if (pRequest->type == AFR_PREREAD && pRequest->nErrNo==0)
	{
		ASSERT(pRequest->buffer==pAF->m_pBuffer);
		pAF->m_nPreReadBytes = pRequest->nReturn;
		pAF->m_pData = pRequest->buffer;
	}
	else
	{
	    pAF->m_nPreReadBytes = 0;
	    pAF->m_pData = NULL;
	}
	// we need this to enable callback-functions to do whatever they want, e.g. delete this.
	m_mtxRequest.unlock();
	s_context.mutex.unlock();
	if (pRequest->nErrNo)
		pAF->OnError();
	else
		pAF->OnSuccess();
	//
	s_context.mutex.lock();
	if (s_context.setAsyncFiles.end()==s_context.setAsyncFiles.find(pAF))
		return; // the object has dissapeared
	m_mtxRequest.lock();
	pAF->m_pRequest->type_compl = AFR_NONE;
	// wake up the object if there is no more requests from it
	if (HaveRequests(pAF))
	{
		m_mtxRequest.unlock();
		return;
	}
	
	pAF->m_waitReady.wakeAll();
	m_mtxRequest.unlock();
}

bool MAFThread::MAFThread::HaveRequests(MAsyncFile* pFrom)
{
	ASSERT(m_mtxRequest.locked());
	for (std::deque<AsyncFileRequest>::iterator it = m_queueRequest.begin(); it != m_queueRequest.end(); ++it)
	{
		if (((MAsyncFile*) (it->dwAsyncFileID)) == pFrom)
		{
			return true;
		}
	}
	return false;
}

bool MAFThread::HaveRequests(MAsyncFile* pFrom, int nRequestMask)
{
	ASSERT(m_mtxRequest.locked());
	for (std::deque<AsyncFileRequest>::iterator it = m_queueRequest.begin(); it != m_queueRequest.end(); ++it)
	{
		if (((MAsyncFile*) (it->dwAsyncFileID)) == pFrom &&
			(it->type & nRequestMask))
		{
			return true;
		}
	}
	return false;
}

int MAFThread::GetConflictingRequestMask(AFRType type)
{
	// this function describes relation between requests
	// return value is a bit-mask of the request for
	// wich completion should the current one wait
	switch (type)
	{
		case AFR_CLOSE:
			return 0; // fully async -- does not depend on anything
		// file state group
		case AFR_OPEN:
		case AFR_ATTACH:
		case AFR_DETACH:
			return AFR_OPEN | AFR_ATTACH | AFR_CLOSE | AFR_DETACH;
		// rather relaxed group
		case AFR_SEEK:
			return AFR_OPEN | AFR_ATTACH | AFR_CLOSE | AFR_DETACH;
		//
		case AFR_PREREAD:
			return AFR_OPEN | AFR_ATTACH | AFR_SEEK | AFR_PREREAD | AFR_READ | AFR_WRITE | AFR_CLOSE | AFR_DETACH;
		// the hardest group -- they conflict with any
		case AFR_WRITE:
		case AFR_GETBUF:
		case AFR_READ:
		case AFR_WAIT:
		case AFR_STOP: // this makes stop request syncronous
			return AFR_OPEN | AFR_ATTACH | AFR_SEEK | AFR_PREREAD | AFR_READ | AFR_WRITE | AFR_CLOSE | AFR_DETACH;
	}
	TRACE("MAFThread::GetConflictingRequestMask: unknown request type");
	return AFR_OPEN | AFR_ATTACH | AFR_SEEK | AFR_PREREAD | AFR_READ | AFR_WRITE | AFR_CLOSE | AFR_DETACH | AFR_STOP; // we dont like all requests
}

bool MAFThread::CheckAndCreatePath(const CString& path)
{
	// remove the last element from the path
	CString part_path = path.substr(0,path.rfind("/"));
	while (part_path.length() && part_path[part_path.length()-1]=='/')
		part_path = part_path.substr(0,part_path.length()-1);
	if (!part_path.length())
		return false;
	// stat the path to check if the directory exists
	struct stat st;
	if (0==stat(part_path.c_str(), &st) && (S_ISDIR(st.st_mode)))
		return true;
	// it doesnt exist -- try to create
	TRACE3("MAFThread::CheckAndCreatePath: creating the directory \'", part_path, "\'");
	if (0==mkdir(part_path.c_str(), S_IRWXU))
		return true;
	// if cannot create -- try to do it recursively up the tree
	if (!CheckAndCreatePath(part_path))
		return false;
	// tree up to the current location has been created somehow -- now create the current dir
	TRACE3("MAFThread::CheckAndCreatePath: creating the directory \'", part_path, "\'");
	return 0==mkdir(part_path.c_str(), S_IRWXU);
}

void MAFThread::run()
{
	AsyncFileRequest request;
	m_mtxRequest.lock();
	int type_cache;
	while (1)
	{
		// request mutex is locked here
		if (m_queueRequest.size()==0)
			m_waitRequest.wait(&m_mtxRequest);
		// request mutex is still locked here
		request=m_queueRequest.front();
		m_mtxRequest.unlock();
		//
		if (request.type & AFR_SEEK)
		{
			if (request.hFile>=0)
			{
				ASSERT(_tell(request.hFile) == request.nFilePos);
				request.nReturn = lseek(request.hFile, request.offset, request.whence);
				if (request.nReturn<0)
					request.nErrNo = errno;
				else
					request.nErrNo = 0;
				request.nFilePos = _tell(request.hFile);
				request.bEoF = _eof(request.hFile);
			}
		}
		switch ( request.type )
		{
			case AFR_OPEN:
				if (request.hFile>=0)
				{
					//TRACE("MAFThread: file was already open");
					close(request.hFile);
				}
				if (request.mode & AFM_WRITE)
				{
					request.hFile = open(request.name, O_CREAT|O_RDWR/*|O_BINARY*/, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
					if (request.hFile < 0 && (request.mode & AFM_CREATEPATH))
					{
						// my be we still can fix this little problem
						if (CheckAndCreatePath(request.name))
						{
							// now try again
							request.hFile = open(request.name, O_CREAT|O_RDWR/*|O_BINARY*/, S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
						}
					}
				}
				else
					request.hFile = open(request.name, O_RDONLY/*|O_BINARY*/ );
				//
				request.nReturn = request.hFile;
				if (request.nReturn<0)
					request.nErrNo = errno;
				else
					request.nErrNo = 0;
			case AFR_ATTACH:
				if (request.hFile>=0)
				{
					request.nFileSize = lseek(request.hFile,0,SEEK_END);
					lseek(request.hFile,0,SEEK_SET);
					request.bEoF = _eof(request.hFile);
				}
				else
				{
					request.nFileSize = 0;
					request.bEoF = true;
				}
				request.nFilePos = 0;
				request.nReturn = 0;
				request.nErrNo = 0;
				//if (!(m_request.mode & AFM_READ)) //if we are opening the file in 'read' mode its good to pre-read data
				break;
			case AFR_PREREAD|AFR_SEEK:
			case AFR_READ|AFR_SEEK:
			case AFR_PREREAD:
			case AFR_READ:
				if (request.hFile>=0)
				{
					//TODO: properly react here -- say close file
					ASSERT(_tell(request.hFile) == request.nFilePos);
					request.nReturn = read(request.hFile, request.buffer,request.size);
					if (request.nReturn <= 0)
						request.nErrNo = errno;
					else
					{
						request.nFilePos += request.nReturn;
						request.nErrNo = 0;
					}
					request.bEoF = _eof(request.hFile);
				}
				break;
			case AFR_WRITE|AFR_SEEK:
			case AFR_WRITE:
				if (request.hFile>=0)
				{
					//TODO: properly react here -- say close file
					ASSERT(_tell(request.hFile) == request.nFilePos);
					request.nReturn = write(request.hFile,request.buffer,request.size);
					if (request.size!=request.nReturn)
					{
						request.nErrNo = errno;
					}
					if (request.nReturn > 0)
					{
						request.nFilePos += request.nReturn;
						request.nErrNo = 0;
						if (request.nFilePos>request.nFileSize)
							request.nFileSize = request.nFilePos;
					}
					request.bEoF = _eof(request.hFile);
				}
				break;
			case AFR_CLOSE:
			case AFR_STOP:
			case AFR_FREE:
				if (request.hFile>=0)
					request.nReturn = close(request.hFile);
				else
					request.nReturn = 0;
				request.hFile = -1;
				if (request.nReturn<0)
					request.nErrNo = errno;
				else
					request.nErrNo = 0;
				break;
			case AFR_DETACH:
				request.nReturn = 0;
				request.nErrNo = 0;
				request.hFile = -1;
				break;
			case AFR_NONE:
				if (request.type != AFR_SEEK)
				{
					TRACE("AFR_NONE: should not happen");
				}
			case AFR_SEEK:
				break;
			default:
				TRACE2("unknown async-file request ", request.type);
		}
		m_mtxRequest.lock();
		m_queueRequest.pop_front();
		m_mtxRequest.unlock();
		if ((request.type & AFR_SEEK) &&
			(request.type & (AFR_READ|AFR_WRITE|AFR_PREREAD)))
		{
			type_cache = request.type;
			request.type = AFR_SEEK;
			OnRequestCompleted(&request);
			request.type = type_cache & ~AFR_SEEK;
		}
		OnRequestCompleted(&request);
		if (request.type == AFR_STOP)
		{
			// queue mutex is unlocked here
			return;
		}
		
		m_mtxRequest.lock();
	}
}

void MAFThread::SendStopRequest()
{
	AsyncFileRequest request;
	request.hFile = -1;
	request.type = AFR_STOP;
	m_mtxRequest.lock();
	m_queueRequest.push_back(request);
	m_waitRequest.wakeAll();
	wait(&m_mtxRequest);
	m_mtxRequest.unlock();
}

#define MAF_REQUEST_CHECK(_request, _bad_return)\
	m_pThread->m_mtxRequest.lock();\
	while (m_pThread->HaveRequests(this, MAFThread::GetConflictingRequestMask(_request)))\
	{\
		if (m_pThread->isCurrent())\
		{\
			m_pThread->m_mtxRequest.unlock();\
			return _bad_return;\
		}\
		m_waitReady.wait(&m_pThread->m_mtxRequest);\
	}

#define MAF_REQUEST_END                               \
	m_pThread->m_queueRequest.push_back(*m_pRequest); \
	m_pThread->m_waitRequest.wakeAll();               \
	m_pThread->m_mtxRequest.unlock();
// calling wakeAll() is inefficient in terms of mumber of context switches
// but it required to eliminate race condition here, when the request
// possibly get processed before the 'wakeAll' call. This results in a
// crash if the object get deleted in the request callback function
	
MAsyncFile::MAsyncFile(int mode /*=AFM_READWRITE*/, int nBufSize = 4096) //: m_mtxWait(true) //TODO: find a better way
{
	
	m_nMode = mode;
	m_nBufferSize = nBufSize;
	ASSERT(m_nBufferSize>0);
	m_pBuffer = NULL;
	m_pData = NULL;
	m_nPreReadBytes = 0;
	m_szName = NULL;
	//
	m_pRequest = new AsyncFileRequest;
	m_pRequest->dwAsyncFileID = (u_int)this;
	m_pRequest->type = AFR_NONE;
	m_pRequest->type_compl = AFR_NONE;
	m_pRequest->hFile=-1;
	//
	s_context.mutex.lock();
	s_context.setAsyncFiles.insert(this);
	if (mode == AFM_READ)
	{
		if (s_context.pAFThreadRead==NULL)
		{
			s_context.pAFThreadRead = new MAFThread();
			s_context.pAFThreadRead->start();
		}
	    m_pThread=s_context.pAFThreadRead;
	    ++s_context.nReadClients;
	}
	else
	{
		if (s_context.pAFThreadWrite==NULL)
		{
			s_context.pAFThreadWrite = new MAFThread();
			s_context.pAFThreadWrite->start();
		}
	    m_pThread=s_context.pAFThreadWrite;
	    ++s_context.nWriteClients;
	}
	s_context.mutex.unlock();
}

MAsyncFile::~MAsyncFile()
{
	ASSERT(m_pThread==NULL);
	//
	delete [] m_pBuffer;
	delete m_pRequest;
	if (m_szName)
		delete [] m_szName;
}

void MAsyncFile::WaitForThreadsStop()
{
	s_context.WaitForThreadsStop();
}

void MAsyncFile::OnSuccess()
{
	//cout << "success\n";
}

void MAsyncFile::OnError()
{
	//cout << "error\n";
}

bool MAsyncFile::IsInProgress()
{
	MLock lock(m_pThread->m_mtxRequest);
	
	return m_pThread->HaveRequests(this);
}

bool MAsyncFile::IsReady()
{
	MLock lock(m_pThread->m_mtxRequest);
	
	return !m_pThread->HaveRequests(this);
}

bool MAsyncFile::WaitTillReady()
{
	MAF_REQUEST_CHECK(AFR_WAIT, false)
	m_pThread->m_mtxRequest.unlock();
}

bool MAsyncFile::IsOpen()
{
	return m_pRequest->hFile>=0;
}

void MAsyncFile::Destroy()
{
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	//MAF_REQUEST_CHECK(AFR_FREE, false)
	m_pThread->m_mtxRequest.lock();
	m_pRequest->type = AFR_FREE;
	MAF_REQUEST_END
}

bool MAsyncFile::Open(LPCSTR szName, bool bBlock /*=false*/)
{
	ASSERT(szName);
	ASSERT(strlen(szName)<65536);
	// copy the name
	if (m_szName)
		delete [] m_szName;
	m_szName = new char[strlen(szName)+1];
	strcpy(m_szName, szName);
	//
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_OPEN, false)
	m_pRequest->type = AFR_OPEN;
	m_pRequest->name = m_szName;
	m_pRequest->mode = m_nMode;
	MAF_REQUEST_END
	//
	if (bBlock)
	{
		VERIFY(WaitTillReady());
		return m_pRequest->hFile>=0;
	}
	return true;
}

bool MAsyncFile::IsClosed()
{
	return m_pRequest->hFile<0;
}

bool MAsyncFile::Close(bool bBlock /*=false*/)
{
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_CLOSE, false)
	m_pRequest->type = AFR_CLOSE;
	MAF_REQUEST_END
	//
	if (bBlock)
		VERIFY(WaitTillReady());
	return true;
}

int MAsyncFile::GetLastError()
{
	return m_pRequest->nErrNo;
}

bool MAsyncFile::Attach(int hFile)
{	
	if (m_pRequest->hFile<0)
		Close(true);
	{
		MAF_REQUEST_CHECK(AFR_ATTACH, false)
		m_pRequest->type = AFR_ATTACH;
		m_pRequest->hFile = hFile;
		MAF_REQUEST_END
	}
	return true;
}

int MAsyncFile::Detach()
{
	MAF_REQUEST_CHECK(AFR_DETACH, -1)
	m_pRequest->type = AFR_DETACH;
	int hFile = m_pRequest->hFile;
	MAF_REQUEST_END
	VERIFY(WaitTillReady());
	return hFile;
}

bool MAsyncFile::Seek(int nOffset, int nWhence, bool bBlock /*=false*/)
{
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	if ( nWhence == SEEK_CUR && nOffset == 0 )
		return true;
	{
		MAF_REQUEST_CHECK(AFR_SEEK, false)
		m_pRequest->type = AFR_SEEK;
		m_pRequest->offset = nOffset;
		m_pRequest->whence = nWhence;
		MAF_REQUEST_END
		if (bBlock)
			VERIFY(WaitTillReady());
	}
	return true;
}

int MAsyncFile::GetPos()
{
	WaitTillReady();
	if (!IsOpen() || m_pRequest->nErrNo != 0)
		return -1;
	return m_pRequest->nFilePos-m_nPreReadBytes; // pre-read bytes are our internal business
}

int MAsyncFile::GetSize()
{
	WaitTillReady();
	if (!IsOpen() || m_pRequest->nErrNo != 0)
		return -1;
	return m_pRequest->nFileSize;
}

int MAsyncFile::BytesAvail()
{
	WaitTillReady();
	return m_nPreReadBytes;
}

bool MAsyncFile::EoF()
{
	return !BytesAvail() && m_pRequest->bEoF;
}

bool MAsyncFile::Prefetch()
{
	if (m_mtxBuffer.locked())
	{
		TRACE("MAsyncFile::Read: buffer is locked");
		return false;
	}
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_PREREAD, false)
	
	if (0==(m_nMode&AFM_READ) ||
	    m_pRequest->hFile<0 ||
		(m_pRequest->type == AFR_NONE && m_nPreReadBytes) ||
		(m_pRequest->type == AFR_PREREAD))
	{
		m_pThread->m_mtxRequest.unlock();
		return false;
	}
	
	ASSERT(m_nPreReadBytes == 0);
	if (!m_pBuffer)
	{
		m_pBuffer = new char[m_nBufferSize];
	}
	m_pRequest->type = AFR_PREREAD;
	m_pRequest->buffer = m_pBuffer;
	m_pRequest->size = m_nBufferSize;
	MAF_REQUEST_END
	return true;
}

bool MAsyncFile::PrefetchSeek(int nOffset, int nWhence) //prefetch with seek
{
	if (m_mtxBuffer.locked())
	{
		TRACE("MAsyncFile::Prefetch: buffer is locked");
		return false;
	}
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_PREREAD, false)
	
	if (0==(m_nMode&AFM_READ) ||
	    m_pRequest->hFile<0 ||
		(m_pRequest->type == AFR_NONE && m_nPreReadBytes) ||
		(m_pRequest->type == AFR_PREREAD))
	{
		m_pThread->m_mtxRequest.unlock();
		return false;
	}
	
	ASSERT(m_nPreReadBytes == 0);
	if (!m_pBuffer)
	{
		m_pBuffer = new char[m_nBufferSize];
	}
	m_pRequest->type = AFR_PREREAD|AFR_SEEK;
	m_pRequest->offset = nOffset;
	m_pRequest->whence = nWhence;	
	m_pRequest->buffer = m_pBuffer;
	m_pRequest->size = m_nBufferSize;
	MAF_REQUEST_END
	return true;
}

bool MAsyncFile::Recycle()
{
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_WAIT, false)
	if (!m_nPreReadBytes)
	{
		m_pThread->m_mtxRequest.unlock();
		return true;
	}
	m_pThread->m_mtxRequest.unlock();
	MAF_REQUEST_CHECK(AFR_SEEK, false)
	m_pRequest->type = AFR_SEEK;
	m_pRequest->offset = -m_nPreReadBytes;
	m_pRequest->whence = SEEK_CUR;
	MAF_REQUEST_END
	return true;
}

char* MAsyncFile::GetBuffer()
{
	if (0==(m_nMode&AFM_READ))
		return NULL;
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_GETBUF, NULL)
	if (0==m_nPreReadBytes)
	{
		m_pThread->m_mtxRequest.unlock();
		return NULL;
	}
	m_mtxBuffer.lock();
	ASSERT(m_pData);
	ASSERT(m_pRequest->type==AFR_NONE);
	m_pThread->m_mtxRequest.unlock();
	return m_pData;
}

int MAsyncFile::Read(char* pBuffer, int nBytes, bool bOnlyCached = true, bool bBlock /*=false*/) // return number of bytes read or -1
{
	if (m_mtxBuffer.locked())
	{
		TRACE("MAsyncFile::Read: buffer is locked");
		return -1;
	}

	if (nBytes<=0)
		return 0;
		
	if (0==(m_nMode&AFM_READ))
		return -1;
	
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_READ, -1)
	if (m_pRequest->hFile<0 || m_pRequest->nErrNo!=0)
	{
		m_pThread->m_mtxRequest.unlock();
		return -1;
	}
	int n = 0;
	if (m_nPreReadBytes)
	{
		n = min(nBytes, m_nPreReadBytes);
		memcpy(pBuffer, m_pData, n);
		m_nPreReadBytes -= n;
		if (m_nPreReadBytes)
			m_pData += n;
		else
			m_pData = NULL;
		if (bOnlyCached)
		{
			//if (0 == m_nPreReadBytes)
			//	Prefetch();
			m_pThread->m_mtxRequest.unlock();
			return n;
		}
	}
	if (bOnlyCached)
	{
		m_pThread->m_mtxRequest.unlock();
	}
	else
	{
		m_pRequest->type = AFR_READ;
		m_pRequest->buffer = pBuffer + n;
		m_pRequest->size = nBytes - n;
		MAF_REQUEST_END
		if (bBlock)
		{
			WaitTillReady();
			if (m_pRequest->nReturn >= 0)
				return m_pRequest->nReturn + n;
			else
				if (n > 0)
					return n;
				else
					return -1;
		}
		return n;
	}
	//ASSERT(0==m_nPreReadBytes);
	//Prefetch();
	return 0;
}

int MAsyncFile::ReadSeek(int nOffset, int nWhence, char* pBuffer, int nBytes, bool bBlock /*=false*/) // return number of bytes read or -1
{
	if (m_mtxBuffer.locked())
	{
		TRACE("MAsyncFile::Read: buffer is locked");
		return -1;
	}
	
	if (nBytes<=0)
		return 0;
		
	if (0==(m_nMode&AFM_READ))
		return -1;
	
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_READ, -1)
	if (m_pRequest->hFile<0 || m_pRequest->nErrNo!=0)
	{
		m_pThread->m_mtxRequest.unlock();
		return -1;
	}
	
	m_pRequest->type   = (AFRType) AFR_READ|AFR_SEEK;
	m_pRequest->offset = nOffset;
	m_pRequest->whence = nWhence;	
	m_pRequest->buffer = pBuffer;
	m_pRequest->size   = nBytes;
	MAF_REQUEST_END
	
	if (bBlock)
	{
		WaitTillReady();
		if (m_pRequest->nReturn >= 0)
			return m_pRequest->nReturn;
		else
			return -1;
	}
	return 0;
}

bool MAsyncFile::ReleaseBuffer(int nBytesUsed /*=-1*/)
{
	if (!m_mtxBuffer.locked())
	{
		TRACE("MAsyncFile::ReleaseBuffer: it is not locked!");
		return false;
	}
	
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_WAIT, false)
	if (nBytesUsed > 0)
	{
		ASSERT(m_pRequest->type==AFR_NONE);
		m_nPreReadBytes -= min(m_nPreReadBytes, nBytesUsed);
		if (m_nPreReadBytes)
			m_pData += nBytesUsed;
		else
			m_pData = NULL;
	}
	else
		m_nPreReadBytes = 0;
	m_pThread->m_mtxRequest.unlock();
	m_mtxBuffer.unlock();
	return true;
	//
	//if (0 == m_nPreReadBytes)
	//	Prefetch();
}

int MAsyncFile::Write(char* pBuffer, int nBytes, bool bCopyData /*=true*/, bool bAllowBufferResize /*=true*/, bool bBlock /*=false*/)
{
	if (0==(m_nMode&AFM_WRITE))
		return -1;
	if (nBytes<=0)
		return 0;
	
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_WRITE, -1)
	if (m_pRequest->hFile<=0 || m_pRequest->nErrNo != 0)
		return -1;
	int nBytesSaved;
	if (bCopyData)
	{
		if (bAllowBufferResize && m_nBufferSize < nBytes)
		{
			delete [] m_pBuffer;
			m_pBuffer = NULL;
			m_nBufferSize = nBytes;
		}
		if (!m_pBuffer)
		{
			m_pBuffer = new char[m_nBufferSize];
		}
		// copy
		nBytesSaved =  min(nBytes, m_nBufferSize);
		memcpy(m_pBuffer, pBuffer, nBytes);
		m_pRequest->buffer = m_pBuffer;
		m_pRequest->size = nBytesSaved;
	}
	else
	{
		m_pRequest->buffer = pBuffer;
		m_pRequest->size = nBytes;
		nBytesSaved = nBytes;
	}
	m_pRequest->type = AFR_WRITE;
	MAF_REQUEST_END
	if (bBlock)
	{
		WaitTillReady();
		nBytesSaved = m_pRequest->nReturn;
	}
	return nBytesSaved;
}

int MAsyncFile::WriteSeek(int nOffset, int nWhence, char* pBuffer, int nBytes, bool bCopyData /*=true*/, bool bAllowBufferResize /*=true*/, bool bBlock /*=false*/)
{
	if (0==(m_nMode&AFM_WRITE))
		return -1;
	if (nBytes<=0)
		return 0;
	
	ASSERT(m_pThread==s_context.pAFThreadRead || m_pThread==s_context.pAFThreadWrite);
	MAF_REQUEST_CHECK(AFR_WRITE, -1)
	if (m_pRequest->hFile<=0 || m_pRequest->nErrNo != 0)
		return -1;
	int nBytesSaved;
	if (bCopyData)
	{
		if (bAllowBufferResize && m_nBufferSize < nBytes)
		{
			delete [] m_pBuffer;
			m_pBuffer = NULL;
			m_nBufferSize = nBytes;
		}
		if (!m_pBuffer)
		{
			m_pBuffer = new char[m_nBufferSize];
		}
		// copy
		nBytesSaved =  min(nBytes, m_nBufferSize);
		memcpy(m_pBuffer, pBuffer, nBytes);
		m_pRequest->buffer = m_pBuffer;
		m_pRequest->size = nBytesSaved;
	}
	else
	{
		m_pRequest->buffer = pBuffer;
		m_pRequest->size = nBytes;
		nBytesSaved = nBytes;
	}
	m_pRequest->type   = (AFRType) AFR_WRITE|AFR_SEEK;
	m_pRequest->offset = nOffset;
	m_pRequest->whence = nWhence;
	MAF_REQUEST_END
	if (bBlock)
	{
		WaitTillReady();
		nBytesSaved = m_pRequest->nReturn;
	}
	return nBytesSaved;
}



