/***************************************************************************
                          property.h  -  description
                             -------------------
    begin                : Sat Jul 7 2001
    copyright            : (C) 2001 by 
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifndef PROPERTY_H
#define PROPERTY_H

#include "string.h"

enum propert_type{
	PT_NONE    = 0,
	PT_BOOL    = 1,
	PT_INT     = 2,
	PT_DWORD   = 3,
	PT_DOUBLE  = 4,
	PT_STRING  = 5,
	PT_IP      = 6,
	PT_BYREF   = 0x10000
};

#define PNAME_SIZE 31
#define DEF_STR_SIZE 256

class MIniFile;

struct Property {
	char name[PNAME_SIZE+1];
	int  type;
	int  size;
	union {
	    // direct value
		bool   bVal; // size == 1
		int    nVal; // size == 4
		DWORD  dwVal; // size == 4
		double dVal; // size == 8
		char * lpstrVal; // size == length of current buffer
		IP     ipVal;
		// reference (pointer) to the value
		bool   * pbVal; // size == 4
		int    * pnVal; // size == 4
		DWORD  * pdwVal; // size == 4
		double * pdVal; // size == 4
		IP     * pipVal;
	};		
};

class MProperty : protected Property {
public:
	MProperty(LPCSTR szName){
		strncpy(name,szName,PNAME_SIZE);
		name[PNAME_SIZE] = '\0';
		type = PT_NONE;
		size = 0;
		m_bPersistent = true;
	}
	int GetPropType(){return type & 0xFFFF;}
	LPCSTR GetPropertyName(){return name;}
	bool   GetBoolValue(){ASSERT(type & PT_BOOL); return (type & PT_BYREF) ? *pbVal : bVal;}
	void   SetBoolValue(bool b){ASSERT(type & PT_BOOL); if (type & PT_BYREF) *pbVal = b; else bVal = b; }
	int    GetIntValue(){ASSERT(type & PT_INT); return (type & PT_BYREF) ? *pnVal : nVal;}
	void   SetIntValue(int n){ASSERT(type & PT_INT); if (type & PT_BYREF) *pnVal = n; else nVal = n; }
	DWORD  GetDwordValue(){ASSERT(type & PT_DWORD); return (type & PT_BYREF) ? *pdwVal : dwVal;}
	void   SetDwordValue(DWORD dw){ASSERT(type & PT_DWORD); if (type & PT_BYREF) *pdwVal = dw; else dwVal = dw; }
	double GetDoubleValue(){ASSERT(type & PT_DOUBLE); return (type & PT_BYREF) ? *pdVal : dVal;}
	void   SetDoubleValue(double d){ASSERT(type & PT_DOUBLE); if (type & PT_BYREF) *pdVal = d; else dVal = d; }
	IP     GetIpValue(){ASSERT(type & PT_IP); return (type & PT_BYREF) ? *pipVal : ipVal;}
	void   SetIpValue(const IP& ip){ASSERT(type & PT_DOUBLE); if (type & PT_BYREF) *pipVal = ip; else ipVal = ip; }
	LPCSTR GetStringValue(){ASSERT(type & PT_STRING); return lpstrVal;}
	bool   SetStringValue(LPCSTR str){
		ASSERT(type & PT_STRING);
		// to make it thread safe we never re-allocate the mem
		strncpy(lpstrVal, str, size-1);// the string can be turnicated
		lpstrVal[size-1] = '\0';
		return strlen(str)<size;		
	}
	
	// save/load to ini file
	bool Read(MIniFile* pIniFile);
	bool Write(MIniFile* pIniFile);
	// useful parser
	bool SetFromStr(char * buf);
	// mode
	bool IsPersistent(){return m_bPersistent;}
	void SetPersistance(bool bSet){m_bPersistent = bSet;}
	
	virtual ~MProperty(){};
protected:
	// overloaded access functions -- required by templates;
	inline void setval(bool   b){bVal = b;}
	inline void setval(int    n){nVal = n;}
	inline void setval(DWORD dw){dwVal = dw;}
	inline void setval(double d){dVal = d;}
	inline void setval(const IP& ip){ipVal = ip;}
	inline void setptr(bool   * pb){pbVal = pb;}
	inline void setptr(int    * pn){pnVal = pn;}
	inline void setptr(DWORD  * pdw){pdwVal = pdw;}
	inline void setptr(double * pd){pdVal = pd;}
	inline void setptr(IP     * pip){pipVal = pip;}
	// data members
	bool m_bPersistent;
	//bool m_bModified;
private:
	MProperty(); // no implementation
	MProperty(const MProperty&); // no implementation
	const MProperty& operator=(const MProperty&); // no implementation
};

template<class T, int PT>
class TProperty : public MProperty {
public: 
	TProperty(LPCSTR szName, T def = 0) : MProperty(szName) {
		type = PT;
		size = sizeof(T);
		setval(def);
	}
	TProperty(LPCSTR szName, T * pdef) : MProperty(szName) {
		ASSERT(pdef);
		type = PT | PT_BYREF;
		size = sizeof(T);
		setptr(pdef);
	}
	TProperty(LPCSTR szName, T * pdef, const T& init) : MProperty(szName) {
		ASSERT(pdef);
		type = PT | PT_BYREF;
		size = sizeof(T);
		setptr(pdef);
		*pdef = init;
	}
};

typedef TProperty<bool, PT_BOOL> MPropBool;
typedef TProperty<int, PT_INT> MPropInt;
typedef TProperty<DWORD, PT_DWORD> MPropDword;
typedef TProperty<double, PT_DOUBLE> MPropDbl;
typedef TProperty<IP, PT_IP> MPropIP;

class MPropString : public MProperty {
public:
	MPropString(LPCSTR szName, int nStrSize = DEF_STR_SIZE) : MProperty(szName) {
		ASSERT(nStrSize>0);
		type = PT_STRING;
		size = nStrSize;
		lpstrVal = new char[nStrSize];
		lpstrVal[0] = '\0';
	}
	MPropString(LPCSTR szName, LPSTR szStr, int nStrSize) : MProperty(szName) {
		ASSERT(szStr);
		ASSERT(nStrSize>0);
		type = PT_STRING | PT_BYREF;
		size = nStrSize;
		lpstrVal = szStr;
	}
	MPropString(LPCSTR szName, LPSTR szStr, int nStrSize, LPCSTR szInit) : MProperty(szName) {
		ASSERT(szStr);
		ASSERT(nStrSize>0);
		type = PT_STRING | PT_BYREF;
		size = nStrSize;
		lpstrVal = szStr;
		strncpy(lpstrVal, szInit, nStrSize-1);
	}
	~MPropString(){if ( (type&PT_BYREF) == 0) delete [] lpstrVal;}
};

class MPropertySection{
	friend class MPropertyContainer;
public:
	typedef map<CString, MProperty*>::iterator prop_iterator;
	//
	static inline MProperty* GetProperty(const prop_iterator& it){return it->second;}
	static inline LPCSTR     GetPropertyName(const prop_iterator& it){return it->first.c_str();}
	inline prop_iterator begin(){return m_props.begin();}
	inline prop_iterator end(){return m_props.end();}
protected:
	map<CString, MProperty*> m_props;
	CString m_sName;
};

class MPropertyContainer {
public:
    typedef map<CString, MProperty*>::iterator prop_iterator;
    typedef map<CString, MPropertySection*>::iterator sec_iterator;
    typedef prop_iterator iterator;

	MPropertyContainer();
	~MPropertyContainer();
	
	int Transfer(MPropertyContainer* pPC);
	inline int Count(){return m_props.size();}
	
	MProperty* AddNewProperty(LPCSTR szName, int nType, int nStrSize = DEF_STR_SIZE);
	MProperty* AddProperty(LPCSTR szName, bool * pb);
	MProperty* AddProperty(LPCSTR szName, bool * pb, bool bInit);
	MProperty* AddProperty(LPCSTR szName, int * pn);
	MProperty* AddProperty(LPCSTR szName, int * pn, int nInit);
	MProperty* AddProperty(LPCSTR szName, DWORD * pdw);
	MProperty* AddProperty(LPCSTR szName, DWORD * pdw, DWORD dwInit);
	MProperty* AddProperty(LPCSTR szName, double * pd);
	MProperty* AddProperty(LPCSTR szName, double * pd, double dInit);
	MProperty* AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize);
	MProperty* AddProperty(LPCSTR szName, LPSTR pstr, int nStrSize, LPCSTR szInit);
	MProperty* AddProperty(LPCSTR szName, IP * pip);
	MProperty* AddProperty(LPCSTR szName, IP * pip, const IP& ipInit);
	
	MProperty* FindProperty(LPCSTR szName);
	
	bool AddSection(LPCSTR szName); // also sets section as current
	bool SetCurrentSection(LPCSTR szName); // AddProperty "context"
	MPropertySection* FindSection(LPCSTR szName);
	
	// properties
	static inline MProperty* GetProperty(const iterator& it){return it->second;}
	static inline LPCSTR     GetPropertyName(const iterator& it){return it->first.c_str();}
	inline iterator begin(){return m_props.begin();}
	inline iterator end(){return m_props.end();}
	// sections
	bool HasSections(){return m_secs.size();}
	static inline MPropertySection* GetSection(const sec_iterator& it){return it->second;}
	static inline LPCSTR            GetSectionName(const sec_iterator& it){return it->first.c_str();}
	inline sec_iterator sec_begin(){return m_secs.begin();}
	inline sec_iterator sec_end(){return m_secs.end();}
	// save/load
	bool Read(LPCSTR szName);
	bool Write(LPCSTR szName);
protected:
	map<CString, MProperty*> m_props;
	map<CString, MPropertySection*> m_secs;
	MPropertySection* m_pCurSec;
};

#endif
