/*
 * pfmon_itanium.c - Itanium PMU support for pfmon
 *
 * Copyright (C) 2001-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <ctype.h>
#include <perfmon/pfmlib_itanium.h>

#include "pfmon_itanium.h"

#define PFMON_SUPPORT_NAME	"itanium"

#define DEAR_REGS_MASK		(M_PMD(2)|M_PMD(3)|M_PMD(17))
#define IEAR_REGS_MASK		(M_PMD(0)|M_PMD(1))
#define BTB_REGS_MASK		(M_PMD(8)|M_PMD(9)|M_PMD(10)|M_PMD(11)|M_PMD(12)|M_PMD(13)|M_PMD(14)|M_PMD(15)|M_PMD(16))


#define PFMON_ITA_MAX_IBRS	8
#define PFMON_ITA_MAX_DBRS	8


static pfmon_ita_options_t pfmon_ita_opt;	/* keep track of global program options */
static pfmlib_ita_input_param_t ita_input_params;
static pfmlib_ita_output_param_t ita_output_params;

#define ITA_INPUT_PARAM(x)	(pfmlib_ita_input_param_t *)(x)->mod_inp
#define ITA_OUTPUT_PARAM(x)	(pfmlib_ita_output_param_t *)(x)->mod_outp

static void
gen_thresholds(char *arg, pfmon_lib_param_t *evt)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	char *p;
	unsigned int thres, maxincr;
	unsigned int cnt=0;
	unsigned int i;

	/*
	 * the default value for the threshold is 0: this means at least once 
	 * per cycle.
	 */
	if (arg == NULL) {
		for (i=0; i < evt->inp.pfp_event_count; i++) param->pfp_ita_counters[i].thres = 0;
		return;
	}

	while (arg) {

		if (cnt == options.max_counters || cnt == evt->inp.pfp_event_count) goto too_many;

		p = strchr(arg,',');

		if ( p ) *p++ = '\0';

		thres = atoi(arg);

		/*
		 *  threshold = multi-occurence -1
		 * this is because by setting threshold to n, one counts only
		 * when n+1 or more events occurs per cycle.
	 	 */
		pfm_ita_get_event_maxincr(evt->inp.pfp_events[cnt].event, &maxincr);
		if (thres > (maxincr-1)) goto too_big;

		param->pfp_ita_counters[cnt++].thres = thres;

		arg = p;
	}
	return;
too_big:
	fatal_error("event %d: threshold must be in [0-%d)\n", cnt, maxincr);
too_many:
	fatal_error("too many thresholds specified\n");
}

static int
install_irange(pfmon_ctxid_t id, pfmon_lib_param_t *evt)
{
	pfmlib_ita_output_param_t *param = ITA_OUTPUT_PARAM(evt);
	pfarg_dbreg_t dbreg[PFMON_ITA_MAX_IBRS];
	unsigned int i, used_dbr;
	int  r;

	used_dbr = param->pfp_ita_irange.rr_nbr_used;

	for(i=0; i < used_dbr; i++) {
		dbreg[i].dbreg_num   = param->pfp_ita_irange.rr_br[i].reg_num; 
		dbreg[i].dbreg_set   = 0;
		dbreg[i].dbreg_value = param->pfp_ita_irange.rr_br[i].reg_value; 
		dbreg[i].dbreg_flags = 0UL;
	}

	r = perfmonctl(id, PFM_WRITE_IBRS, dbreg, used_dbr);
	if (r == -1) warning("cannot install code range restriction: %s\n", strerror(errno));
	return r;
}

static int
install_drange(pfmon_ctxid_t id, pfmon_lib_param_t *evt)
{
	pfmlib_ita_output_param_t *param = ITA_OUTPUT_PARAM(evt);
	pfarg_dbreg_t dbreg[PFMON_ITA_MAX_DBRS];
	unsigned int i, used_dbr;
	int r;

	used_dbr = param->pfp_ita_drange.rr_nbr_used;

	for(i=0; i < used_dbr; i++) {
		dbreg[i].dbreg_num   = param->pfp_ita_drange.rr_br[i].reg_num; 
		dbreg[i].dbreg_set   = 0;
		dbreg[i].dbreg_value = param->pfp_ita_drange.rr_br[i].reg_value; 
		dbreg[i].dbreg_flags = 0UL;
	}

	r = perfmonctl(id, PFM_WRITE_DBRS, dbreg, used_dbr);
	if (r == -1) warning("cannot install data range restriction: %s\n", strerror(errno));
	return r;
}
static int
install_btb(pfmon_ctxid_t id, pfmon_lib_param_t *evt)
{
	/* 
	 * we do not really need to clear PMD16, because the kernel 
	 * clear pmd16 for each newly created context
	 */
	return 0;
}

static int
prepare_btb(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	unsigned int i;
	int found_btb = -1;

	for(i=0; i < evt->inp.pfp_event_count; i++) {
		if (pfm_ita_is_btb(evt->inp.pfp_events[i].event)) {
			found_btb = i;
			goto found;
		}
	}
	/*
	 * check for no BTB event, but just BTB options.
	 */
	if (param->pfp_ita_btb.btb_used == 0) return 0;
found:
	/*
	 * in case of no BTB event found, we are in free running mode (no BTB sampling)
	 * therefore we include the BTB PMD in all samples
	 */
	if (found_btb != -1 && (options.short_rates[i].flags & PFMON_RATE_VAL_SET)) {
		options.smpl_pmds[found_btb]  |=  BTB_REGS_MASK;
	} else {
		options.common_smpl_pmds      |=  BTB_REGS_MASK;
		options.common_reset_pmds     |=  BTB_REGS_MASK;
	}
	return 0;
}

static int
prepare_ears(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	unsigned int i;
	int ev, is_iear;

	/*
	 * search for an EAR event
	 */
	for(i=0; i < evt->inp.pfp_event_count; i++) {

		ev = evt->inp.pfp_events[i].event;

		/* look for EAR event */
		if (pfm_ita_is_ear(ev) == 0) continue;

		is_iear = pfm_ita_is_iear(ev);

		/*
		 * when used as sampling period, then just setup the bitmask
		 * of PMDs to record in each sample
		 */
		if (options.short_rates[i].flags & PFMON_RATE_VAL_SET) {
			options.smpl_pmds[i]  |=  is_iear ? IEAR_REGS_MASK : DEAR_REGS_MASK;
			continue;
		}

		/*
		 * for D-EAR, we must clear PMD3.stat and PMD17.vl to make
		 * sure we do not interpret the register in the wrong manner.
		 *
		 * for I-EAR, we must clear PMD0.stat to avoid interpreting stale
		 * entries
		 *
		 * This is ONLY necessary when the events are not used as sampling
		 * periods.
		 */
		options.common_reset_pmds |= is_iear ? M_PMD(0) : M_PMD(17) | M_PMD(3);
	}
	return 0;
}

/*
 * executed once for the entire session (does not make any perfmonctl() calls)
 */
static int
pfmon_ita_prepare_registers(pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	int r = 0;

	if (param->pfp_ita_btb.btb_used) r = prepare_btb(evt, pc, pd);

	if (r == 0) r = prepare_ears(evt, pc, pd);

	return r;
}

static int
pfmon_ita_install_registers(pfmon_ctxid_t id, pfmon_lib_param_t *evt, pfarg_reg_t *pc, pfarg_reg_t *pd)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	int r = 0;

	if (param->pfp_ita_irange.rr_used) r = install_irange(id, evt);

	if (r == 0 && param->pfp_ita_drange.rr_used) r = install_drange(id, evt);

	if (r == 0 && param->pfp_ita_btb.btb_used) r = install_btb(id, evt);

	return r;
}

static void
pfmon_ita_usage(void)
{
	printf(
		"--event-thresholds=thr1,thr2,...\tset event thresholds (no space)\n"
		"--opc-match8=val\t\t\tset opcode match for PMC8\n"
		"--opc-match9=val\t\t\tset opcode match for PMC9\n"
		"--btb-no-tar\t\t\t\tdon't capture TAR predictions\n"
		"--btb-no-bac\t\t\t\tdon't capture BAC predictions\n"
		"--btb-no-tac\t\t\t\tdon't capture TAC predictions\n"
		"--btb-tm-tk\t\t\t\tcapture taken IA-64 branches only\n"
		"--btb-tm-ntk\t\t\t\tcapture not taken IA-64 branches only\n"
		"--btb-ptm-correct\t\t\tcapture branch if target predicted correctly\n"
		"--btb-ptm-incorrect\t\t\tcapture branch if target is mispredicted\n"
		"--btb-ppm-correct\t\t\tcapture branch if path is predicted correctly\n"
		"--btb-ppm-incorrect\t\t\tcapture branch if path is mispredicted\n"
		"--btb-all-mispredicted\t\t\tcapture all mispredicted branches\n"
		"--irange=start-end\t\t\tspecify an instruction address range constraint\n"
		"--drange=start-end\t\t\tspecify a data address range constraint\n"
		"--checkpoint-func=addr\t\t\ta bundle address to use as checkpoint\n"
		"--ia32\t\t\t\t\tmonitor IA-32 execution only\n"
		"--ia64\t\t\t\t\tmonitor IA-64 execution only\n"
		"--insn-sets=set1,set2,...\t\tset per event instruction set (setX=[ia32|ia64|both])\n"
		"--no-qual-check\t\t\t\tdo not check qualifier constraints on events\n"
	);
}

static void
setup_ears(pfmon_lib_param_t *evt)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	pfmlib_ita_ear_mode_t mode;
	int done_iear = 0, done_dear = 0;
	unsigned int i, ev;

	for (i=0; i < evt->inp.pfp_event_count; i++) {

		ev = evt->inp.pfp_events[i].event;

		if (pfm_ita_is_ear(ev) == 0) continue;

		pfm_ita_get_ear_mode(ev, &mode);

		if (pfm_ita_is_dear(ev)) {

			if (done_dear) {
				fatal_error("cannot specify more than one D-EAR event at the same time\n");
			}

			pfmon_ita_opt.opt_use_dear_tlb = mode == PFMLIB_ITA_EAR_TLB_MODE ? 1 : 0;

			param->pfp_ita_dear.ear_used = 1;
			param->pfp_ita_dear.ear_mode = mode;
			param->pfp_ita_dear.ear_plm  = evt->inp.pfp_events[i].plm; /* use plm from event */
			param->pfp_ita_dear.ear_ism  = param->pfp_ita_counters[i].ism;
			pfm_ita_get_event_umask(ev, &param->pfp_ita_dear.ear_umask);

			done_dear = 1;
		}

		if (pfm_ita_is_iear(ev)) {

			if (done_iear) {
				fatal_error("cannot specify more than one D-EAR event at the same time\n");
			}

			pfmon_ita_opt.opt_use_iear_tlb = mode == PFMLIB_ITA_EAR_TLB_MODE ? 1 : 0;

			param->pfp_ita_iear.ear_used = 1;
			param->pfp_ita_iear.ear_mode = mode;
			param->pfp_ita_iear.ear_plm  = evt->inp.pfp_events[i].plm; /* use plm from event */
			param->pfp_ita_iear.ear_ism  = param->pfp_ita_counters[i].ism;

			pfm_ita_get_event_umask(ev, &param->pfp_ita_iear.ear_umask);

			done_iear = 1;
		}
	}
}

static void
setup_btb(pfmon_lib_param_t *evt)
{
	unsigned int i;
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);

	/*
	 * For pfmon, we do not activate the BTB registers unless a BRANCH_EVENT
	 * is specified in the event list. The libpfm library does not have this restriction.
	 *
	 * XXX: must make sure BRANCH_EVENT shows up only once
	 */
	for (i=0; i < evt->inp.pfp_event_count; i++) {
		if (pfm_ita_is_btb(evt->inp.pfp_events[i].event)) {
			goto found;
		}
	}
	/*
	 * if the user specified a BTB option (but not the event) 
	 * then we program the BTB as a free running config.
	 *
	 * XXX: cannot record ALL branches
	 */
	if (  !pfmon_ita_opt.opt_btb_notar
	   && !pfmon_ita_opt.opt_btb_notac
	   && !pfmon_ita_opt.opt_btb_nobac
	   && !pfmon_ita_opt.opt_btb_tm
	   && !pfmon_ita_opt.opt_btb_ptm
	   && !pfmon_ita_opt.opt_btb_ppm) return;
found:

	/*
	 * set the use bit, such that the library will program PMC12
	 */
	param->pfp_ita_btb.btb_used = 1;

	/* by default, the registers are setup to 
	 * record every possible branch.
	 * The record nothing is not available because it simply means
	 * don't use a BTB event.
	 * So the only thing the user can do is narrow down the type of
	 * branches to record. This simplifies the number of cases quite
	 * substantially.
	 */
	param->pfp_ita_btb.btb_tar = 1;
	param->pfp_ita_btb.btb_tac = 1;
	param->pfp_ita_btb.btb_bac = 1;
	param->pfp_ita_btb.btb_tm  = 0x3;
	param->pfp_ita_btb.btb_ptm = 0x3;
	param->pfp_ita_btb.btb_ppm = 0x3;
	param->pfp_ita_btb.btb_plm = evt->inp.pfp_events[i].plm; /* use the plm from the BTB event */

	if (pfmon_ita_opt.opt_btb_notar) param->pfp_ita_btb.btb_tar = 0;
	if (pfmon_ita_opt.opt_btb_notac) param->pfp_ita_btb.btb_tac = 0;
	if (pfmon_ita_opt.opt_btb_nobac) param->pfp_ita_btb.btb_bac = 0;
	if (pfmon_ita_opt.opt_btb_tm)    param->pfp_ita_btb.btb_tm  = (unsigned char)pfmon_ita_opt.opt_btb_tm  & 0x3;
	if (pfmon_ita_opt.opt_btb_ptm)   param->pfp_ita_btb.btb_ptm = (unsigned char)pfmon_ita_opt.opt_btb_ptm & 0x3;
	if (pfmon_ita_opt.opt_btb_ppm)   param->pfp_ita_btb.btb_ppm = (unsigned char)pfmon_ita_opt.opt_btb_ppm & 0x3;

	vbprintf("btb options:\n\tplm=%d tar=%c tac=%c bac=%c tm=%d ptm=%d ppm=%d\n",
		param->pfp_ita_btb.btb_plm,
		param->pfp_ita_btb.btb_tar ? 'Y' : 'N',
		param->pfp_ita_btb.btb_tac ? 'Y' : 'N',
		param->pfp_ita_btb.btb_bac ? 'Y' : 'N',
		param->pfp_ita_btb.btb_tm,
		param->pfp_ita_btb.btb_ptm,
		param->pfp_ita_btb.btb_ppm);
}

/*
 * Itanium-specific options
 *
 * 000-255   reserved for generic options
 * 400-499   reserved for PMU specific options
 * 500-599   reserved for format specific options
 */
static struct option cmd_ita_options[]={
	{ "event-thresholds", 1, 0, 400 },
	{ "opc-match8", 1, 0, 401},
	{ "opc-match9", 1, 0, 402},
	{ "btb-all-mispredicted", 0, 0, 403},
	{ "checkpoint-func", 1, 0, 404},
	{ "irange", 1, 0, 405},
	{ "drange", 1, 0, 406},
	{ "insn-sets", 1, 0, 407},

	{ "btb-no-tar", 0, &pfmon_ita_opt.opt_btb_notar, 1},
	{ "btb-no-bac", 0, &pfmon_ita_opt.opt_btb_nobac, 1},
	{ "btb-no-tac", 0, &pfmon_ita_opt.opt_btb_notac, 1},
	{ "btb-tm-tk", 0, &pfmon_ita_opt.opt_btb_tm, 0x2},
	{ "btb-tm-ntk", 0, &pfmon_ita_opt.opt_btb_tm, 0x1},
	{ "btb-ptm-correct", 0, &pfmon_ita_opt.opt_btb_ptm, 0x2},
	{ "btb-ptm-incorrect", 0, &pfmon_ita_opt.opt_btb_ptm, 0x1},
	{ "btb-ppm-correct", 0, &pfmon_ita_opt.opt_btb_ppm, 0x2},
	{ "btb-ppm-incorrect", 0, &pfmon_ita_opt.opt_btb_ppm, 0x1},
	{ "ia32", 0, &pfmon_ita_opt.opt_ia32, 0x1},
	{ "ia64", 0, &pfmon_ita_opt.opt_ia64, 0x1},
	{ "no-qual-check", 0, &pfmon_ita_opt.opt_no_qual_check, 0x1},
	{ 0, 0, 0, 0}
};

static int
pfmon_ita_initialize(pfmon_lib_param_t *evt)
{
	int r;

	r = pfmon_register_pmu_options(cmd_ita_options, sizeof(cmd_ita_options));
	if (r == -1) return -1;

	/* connect model specific library parameters */
	evt->mod_inp  = &ita_input_params;
	evt->mod_outp = &ita_output_params;

	/* connect pfmon model specific options */
	options.model_options = &pfmon_ita_opt;

	return 0;
}

static int
pfmon_ita_parse_options(int code, char *optarg, pfmon_lib_param_t *evt)
{
	switch(code) {
		case  400:
			if (pfmon_ita_opt.thres_arg) fatal_error("thresholds already defined\n");
			pfmon_ita_opt.thres_arg = optarg;
			break;
		case  401:
			if (pfmon_ita_opt.opcm8_str) fatal_error("opcode matcher pmc8 is specified twice\n");
			pfmon_ita_opt.opcm8_str = optarg;
			break;
		case  402:
			if (pfmon_ita_opt.opcm9_str) fatal_error("opcode matcher pmc9 is specified twice\n");
			pfmon_ita_opt.opcm9_str = optarg;
			break;
		case  403:
			/* shortcut to the following options
			 * must not be used with other btb options
			 */
			pfmon_ita_opt.opt_btb_notar = 0;
			pfmon_ita_opt.opt_btb_nobac = 0;
			pfmon_ita_opt.opt_btb_notac = 0;
			pfmon_ita_opt.opt_btb_tm    = 0x3;
			pfmon_ita_opt.opt_btb_ptm   = 0x1;
			pfmon_ita_opt.opt_btb_ppm   = 0x1;
			break;
		case  404:
			if (pfmon_ita_opt.irange_str) {
				fatal_error("cannot use checkpoints and instruction range at the same time\n");
			}
			if (pfmon_ita_opt.chkp_func_str) {
				fatal_error("checkpoint already  defined for %s\n", pfmon_ita_opt.chkp_func_str);
			}
			pfmon_ita_opt.chkp_func_str = optarg;
			break;

		case  405:
			if (pfmon_ita_opt.chkp_func_str) {
				fatal_error("cannot use instruction range and checkpoints at the same time\n");
			}
			if (pfmon_ita_opt.irange_str) {
				fatal_error("cannot specify more than one instruction range\n");
			}
			pfmon_ita_opt.irange_str = optarg;
			break;

		case  406:
			if (pfmon_ita_opt.drange_str) {
				fatal_error("cannot specify more than one data range\n");
			}
			pfmon_ita_opt.drange_str = optarg;
			break;
		case  407:
			if (pfmon_ita_opt.insn_str) fatal_error("per event instruction sets already defined");
			pfmon_ita_opt.insn_str = optarg;
			break;
		default:
			return -1;
	}
	return 0;
}

static void
setup_opcm(pfmon_lib_param_t *evt)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	char *endptr = NULL;

	if (pfmon_ita_opt.opcm8_str) {
		if (isdigit(pfmon_ita_opt.opcm8_str[0])) {
			param->pfp_ita_pmc8.pmc_val = strtoul(pfmon_ita_opt.opcm8_str, &endptr, 0);
			if (*endptr != '\0') 
				fatal_error("invalid value for opcode match pmc8: %s\n", pfmon_ita_opt.opcm8_str);
		} else if (find_opcode_matcher(pfmon_ita_opt.opcm8_str, &param->pfp_ita_pmc8.pmc_val) == 0) 
				fatal_error("invalid opcode matcher value: %s\n", pfmon_ita_opt.opcm8_str);

		param->pfp_ita_pmc8.opcm_used = 1;
	}

	if (pfmon_ita_opt.opcm9_str) {
		if (isdigit(pfmon_ita_opt.opcm9_str[0])) {
			param->pfp_ita_pmc9.pmc_val = strtoul(pfmon_ita_opt.opcm9_str, &endptr, 0);
			if (*endptr != '\0') 
				fatal_error("invalid value for opcode match pmc9: %s\n", pfmon_ita_opt.opcm8_str);
		} else if (find_opcode_matcher(pfmon_ita_opt.opcm9_str, &param->pfp_ita_pmc9.pmc_val) == 0) 
				fatal_error("invalid opcode matcher value: %s\n", pfmon_ita_opt.opcm9_str);

		param->pfp_ita_pmc9.opcm_used = 1;
	}
}


static void
setup_rr(pfmon_lib_param_t *evt)
{
	uintptr_t start, end;
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);

	if (pfmon_ita_opt.chkp_func_str) {
		if (options.priv_lvl_str)
			fatal_error("cannot use both a checkpoint function and per-event privilege level masks\n");

		gen_code_range(pfmon_ita_opt.chkp_func_str, &start, &end);
		
		/* just one bundle for this one */
		end = start + 0x10;

		vbprintf("checkpoint function at %p\n", start);
	} else if (pfmon_ita_opt.irange_str) {

		if (options.priv_lvl_str)
			fatal_error("cannot use both a code range function and per-event privilege level masks\n");

		gen_code_range(pfmon_ita_opt.irange_str, &start, &end); 

		if ((unsigned long)start & 0xf) fatal_error("code range does not start on bundle boundary : %p\n", start);
		if ((unsigned long)end & 0xf) fatal_error("code range does not end on bundle boundary : %p\n", end);

		vbprintf("irange is [%p-%p)=%ld bytes\n", start, end, end-start);
	}

	/*
	 * now finalize irange/chkp programming of the range
	 */
	if (pfmon_ita_opt.irange_str || pfmon_ita_opt.chkp_func_str) { 

		param->pfp_ita_irange.rr_used = 1;

		param->pfp_ita_irange.rr_limits[0].rr_start = (unsigned long)start;
		param->pfp_ita_irange.rr_limits[0].rr_end   = (unsigned long)end;
		param->pfp_ita_irange.rr_limits[0].rr_plm   = evt->inp.pfp_dfl_plm; /* use default */
	}

	if (pfmon_ita_opt.drange_str) {
		if (options.priv_lvl_str)
			fatal_error("cannot use both a data range and  per-event privilege level masks\n");

		gen_data_range(pfmon_ita_opt.drange_str, &start, &end);

		vbprintf("drange is [%p-%p)=%lu bytes\n", start, end, end-start);
		
		param->pfp_ita_drange.rr_used = 1;

		param->pfp_ita_drange.rr_limits[0].rr_start = (unsigned long)start;
		param->pfp_ita_drange.rr_limits[0].rr_end   = (unsigned long)end;
		param->pfp_ita_drange.rr_limits[0].rr_plm   = evt->inp.pfp_dfl_plm; /* use default */
	}

}


/*
 * This function checks the configuration to verify
 * that the user does not try to combine features with
 * events that are incompatible.The library does this also
 * but it's hard to then detail the cause of the error.
 */
static void
check_ita_event_combinations(pfmon_lib_param_t *evt)
{
	unsigned int i, use_opcm, ev;
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	char name[PFMON_MAX_EVTNAME_LEN];

	use_opcm = param->pfp_ita_pmc8.opcm_used || param->pfp_ita_pmc9.opcm_used; 
	for (i=0; i < evt->inp.pfp_event_count; i++) {

		ev = evt->inp.pfp_events[i].event;

		pfm_get_event_name(ev, name, PFMON_MAX_EVTNAME_LEN);

		if (use_opcm && pfm_ita_support_opcm(ev) == 0)
			fatal_error("event %s does not support opcode matching\n", name);

		if (param->pfp_ita_irange.rr_used && pfm_ita_support_iarr(ev) == 0)
			fatal_error("event %s does not support instruction address range restrictions\n", name);

		if (param->pfp_ita_drange.rr_used && pfm_ita_support_darr(ev) == 0)
			fatal_error("event %s does not support data address range restrictions\n", name);

		if (pfmon_ita_opt.opt_ia32 && pfmon_ita_opt.opt_ia64 == 0 && pfm_ita_is_btb(ev))
			fatal_error("cannot use BTB event (%s) when only monitoring IA-32 execution\n", name);
	}
	/*
	 * we do not call check_counter_conflict() because Itanium does not have events
	 * which can only be measured on one counter, therefore this routine would not
	 * catch anything at all.
	 */
}

static void
setup_insn(pfmon_lib_param_t *evt)
{
	static const struct {
		char *name;
		pfmlib_ita_ism_t val;
	} insn_sets[]={
		{ "ia32", PFMLIB_ITA_ISM_IA32 },
		{ "ia64", PFMLIB_ITA_ISM_IA64 },
		{ "both", PFMLIB_ITA_ISM_BOTH },
		{ NULL  , PFMLIB_ITA_ISM_BOTH }
	};
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	char *p, *arg;
	pfmlib_ita_ism_t dfl_ism;
	unsigned int i, cnt=0;

	/* 
	 * set default instruction set 
	 */
	if (pfmon_ita_opt.opt_ia32  && pfmon_ita_opt.opt_ia64)
		dfl_ism = PFMLIB_ITA_ISM_BOTH;
	else if (pfmon_ita_opt.opt_ia64)
		dfl_ism = PFMLIB_ITA_ISM_IA64;
	else if (pfmon_ita_opt.opt_ia32)
		dfl_ism = PFMLIB_ITA_ISM_IA32;
	else
		dfl_ism = PFMLIB_ITA_ISM_BOTH;

	/*
	 * propagate default instruction set to all events
	 */
	for(i=0; i < evt->inp.pfp_event_count; i++) param->pfp_ita_counters[i].ism = dfl_ism;

	/*
	 * apply correction for per-event instruction set
	 */
	for (arg = pfmon_ita_opt.insn_str; arg; arg = p) {
		if (cnt == evt->inp.pfp_event_count) goto too_many;

		p = strchr(arg,',');
			
		if (p) *p = '\0';

		if (*arg) {
			for (i=0 ; insn_sets[i].name; i++) {
				if (!strcmp(insn_sets[i].name, arg)) goto found;
			}
			goto error;
found:
			param->pfp_ita_counters[cnt++].ism = insn_sets[i].val;
		}
		/* place the comma back so that we preserve the argument list */
		if (p) *p++ = ',';
	}
	return;
error:
	fatal_error("unknown per-event instruction set %s (choices are ia32, ia64, or both)\n", arg);
	/* no return */
too_many:
	fatal_error("too many per-event instruction sets specified, max=%d\n", evt->inp.pfp_event_count);
}

static void
pfmon_ita_no_qual_check(pfmon_lib_param_t *evt)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);
	unsigned int i;

	/*
	 * set the "do not check constraint" flags for all events 
	 */
	for (i=0; i < evt->inp.pfp_event_count; i++) {
		param->pfp_ita_counters[i].flags |= PFMLIB_ITA_FL_EVT_NO_QUALCHECK;
	}
}



static int
pfmon_ita_post_options(pfmon_lib_param_t *evt)
{
	pfmlib_ita_input_param_t *param = ITA_INPUT_PARAM(evt);

	if (options.code_trigger_start || options.code_trigger_stop || options.data_trigger_start || options.data_trigger_stop) {
		if (pfmon_ita_opt.irange_str)
			fatal_error("cannot use a trigger address with instruction range restrictions\n");
		if (pfmon_ita_opt.drange_str)
			fatal_error("cannot use a trigger address with data range restrictions\n");
		if (pfmon_ita_opt.chkp_func_str)
			fatal_error("cannot use a trigger address with function checkpoint\n");
	}

	/*
	 * setup the instruction set support
	 *
	 * and reject any invalid combination for IA-32 only monitoring
	 *
	 * We do not warn of the fact that IA-32 execution will be ignored
	 * when used with incompatible features unless the user requested IA-32
	 * ONLY monitoring. 
	 */
	if (pfmon_ita_opt.opt_ia32 == 1 && pfmon_ita_opt.opt_ia64 == 0) {

		/*
		 * Code & Data range restrictions are ignored for IA-32
		 */
		if (pfmon_ita_opt.irange_str|| pfmon_ita_opt.drange_str) 
			fatal_error("you cannot use range restrictions when monitoring IA-32 execution only\n");

		/*
		 * Code range restriction (used by checkpoint) is ignored for IA-32
		 */
		if (pfmon_ita_opt.chkp_func_str) 
			fatal_error("you cannot use function checkpoint when monitoring IA-32 execution only\n");

		/*
		 * opcode matcher are ignored for IA-32
		 */
		if (param->pfp_ita_pmc8.opcm_used || param->pfp_ita_pmc9.opcm_used) 
			fatal_error("you cannot use the opcode matcher(s) when monitoring IA-32 execution only\n");

	}
	setup_insn(evt);

	setup_rr(evt);

	setup_btb(evt);

	setup_opcm(evt);
	setup_ears(evt);

	/*
	 * BTB is only valid in IA-64 mode
	 */
	if ( pfmon_ita_opt.opt_ia32 && pfmon_ita_opt.opt_ia64 == 0&& 
	    (pfmon_ita_opt.opt_btb_notar || 
	     pfmon_ita_opt.opt_btb_notac || 
	     pfmon_ita_opt.opt_btb_nobac || 
	     pfmon_ita_opt.opt_btb_tm    || 
	     pfmon_ita_opt.opt_btb_ptm   || 
	     pfmon_ita_opt.opt_btb_ppm)) {
		fatal_error("cannot use the BTB when only monitoring IA-32 execution\n");
	}


	/* 
	 * we systematically initialize thresholds to their minimal value
	 * or requested value
	 */
	gen_thresholds(pfmon_ita_opt.thres_arg, evt);

	if (pfmon_ita_opt.opt_no_qual_check) 
		pfmon_ita_no_qual_check(evt);
	else
		check_ita_event_combinations(evt);

	return 0;
}

static int
pfmon_ita_print_header(FILE *fp)
{
	int i, isn;
	char name[PFMON_MAX_EVTNAME_LEN];
	static const char *insn_str[]={
		"ia32/ia64",
		"ia32", 
		"ia64"
	};


	fprintf(fp, "#\n#\n# instruction sets:\n");

	for(i=0; i < PFMON_MAX_PMDS; i++) {
		if (options.rev_pc[i] == -1) continue;

		pfm_get_event_name(options.events[options.rev_pc[i]].event, name, PFMON_MAX_EVTNAME_LEN);

		isn = ita_input_params.pfp_ita_counters[options.rev_pc[i]].ism;
		fprintf(fp, "#\tPMD%d: %-*s = %s\n", 
			i,
			(int)options.max_event_name_len,
			name,
			insn_str[isn]);
	} 
	fprintf(fp, "#\n");

	return 0;
}

static void
pfmon_ita_detailed_event_name(int evt)
{
	unsigned long umask;
	unsigned int maxincr;

	pfm_ita_get_event_umask(evt, &umask);
	pfm_ita_get_event_maxincr(evt, &maxincr);

	printf("umask=0x%02lx incr=%u iarr=%c darr=%c opcm=%c ", 
			umask, 
			maxincr,
			pfm_ita_support_iarr(evt) ? 'Y' : 'N',
			pfm_ita_support_darr(evt) ? 'Y' : 'N',
			pfm_ita_support_opcm(evt) ? 'Y' : 'N');
}


pfmon_support_t pfmon_itanium={
	.name				= "Itanium",
	.pmu_type			= PFMLIB_ITANIUM_PMU,
	.default_event			= "CPU_CYCLES",
	.default_plm			= PFM_PLM3,
	.pfmon_initialize		= pfmon_ita_initialize,
	.pfmon_usage			= pfmon_ita_usage,
	.pfmon_parse_options		= pfmon_ita_parse_options,
	.pfmon_post_options		= pfmon_ita_post_options,
	.pfmon_prepare_registers	= pfmon_ita_prepare_registers,
	.pfmon_install_registers	= pfmon_ita_install_registers,
	.pfmon_print_header		= pfmon_ita_print_header,
	.pfmon_detailed_event_name	= pfmon_ita_detailed_event_name
};

