/* listing.c - Running and parsing the output of ls(1). 
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif
/* As recommended by autoconf for AC_HEADER_SYS_WAIT */
#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif
/* POSIX */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif
#include <sys/stat.h>
/* POSIX/gnulib */
#include <getopt.h>
#include <stdbool.h>
/* gettext */
#include <gettext.h> /* will include <libintl.h> if ENABLE_NLS */
#define _(String) gettext(String)
/* C89 */
#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
/* common */
#include "common/common.h"
#include "common/io-utils.h"
#include "common/memory.h"
#include "common/string-utils.h"
#include "common/strbuf.h"
#include "common/error.h"
#include "common/llist.h"
#include "common/hmap.h"
/* qmv */
#include "qmv.h"

#define ls_assert(e)	\
	if (!(e)) { \
		internal_error("unexpected output from `ls' (%s:%d)\nLine:\n%s\n", \
				__FILE__, __LINE__, line); \
	}

static bool run_ls(char **args, pid_t *ls_pid, int *ls_fd);
static bool clean_ls(pid_t ls_pid, int ls_fd, int *status);
static bool process_ls_output(char *firstdir, LList *files, int ls_fd);
static void remove_unnecessary_files(LList *files);

static LList *ls_options;
static int old_dir = -1;

#define SORT_OPT		1001
#define TIME_OPT		1002

static struct option ls_option_table[] = {
	{ "all",				no_argument,		NULL,	'a'				},
	{ "almost-all",			no_argument,		NULL,	'A'				},
	{ "ignore-backups",		no_argument,		NULL,	'B'				},
	{ "directory",			no_argument,		NULL,	'd'				},
	{ "reverse",			no_argument,		NULL,	'r'				},
	{ "recursive",			no_argument,		NULL,	'R'				},
	{ "sort",				required_argument,	NULL,	SORT_OPT		},
	{ "time",				required_argument,	NULL,	TIME_OPT		},
	{ 0,					0,					0,		0				},
};

struct option *
append_ls_options(const struct option *options)
{
	struct option *new_options;
	int c;
	int d;

	ls_options = llist_new();	/* this should be static, but since
								 * append_ls_options is called early this
								 * is ok */

	for (c = 0; options[c].name != NULL; c++);
	d = sizeof(ls_option_table)/sizeof(*ls_option_table);

	new_options = xmalloc((c+d) * sizeof(struct option));
	memcpy(new_options, options, c * sizeof(struct option));
	memcpy(new_options+c, ls_option_table, d * sizeof(struct option));

	return new_options;
}

void
display_ls_help(FILE *out)
{
	fprintf(out, _("\
Listing (`ls') options:\n\
  -a, --all                  do not hide entries starting with .\n\
  -A, --almost-all           do not list implied . and ..\n\
  -B, --ignore-backups       do not list implied entries ending with ~\n\
  -c                         sort by ctime\n\
  -d, --directory            list directory entries instead of contents\n\
  -r, --reverse              reverse order while sorting\n\
  -R, --recursive            list subdirectories recursively\n\
  -S                         sort by file size\n\
      --sort=WORD            extension -X, none -U, size -S, time -t,\n\
                               version -v, status -c, time -t, atime -u,\n\
                               access -u, use -u\n\
      --time=WORD            show time as WORD instead of modification time:\n\
                               atime, access, use, ctime or status; use\n\
                               specified time as sort key if --sort=time\n\
  -t                         sort by modification time\n\
  -u                         sort by access time\n\
  -U                         do not sort; list entries in directory order\n\
  -X                         sort alphabetically by entry extension\n"));
}

void
free_renames(LList *list)
{
	llist_iterate(list, (IteratorFunc) free_file_spec);
	llist_clear(list);
}

void
list_command(char **args)
{
	int argc;
	int c;

	for (argc = 1; args[argc] != NULL; argc++);

	llist_clear(ls_options);

	optind = 0;
	while (true) {
		c = getopt_long(argc, args, "aABcdrRStuUX",
				ls_option_table, NULL);
		if (c == -1)
			break;
		if (c == '?')
			exit(1);	//don't exit here
		process_ls_option(c);
	}

	if (list_files(args+optind)) {
		if (llist_is_empty(renames))
			printf(_("no files listed\n"));
		else
			printf(_("%d files listed\n"), llist_size(renames));
	}
}

void
process_ls_option(int c)
{
	switch (c) {
	case 'a': /* all */
		llist_add(ls_options, "--all");
		break;
	case 'A': /* almost-all */
		llist_add(ls_options, "--almost-all");
		break;
	case 'B': /* ignore-backups */
		llist_add(ls_options, "--ignore-backups");
		break;
	case 'c':
		llist_add(ls_options, "--time=ctime");
		break;
	case 'd': /* directory */
		llist_add(ls_options, "--directory");
		break;
	case 'r': /* reverse */
		llist_add(ls_options, "--reverse");
		break;
	case 'R': /* recursive */
		llist_add(ls_options, "--recursive");
		break;
	case 'S':
		llist_add(ls_options, "--sort=size");
		break;
	case SORT_OPT: /* sort */
		llist_add(ls_options, "--sort");
		llist_add(ls_options, optarg);
		break;
	case 't':
		llist_add(ls_options, "--sort=time");
		break;
	case TIME_OPT: /* time */
		llist_add(ls_options, "--time");
		llist_add(ls_options, optarg);
		break;
	case 'u':
		llist_add(ls_options, "--time=atime");
		break;
	case 'U':
		llist_add(ls_options, "--sort=none");
		break;
	case 'X':
		llist_add(ls_options, "--sort=extension");
		break;
	}
}

bool
cwd_from_renames_directory()
{
	if (old_dir != -1) {
		if (fchdir(old_dir) < 0) {
			close(old_dir);
			warn_errno(_("cannot change back to old directory"));
			return false;
		}
		close(old_dir);
		old_dir = -1;
	}
	return true;
}

bool
cwd_to_renames_directory()
{
	if (strcmp(renames_directory, ".") == 0) {
		old_dir = -1;
		return true;
	}

	if ((old_dir = open(".", O_RDONLY)) < 0) {
		warn_errno(_("cannot get current directory"));
		return false;
	}
	if (chdir(renames_directory) < 0) {
		close(old_dir);
		warn_errno(_("%s: cannot change to directory"), renames_directory);
		return false;
	}

	return true;
}

/**
 * Generate a list of specified files.
 * Return true if listing succeeded.
 *
 * XXX: clean up management of renames_directory, old and old_dir
 */
bool
list_files(char **args)
{
	char *firstdir;
	LList *ls_args_list;
	char **ls_args;
	pid_t ls_pid;
	int ls_fd;
	int status;

	free(renames_directory);
	renames_directory = xstrdup(".");

	ls_args_list = llist_clone(ls_options);	/* llist_add_all! */
	llist_add_last(ls_args_list, "--");
	llist_add_first(ls_args_list, "--quoting-style=c");
	llist_add_first(ls_args_list, "ls");

	if (llist_contains(ls_options, "--directory")) {
		firstdir = ".";
	} else {
		firstdir = (*args == NULL ? NULL : *args);
	}

	for (; *args != NULL; args++) {
		if (*args == firstdir
				&& args[1] == NULL
				&& is_directory(*args)) {
			char *old = renames_directory;
			renames_directory = xstrdup(*args);
			if (!cwd_to_renames_directory()) {
				free(renames_directory);
				renames_directory = old;
				return false;
			}
			free(old);
			llist_add(ls_args_list, ".");
			firstdir = ".";
		} else {
			llist_add(ls_args_list, *args);
		}
	}

	ls_args = (char **) llist_to_null_terminated_array(ls_args_list);
	llist_free(ls_args_list);
	if (!run_ls(ls_args, &ls_pid, &ls_fd)) {
		if (old_dir != -1) {
			old_dir = -1;
			close(old_dir);
		}
		warn_errno(_("cannot execute `ls'"));
		free(ls_args);
		clean_ls(ls_pid, ls_fd, &status);
		return false;
	}
	free(ls_args);

	if (!cwd_from_renames_directory()) {
		clean_ls(ls_pid, ls_fd, &status);
		return false;
	}

	free_renames(renames);

	if (!process_ls_output(firstdir, renames, ls_fd)) {
		warn_errno(_("cannot read `ls' output"));
		clean_ls(ls_pid, ls_fd, &status);
		return false;
	}
	if (!clean_ls(ls_pid, ls_fd, &status))
		return false;
	if (!WIFEXITED(status)) {
		warn(_("ls: abnormal exit"));
		return false;
	}

	remove_unnecessary_files(renames);
	return true;
}

/**
 * Execute ls(1) in a child process. A pipe is created
 * so that the output of ls can be read from the parent
 * process.
 *
 * @param args
 *   The argument vector to use when executing ls.
 */
static bool
run_ls(char **args, pid_t *ls_pid, int *ls_fd)
{
	int child_pipe[2];
	pid_t child_pid;

	*ls_fd = -1;
	*ls_pid = -1;

	if (pipe(child_pipe) == -1)
		return false;
	*ls_fd = child_pipe[0];

	child_pid = fork();
	if (child_pid == -1) {
		close(child_pipe[1]);
		return false;
	}
	if (child_pid == 0) {
		if (close(child_pipe[0]) == -1)
			die_errno(NULL); /* OK */
		if (dup2(child_pipe[1], STDOUT_FILENO) == -1)
			die_errno(NULL); /* OK */
		execvp("ls", args);
		die_errno(_("cannot execute `ls'")); /* OK */
	}
	*ls_pid = child_pid;

	if (close(child_pipe[1]) == -1)
		return false;

	return true;
}

/**
 * Wait for the ls child process to exit.
 * @returns
 *   true if everything went successfully.
 */
static bool
clean_ls(pid_t ls_pid, int ls_fd, int *status)
{

	if (ls_fd != -1 && close(ls_fd) == -1)
		return false;
	if (ls_pid != -1 && waitpid(ls_pid, status, 0) != ls_pid)
		return false;
	/* We ignore the return code of the ls child process. The error
	 * will be discovered when reading the output of ls anyway.
	 */
	return true;
}

/** 
 * Read and parse output from ls(1), converting it
 * into a list of file paths.
 *
 * @param firstdir
 *   The first non-option argument to ls. If this is a directory,
 *   it is used as the default path for files listed.
 * @returns
 *   true if processing succeeded (otherwise errno will be set).
 */
static bool
process_ls_output(char *firstdir, LList *files, int ls_fd)
{
	HMap *map_files;
	FILE *lsout;
	char *dir;
	char *line;

	map_files = hmap_new();

	lsout = fdopen(ls_fd, "r");
	if (lsout == NULL)
		return false;

	dir = strbuf_new();
	if (firstdir != NULL
			&& strcmp(firstdir, ".") != 0
			&& is_directory(firstdir)) {
		strbuf_set(&dir, firstdir);
		if (firstdir[strlen(firstdir)-1] != '/')
			strbuf_append_char(&dir, '/');
	}

	while ((line = read_line(lsout)) != NULL) {
		int len;
		char *tmp;

		chomp(line);
		len = strlen(line);
		if (len != 0) {
			ls_assert(line[0] == '"');

			if (line[len-1] == ':') {
				ls_assert(line[len-2] == '"');
				line[len-2] = '\0';

				tmp = dequote_output_file(&line[1]);
				ls_assert(tmp != NULL);

				strbuf_set(&dir, tmp);
				if (dir[strlen(dir)-1] != '/')
					strbuf_append_char(&dir, '/');
			} else {
				FileSpec *spec;
			
				ls_assert(line[len-1] == '"');
				line[len-1] = '\0';

				tmp = dequote_output_file(&line[1]);
				ls_assert(tmp != NULL);

				spec = new_file_spec();
				spec->old_name = xasprintf("%s%s", dir, tmp);
				spec->new_name = xstrdup(spec->old_name);

				if (hmap_contains_key(map_files, spec->old_name)) {
					warn(_("%s: file already listed"), spec->old_name);
				} else {
					hmap_put(map_files, spec->old_name, spec);
					llist_add(files, spec);
				}
			}

			free(tmp);
		}
		free(line);
	}
	strbuf_free(&dir);
	hmap_free(map_files);

	return !ferror(lsout);
}

void
free_file_spec(FileSpec *spec)
{
	free(spec->old_name);
	free(spec->new_name);
	free(spec);
}

/**
 * Create and initialize a FileSpec instance.
 *
 * @returns
 *   A new FileSpec instance, which should be freed with free when
 *   no longer used.
 */
FileSpec *
new_file_spec(void)
{
	FileSpec *spec;
	
	spec = xmalloc(sizeof(FileSpec));
	spec->old_name = NULL;
	spec->new_name = NULL;
	spec->status = STATUS_NONE;
	spec->renamed = RENAME_UNDONE;
	spec->dependency = NULL;

	return spec;
}

/**
 * Remove the directory "." and ".." from the list of files,
 * if available.
 *
 * @param files
 *   A list of files.
 * @returns
 *   The same list, but without "." and ".." strings.
 */
static void
remove_unnecessary_files(LList *files)
{
	Iterator *it;
	for (it = llist_iterator(files); iterator_has_next(it); ) {
		FileSpec *spec = iterator_next(it);
		if (strcmp(spec->old_name, ".") == 0 || strcmp(spec->old_name, "..") == 0)
			iterator_remove(it);
	}
	iterator_free(it);
}
