/* sensor_sweep_applet.c - main source file for sensor_sweep_applet. */

/* Includes */
#include <applet-widget.h>

/* Sensor includes */
#include <sensors/sensors.h>
#include <sensors/error.h>
#include <sensors/chips.h>
/* #include <linux/sensors.h> */

/* Local Includes */
#include "sensorFunc.h"
#include "properties.h"
#include "structures.h"

/* Function Prototypes */
GList *setupSensors(void);
gint updateGUI(gpointer data);
gint updateSensors(gpointer data);
void aboutWindowCB(AppletWidget *caller, gpointer data);
void setupApplet(struct appletInfo *ai);
void destroy(GtkWidget *widget, gpointer data);
void print_hello(GtkWidget *widget, gpointer data);
void updateGUIWrapper(GtkWidget *widget, GdkEventAny* e, gpointer data);
static void applet_change_orient(GtkWidget *widget, PanelOrientType o, gpointer data);
static void applet_change_pixel_size(GtkWidget *widget, int size, gpointer data);
gchar* buildDisplayString(GList *list, struct appletInfo *ai);
void setTooltip(struct appletInfo *info);

#define VERSION "0.20.0"
/* Define NEW_LOOK for the 1.1.5+ gen_util panel look */
#define NEW_LOOK
#define LARGE_PANEL_ROWS 3
#define HUGE_PANEL_ROWS 4
#define RIDICULOUS_PANEL_ROWS 6

/* Main */
int main(int argc, char **argv) {
	gint ret;
	struct appletInfo *ai;
	FILE *configFile;
	GtkWidget *okdialog;

	/* Initialize the applet */
	applet_widget_init("sensor_sweep_applet", VERSION, argc, argv, NULL, 0, NULL);

	/* Allocate the memory for the main struct */
	ai = g_malloc(sizeof(struct appletInfo));
	ai->displayPos = 0;

	/* Open and initialize the configuration file */
	configFile = fopen("/etc/sensors.conf", "ro");
	if(configFile == NULL) {
		okdialog = gnome_ok_dialog("Could not open the lm_sensors config file (/etc/sensors.conf)");
		gtk_signal_connect(GTK_OBJECT(okdialog), "destroy", GTK_SIGNAL_FUNC(destroy), NULL);
		applet_widget_gtk_main();
		exit(1);
	}

	ret = sensors_init(configFile);
	if(ret != 0) {
		/* Could not initialize the config */
		if(ret == SENSORS_ERR_PROC) {
			/* Modules not loaded */
			okdialog = gnome_ok_dialog("/proc/sys/dev/sensors/chips or /proc/bus/i2c unreadable.\nMake sure you have the lm_sensor modules loaded.");
			gtk_signal_connect(GTK_OBJECT(okdialog), "destroy", GTK_SIGNAL_FUNC(destroy), NULL);
			applet_widget_gtk_main();
			exit(1);
		} else {
			/* Some other error */
			okdialog = gnome_ok_dialog("Could not initialize the config file.\nMake sure the lm_sensors modules are loaded.\n");
			gtk_signal_connect(GTK_OBJECT(okdialog), "destroy", GTK_SIGNAL_FUNC(destroy), NULL);
			applet_widget_gtk_main();
			exit(1); /* Exit the applet */
		}
	}

	/* Initialize the Gnome config system */
	gnome_config_push_prefix("/sensor_sweep_applet/");

	/* Setup the applets variables */
	setupApplet(ai);

	/* Setup the sensor Glist */
	ai->sensorList = setupSensors();
	ai->applet = applet_widget_new("sensor_sweep_applet");
	if (!ai->applet)
		g_error("Can't create applet!\n");

	/* Setup the panel right click menus */
	applet_widget_register_stock_callback(APPLET_WIDGET(ai->applet), "Properties", GNOME_STOCK_MENU_PROP, _("Properties"), properties, ai);
	applet_widget_register_stock_callback(APPLET_WIDGET(ai->applet), "About", GNOME_STOCK_MENU_ABOUT, _("About"), aboutWindowCB, NULL);

	/* Bind the orientation and size signals */
	gtk_signal_connect(GTK_OBJECT(ai->applet), "change_orient", GTK_SIGNAL_FUNC(applet_change_orient), ai);
	gtk_signal_connect(GTK_OBJECT(ai->applet), "change_pixel_size", GTK_SIGNAL_FUNC(applet_change_pixel_size), ai);

	/* Create and setup the event box widget and it's tooltip */
	ai->eventbox = gtk_event_box_new();
	applet_widget_add(APPLET_WIDGET(ai->applet), ai->eventbox);
	gtk_widget_set_events(ai->eventbox, GDK_BUTTON_PRESS_MASK);
	gtk_signal_connect(GTK_OBJECT(ai->eventbox), "button_press_event", GTK_SIGNAL_FUNC(updateGUIWrapper), ai);
	ai->tooltip = gtk_tooltips_new();

	/* Create and setup the frame widget */
	ai->frame = gtk_frame_new(NULL);

	#ifdef NEW_LOOK
		gtk_frame_set_shadow_type(GTK_FRAME(ai->frame), GTK_SHADOW_ETCHED_IN);
	#else
		gtk_frame_set_shadow_type(GTK_FRAME(ai->frame), GTK_SHADOW_IN);
	#endif

	gtk_container_add(GTK_CONTAINER(ai->eventbox), ai->frame);

	/* Create and setup the alignment widget */
	ai->align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
	gtk_container_set_border_width(GTK_CONTAINER(ai->align), 3);
	gtk_container_add(GTK_CONTAINER(ai->frame), ai->align);

	/* The Vbox widget */
	ai->vbox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(ai->align), ai->vbox);

	/* Create the label and pack it */
	ai->label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(ai->vbox), ai->label, TRUE, TRUE, 0);

	/* All the main widgets are gtk_widget_show()n in the */
	/* 'change_pixel_size' and 'change_orient' signal handlers */
	/* This is done to avoid a nasty problem where the applet did not */
	/* set it's size properly until one of these signals were emitted. */
	
	/* Update the sensors */
	updateSensors(ai);
	ai->sensorsTimeoutTag = gtk_timeout_add(ai->sensorFreq, updateSensors, ai);

	/* Start the GUI update function that updates the display */
	updateGUI(ai);
	ai->GUITimeoutTag = gtk_timeout_add(ai->displayFreq, updateGUI, ai);

	/* Start the applet */
	applet_widget_gtk_main();
	
	return(0);
}

/*************/
/* Functions */
/*************/

#if DEBUG == 1
void print_hello(GtkWidget *widget, gpointer data) {
	/* Little function used to verify that a signal handler is being called */
	g_print("print_hello()\n");
}
#endif

void setupApplet(struct appletInfo *ai) {
	ai->height = gnome_config_get_int("Global/height=44");
	ai->width = gnome_config_get_int("Global/width=100");
	ai->sensorFreq = gnome_config_get_int("Global/sensorFreq=10000");
	ai->displayFreq = gnome_config_get_int("Global/displayFreq=10000");
}

GList *setupSensors(void) {
	/* Read in the configured sensors and setup the GList structure to hold sensors */
	struct sensor *tempSensor;
	GList *list=NULL; /* Empty list */
	gint numSensors, i;
	gchar *string;

	/* Setup our string buffer */
	string = g_malloc(40);

	/* How many sensors are defined? Default to 0 */
	numSensors = gnome_config_get_int("Global/Number_Of_Sensors=0");

	if(numSensors != 0) {
		/* Iterate through sensors getting info */

		for(i=0;i<numSensors;i++) {
			tempSensor = g_malloc(sizeof(struct sensor));

			/* Read in the values */
			sprintf(string, "Sensor%i/chipName=NULL", i);
			tempSensor->chipName = gnome_config_get_string(string);
			sprintf(string, "Sensor%i/sensorName=NULL", i);
			tempSensor->sensorName = gnome_config_get_string(string);
			sprintf(string, "Sensor%i/displayName=NULL", i);
			tempSensor->displayName = gnome_config_get_string(string);

			/* If any of the above returned NULL there is a problem with that entry. Do not process */
			if(strcmp(tempSensor->chipName, "NULL") != 0 && strcmp(tempSensor->sensorName, "NULL") != 0 && strcmp(tempSensor->displayName, "NULL") != 0) {

				/* Get the sensor_chip_name struct filled out */
				if(getName(tempSensor->chipName, &tempSensor->nameStruct) != 0) {
					g_print("Could not get sensor Name\n");
					g_free(tempSensor);
				} else {
					/* Get the sensor number */
					tempSensor->sensorNum = getSensorNumber(&tempSensor->nameStruct, tempSensor->sensorName);
					if(tempSensor->sensorNum != 0) {
						/* Add it to the GList */
						list = g_list_append(list, tempSensor);
					} else {
						g_print("Could not get sensor number %i\n", i);
						g_free(tempSensor);
					}
				}
			} else {
				/* We don't need this GList entry since something is wrong with it */
				g_free(tempSensor);
			}
		}
	}

	/* Free the buffer */
	g_free(string);

	return(list);
}

void updateGUIWrapper(GtkWidget *widget, GdkEventAny* e, gpointer data) {
	/* GTK signal func wrapper to update the display */
	struct appletInfo *ai;
	ai = (struct appletInfo *) data;

	#if DEBUG == 1
		g_print("updateGUIWrapper() called\n");
	#endif

	/* Remove the timeout functions */
	gtk_timeout_remove(ai->sensorsTimeoutTag);
	gtk_timeout_remove(ai->GUITimeoutTag);

	/* Update the GUI */
	updateGUI(data);

	/* Re-instate the timeout funcs */
	ai->sensorsTimeoutTag = gtk_timeout_add(ai->sensorFreq, updateSensors, ai);
	ai->GUITimeoutTag = gtk_timeout_add(ai->displayFreq, updateGUI, ai);
}

gint updateGUI(gpointer data) {
	struct sensor *temp;
	struct appletInfo *ai;
	GList *list;
	gint pos;
	gchar *string1, *tempstring, *displayString;
	gint i;

	#if DEBUG == 1
		g_print("updateGUI() called\n");
	#endif

	ai = (struct appletInfo *) data;
	list = ai->sensorList;

	/* If this list is empty then just clear the UI labels and
	return FALSE so this timeout doesn't keep going */
	if(ai->sensorList == NULL) {
		gtk_label_set_text(GTK_LABEL(ai->label), "Empty");
		return(FALSE);
	}

	/* Read the values and update the GUI */

	/* Advance the list pointer to the proper position */
        for(pos=0; pos < ai->displayPos; pos++) {
                list = g_list_next(list);
        }

	/* Get 1st Value */
	temp = (struct sensor *) list->data;

	ai->displayPos++;
	if(g_list_next(list) == NULL) {
		ai->displayPos = 0;
	}
	list = g_list_next(list);

	/* Do things for the panel size and orientation */
	if(ai->orient == ORIENT_UP || ai->orient == ORIENT_DOWN) {
		#if DEBUG == 1
			g_print("ORIENT_UP or ORIENT_DOWN Update\n");
		#endif

		string1 = g_strdup_printf(" %s: %.2f ", temp->displayName,
				temp->value);

		if(ai->panelSize <= PIXEL_SIZE_SMALL) {
			/* No second sensor for this panel size */
			#if DEBUG == 1
				g_print("SIZE_TINY or SIZE_SMALL Update\n");
			#endif
			displayString = string1;

		} else if(ai->panelSize == PIXEL_SIZE_STANDARD) {
			#if DEBUG == 1
				g_print("SIZE_STANDARD Update\n");
			#endif
	
			tempstring = buildDisplayString(list, ai);
			displayString = g_strconcat(string1, tempstring, NULL);
			g_free(string1);
			g_free(tempstring);

		} else if(ai->panelSize == PIXEL_SIZE_LARGE) {
			#if DEBUG == 1
				g_print("SIZE_LARGE Update\n");
			#endif
	
			for(i=0;i < LARGE_PANEL_ROWS - 1;i++) {
				tempstring = buildDisplayString(list, ai);
				displayString = g_strdup_printf("%s%s", string1,
					tempstring);
				g_free(tempstring);
				g_free(string1);
				string1 = displayString;
				list = g_list_next(list);
			}
	
		} else if(ai->panelSize == PIXEL_SIZE_HUGE) {
			#if DEBUG == 1
				g_print("SIZE_HUGE Update\n");
			#endif
	
			for(i=0;i < HUGE_PANEL_ROWS - 1;i++) {
				tempstring = buildDisplayString(list, ai);
				displayString = g_strdup_printf("%s%s", string1,
					tempstring);
				g_free(tempstring);
				g_free(string1);
				string1 = displayString;
				list = g_list_next(list);
			}

		} else if(ai->panelSize == PIXEL_SIZE_RIDICULOUS) {
			#if DEBUG == 1
				g_print("SIZE_RIDICULOUS Update\n");
			#endif
	
			for(i=0;i < RIDICULOUS_PANEL_ROWS - 1;i++) {
				tempstring = buildDisplayString(list, ai);
				displayString = g_strdup_printf("%s%s", string1,
					tempstring);
				g_free(tempstring);
				g_free(string1);
				string1 = displayString;
				list = g_list_next(list);
			}
	
		} else {
			/* We have no clue what to do with this size so */
			#if DEBUG == 1
				g_print("SIZE UNKNOWN Update\n");
			#endif
			displayString = string1;
		}
	} else {
		#if DEBUG == 1
			g_print("ORIENT_LEFT or ORIENT_RIGHT Update\n");
		#endif

		string1 = g_strdup_printf("%s:\n%.2f", temp->displayName,
				temp->value);

		displayString = string1;
	}

	/* Set the label */
	gtk_label_set_text(GTK_LABEL(ai->label), displayString);

	/* Set the tooltip */
	setTooltip(ai);

	/* Free the string buffers */
	g_free(displayString);

	#if DEBUG == 1
		g_print("updateGUI() done.\n");
	#endif

	return(TRUE); /* Return TRUE to continue the timeout function */
}

void setTooltip(struct appletInfo *ai) {
	GList	*list;
	gchar	*string, *string2;
	struct	sensor	*temp;

	list = ai->sensorList;
	if (list != NULL) {
	    	temp = list->data;

		string = g_strdup_printf("Latest Sensor Readings\n%s: %.2f",
			temp->displayName, temp->value);
		list = list->next;

		while (list != NULL) {
			temp = list->data;

			string2 = g_strdup_printf("%s\n%s: %.2f ",
			    string, temp->displayName, temp->value);
			g_free(string);
			string = string2;

			list = list->next;
		}

		gtk_tooltips_set_tip(ai->tooltip, ai->eventbox, string, NULL);
		g_free(string);
	}
}

gint updateSensors(gpointer data) {
	/* Update all of the sensor values */
	struct sensor *temp;
	struct appletInfo *ai;
	GList *list;

	#if DEBUG == 1
		g_print("updateSensors() called\n");
	#endif

	ai = (struct appletInfo *) data;

	/* If there list is empty return FALSE to stop
	calling this function */
	if(ai->sensorList == NULL) {
		return(FALSE);
	}

	list = ai->sensorList;

	while(list != NULL) {
		temp = (struct sensor *) list->data;
		getFeature(&temp->nameStruct, temp->sensorNum, &temp->value);

		list = list->next;
	}

	#if DEBUG == 1
		g_print("updateSensors() done\n");
	#endif

	return(TRUE); /* Return TRUE to continue the timeout function */
}

void aboutWindowCB(AppletWidget *caller, gpointer data) {
	const gchar *authors[] = {
		"Dan Siemon",
		NULL
	};

	gtk_widget_show (gnome_about_new ("sensor_sweep_applet", VERSION,
	"Copyright Dan Siemon <dan@coverfire.com> 2000.",
	(const gchar **) authors,
	_("http://www.coverfire.com/sensor_sweep_applet/\nsensor_sweep_applet is a small Gnome panel applet that shows your systems health via the lm_sensors kernel modules."),
	NULL));
}

void destroy(GtkWidget *widget, gpointer data) {
	#if DEBUG == 1
		g_print("destroy() called\n");
	#endif

	gtk_main_quit();
}

static void applet_change_orient(GtkWidget *widget, PanelOrientType o, gpointer data) {
	struct appletInfo *ai;

	#if DEBUG == 1
		g_print("Applet Change Orientation Signal\n");
		switch(o) {
			case ORIENT_UP: g_print("ORIENT UP\n"); break;
			case ORIENT_DOWN: g_print("ORIENT DOWN\n"); break;
			case ORIENT_LEFT: g_print("ORIENT LEFT\n"); break;
			case ORIENT_RIGHT: g_print("ORIENT RIGHT\n"); break;
		}
	#endif

	ai = (struct appletInfo *) data;

	/* Set the frame widget for verticle or horizontal orientation */
	ai->orient = o;

	if(o == ORIENT_UP || o == ORIENT_DOWN) {
		gtk_widget_set_usize(GTK_WIDGET(ai->frame), ai->width, -1);
	} else {
		gtk_widget_set_usize(GTK_WIDGET(ai->frame), ai->panelSize, ai->height);
	}

	/* Show the widgets in case this is the first time */
	/* either the 'change_pixel_size' or 'change_orient' */
	/* signals have been emitted */
	gtk_widget_show(ai->label);
	gtk_widget_show(ai->vbox);
	gtk_widget_show(ai->align);
	gtk_widget_show(ai->frame);
	gtk_widget_show(ai->eventbox);
	gtk_widget_show(ai->applet);

	/* Update the GUI */
	updateGUI(ai);
}

static void applet_change_pixel_size(GtkWidget *widget, int size, gpointer data) {
	struct appletInfo *ai;

	#if DEBUG == 1
		g_print("Applet Change Size Signal: %i\n", size);
	#endif

	ai = (struct appletInfo *) data;

	ai->panelSize = size;

	if(ai->orient == ORIENT_UP || ai->orient == ORIENT_DOWN) {
		gtk_widget_set_usize(GTK_WIDGET(ai->frame), ai->width, -1);
	} else {
		gtk_widget_set_usize(GTK_WIDGET(ai->frame), ai->panelSize, ai->height);
	}

	/* Set the align container border for each panel size */
	/* This is so that this applet matches the gen_util clock */
	if(size < PIXEL_SIZE_SMALL) {
		gtk_container_set_border_width(GTK_CONTAINER(ai->align), 0);
	} else if(size >= PIXEL_SIZE_SMALL) {
		gtk_container_set_border_width(GTK_CONTAINER(ai->align), 3);
	}

	/* Show the widgets in case this is the first time */
	/* either the 'change_pixel_size' or 'change_orient' */
	/* signals have been emitted */
	gtk_widget_show(ai->label);
	gtk_widget_show(ai->vbox);
	gtk_widget_show(ai->align);
	gtk_widget_show(ai->frame);
	gtk_widget_show(ai->eventbox);
	gtk_widget_show(ai->applet);

	/* Update the GUI */
	updateGUI(ai);
}

gchar* buildDisplayString(GList *list, struct appletInfo *ai) {
	struct sensor *temp;
	gchar *string;

	#if DEBUG == 1
		g_print("buildDisplayString() called.\n");
	#endif

	if(list != NULL) {
		temp = (struct sensor *) list->data;

		string = g_strdup_printf("\n %s: %.2f ", temp->displayName, temp->value);

		if(g_list_next(list) == NULL) {
			ai->displayPos = 0;
		} else {
			ai->displayPos++;
		}
	} else {
		ai->displayPos = 0;
		string = g_strdup_printf("\n");
	}

	#if DEBUG == 1
		g_print("buildDisplayString() done.\n");
	#endif


	return (string);
}
