# 
# This module encapsulated the status function. It sends information to the
# Spong server. 
#
# It reports the current status of a service, and sends along
# a string of information that might be helpful in diagnosing the problem.
# This code is modeled after the bb program, but is a little different in
# that it handles multi-line messages and send over the time as an int,
# rather then a string.
#
# methods
#  status( SERVERADDR, HOST, SERVICE, COLOR, SUMMARY, MESSAGE, [TTL] )

package Spong::Status;

require Exporter;

use strict;
use vars qw(@ISA @EXPORT_OK $VERSION);
use Carp;
use IO::Socket;

@ISA = qw(Exporter);
@EXPORT_OK = qw(status event);
$VERSION = 0.02;

# Format and send a status update message

sub status {
   my( $addr, $host, $cat, $color, $summary, $message, $ttl ) = @_;
   my( $ts );
   if ( defined $ttl ) { $ts = time() . ":$ttl"; } else { $ts = time(); }

   my $msg = "status $host $cat $color $ts $summary\n$message\n";

   my $errmsg = SendMsg( $addr, $main::SPONG_UPDATE_PORT, $msg );

   main::error("Status: $errmsg") if $errmsg;

}

# Format and send an event message

sub event {
   my( $addr, $host, $cat, $color, $summary, $message, $ttl ) = @_;
   my $ts = time(); 

   my $msg = "event $host $cat $color $ts $summary\n$message\n";

   my $errmsg = SendMsg( $addr, $main::SPONG_UPDATE_PORT, $msg );

   main::error("Event: $errmsg") if $errmsg;

}

# Format and send a paging message

sub page {
   my( $addr, $host, $cat, $color, $summary, $message, $ttl ) = @_;
   my( $ts ) = time();

   my $msg = "page $host $cat $color $ts $summary\n$message\n"; 

   my $errmsg = SendMsg( $addr, $main::SPONG_UPDATE_PORT, $msg );

   main::error("Page: $errmsg") if $errmsg;
}


#
# This routine handles the actual details of sending a message to the 
# spong-server.
#

sub SendMsg {
   my( $addr, $port, $msg ) = @_;
   my($errmsg) = "";
   my($sock);


   foreach my $server (split /\s+/,$addr) {

      # Parse the spong server name specified (host[:port])
      my($host, $p ) = split /:/,$server;
      $p = $port if (! $p);               # Default to $port if no port

      main::debug("Status.pm: host $host, port $p",7);

      # Set an alarm on this block in case we run into problem talking to
      # the spong server.
      eval 
      {
         local $SIG{'ALRM'} = sub { die("socket timed out to $server"); };
         alarm(30);

         $sock = IO::Socket::INET->new( PeerAddr => $host,
                                     PeerPort => $p,
                                     Proto    => 'tcp',
                                     Timeout  => 30,
                                     Reuse    => 1,
                                   );

         if ( ! defined $sock ) {
            alarm(0);
            die "server: $server - $@";
         }

         $sock->autoflush(1);
         $sock->print($msg);
         $sock->close();

         undef $sock;
         alarm(0);
      };

      # If we got an error message, add it to $errmsg
      if ($@) { $errmsg .= ($errmsg) ? "\n" : "" . $@; }

   }

   # Return any error messages where done
   return $errmsg;
}

