#!/usr/bin/tclsh
#
# Author: Vincent Ricard <vincent@magicninja.org>
#

package require fileutil

set sourceDir [lindex $argv 0]
if {[lindex $argv 1] == "-v"} {
    set invertMatch true
    set argv [lreplace $argv 1 1]
} else {
    set invertMatch false
}
switch -- [llength $argv] {
    3 {
	set translationFile [lindex $argv 1]
	set lang [lindex $argv 2]
    }
    2 {
	set lang "??"
	set translationFile [lindex $argv 1]
	regexp {([-a-z]+)\.msg$} $translationFile -> lang
    }
    1 {
	if {$invertMatch == "true"} {
	    puts stderr {extract sourceDir [[-v ] translationFile [lang]]}
	    exit 1
	}
	set lang "??"
	set translationFile ""
    }
    default {
	# -v: show useless translations
	puts stderr {extract sourceDir [[-v ] translationFile [lang]]}
	exit 1
    }
}

# Read all tcl file from sourceDir
set tclFileList [::fileutil::findByPattern $sourceDir -glob -- *tcl]
foreach filename $tclFileList {
    set fd [open $filename]

    while {-1 < [gets $fd line]} {
	while {[string match {*\\} $line] && \
		    ![string match {*\\\\} $line] && \
		    [gets $fd line1] >= 0} {
	    set line [string replace $line end end " [string trimleft $line1]"]
	}
        # Search: [ ::msgcat::mc "translation key"
        while {[regexp -- {\[::msgcat::mc[ \t\r\n]+\"(([^\"]|\\\")*)\"} $line whole key] || \
		    [regexp -- {\[::msgcat::mc[ \t\r\n]+{([^\}]*)}} $line whole key] || \
		    [regexp -- {\[::msgcat::mc[ \t\r\n]+([^ \t\r\n\{\"\[\]]*)} $line whole key]} {
	    if {$key != ""} {
		if {![info exists keyHash($filename)]} {
		    # Create a new list (with the current key) for this file
		    set keyHash($filename) [list $key]
		} elseif {[lsearch -exact $keyHash($filename) $key]<0} {
		    # key doesn't exist for this file
		    lappend keyHash($filename) $key
		}
	    }
	    set idx [string first $whole $line]
	    set line [string replace $line 0 [expr {$idx + [string length $whole] - 1}]]
        }
    }
    close $fd
}

# Remove duplicated keys (through all files)
set fileList [array names keyHash]
for {set i 0} {$i < [llength $fileList]} {incr i} {
    for {set j [expr $i + 1]} {$j < [llength $fileList]} {incr j} {
        foreach k $keyHash([lindex $fileList $i]) {
            set J [lindex $fileList $j]
            set ix [lsearch -exact $keyHash($J) $k]
            if {-1 < $ix} {
                set keyHash($J) [lreplace $keyHash($J) $ix $ix]
            }
        }
    }
}

if {0 != [string compare "" $translationFile] && [file exists $translationFile]} {
    # Read translation file
    set fd [open $translationFile]
    set translated [list]

    while {-1 < [gets $fd line]} {
        # Search: ::msgcat::mcset lang "translation key"
        if {[regexp -- {::msgcat::mcset [a-zA-Z]+[ \t\r\n]+\"(([^\"]|\\\")*)\"} $line whole key]} {
            lappend translated $key
        }
    }
    close $fd

    if {false == $invertMatch} {
        # Display untranslated keys
        foreach f [array names keyHash] {
            set displayFileName true
            foreach k $keyHash($f) {
                if {-1 == [lsearch -exact $translated $k] } {
                    if {true == $displayFileName} {
                        set displayFileName false
                        puts "# $f"
                    }
                    puts "::msgcat::mcset $lang \"$k\""
                }
            }
            if {false == $displayFileName} {
                puts ""
            }
        }
    } else {
        # Remove useless keys
        foreach t $translated {
            set found false
            foreach f [array names keyHash] {
                if {-1 < [lsearch -exact $keyHash($f) $t] } {
                    set found true
                }
            }
            if {false == $found} {
                puts "\"$t\""
            }
        }
    }
} else {
    if {false == $invertMatch} {
	# Print result
	foreach f [array names keyHash] {
	    if {0 < [llength $keyHash($f)]} {
		puts "# $f"
		foreach k $keyHash($f) {
                    puts "::msgcat::mcset $lang \"$k\""
		}
		puts ""
	    }
	}
    }
}
