# $Id: headlines.tcl,v 1.6 2006/02/09 20:08:51 aleksey Exp $

namespace eval headlines {
    variable headid 0

    variable headlines
    array set headlines {}

    variable options
    variable trees {}

    custom::defvar send_jids {} \
	[::msgcat::mc "List of JIDs to whom headlines have been sent."] \
	-group Hidden

    custom::defvar options(cache) 0 \
	[::msgcat::mc "Cache headlines on exit and restore on start."] \
	-group Messages -type boolean

    custom::defvar options(multiple) 0 \
	[::msgcat::mc "Display headlines in single/multiple windows."] \
	-group Messages -type radio -layout vertical \
	-values [list 0 [::msgcat::mc "Single window"] \
		      1 [::msgcat::mc "One window per bare JID"] \
		      2 [::msgcat::mc "One window per full JID"]]

    custom::defvar options(display_subject_only) 1 \
	[::msgcat::mc "Do not display headline descriptions as tree nodes."] \
	-group Messages -type boolean

    custom::defvar options(timestamp_format) {[%R] } \
	[::msgcat::mc "Format of timestamp in headline tree view. Set to\
		       empty string if you don't want to see timestamps."] \
	-group Messages -type string
}

package require md5

proc headlines::process_message {connid from id type is_subject subject body err thread priority x} {
    switch -- $type {
	headline {
	    show $connid $from $type $subject $body $thread $priority $x
	    return stop
	}
    }
    return
}

hook::add process_message_hook \
    [namespace current]::headlines::process_message

proc headlines::show {connid from type subject body thread priority x {data {}}} {
    global tcl_platform
    variable headid
    variable headlines
    variable trees
    variable options
    global font

    set subject [string trim $subject]
    set body [string trim $body]

    set desc ""
    set url ""
    set seconds [clock seconds]
    foreach extra $x {
        jlib::wrapper:splitxml $extra tag vars isempty chdata children
	switch -- [jlib::wrapper:getattr $vars xmlns] {
	    jabber:x:oob {
		foreach item $children {
		    jlib::wrapper:splitxml $item tag vars isempty chdata children   

		    switch -- $tag {
			desc -
			url {
			    set $tag [string trim $chdata]
			}
		    }
		}
	    }
	    jabber:x:delay {
		set stamp [jlib::wrapper:getattr $vars stamp]
		catch { set seconds [clock scan $stamp -gmt 1] }
	    }
	}
    }
    if {[cequal $url ""]} {
        return
    }
    if {[cequal $subject ""]} {
	set subject $desc
    } else {
	if {$options(display_subject_only)} {
	    set desc $subject
	}
    }
    if {[cequal $subject ""]} {
	return
    }

    switch -- $options(multiple) {
        0 {
            set hw .headlines
            set title [::msgcat::mc "Headlines"]
            set tabtitle [::msgcat::mc "Headlines"]
        }

        1 {
            set user [node_and_server_from_jid $from]
            set hw .headlines_[jid_to_tag $user]
            set title [format [::msgcat::mc "%s Headlines"] $user]
            set tabtitle [node_from_jid $from]
        }

        default {
            set hw .headlines_[jid_to_tag $from]
            set title [format [::msgcat::mc "%s Headlines"] $from]
            set tabtitle [node_from_jid $from]/[resource_from_jid $from]
        }
    }
    if {[lsearch -exact $trees [set tw $hw.tree]] < 0} {
        lappend trees $tw
    }

    if {![winfo exists $hw]} {
        add_win $hw -title $title -tabtitle $tabtitle \
            -raisecmd "focus [list $hw.tree]
                       tab_set_updated [list $hw]" -class JBrowser

	PanedWin $hw.pw -side right -pad 0 -width 8
	pack $hw.pw -fill both -expand yes

	set uw [$hw.pw add -weight 1]
	set dw [$hw.pw add -weight 1]

	frame $dw.date
	label $dw.date.label -anchor w -text [::msgcat::mc "Date:"]
	label $dw.date.ts -font $font -anchor w
	pack $dw.date -fill x
	pack $dw.date.label -side left
	pack $dw.date.ts -side left

	frame $dw.from
	label $dw.from.label -anchor w -text [::msgcat::mc "From:"]
	label $dw.from.jid -font $font -anchor w
	pack $dw.from -fill x
	pack $dw.from.label -side left
	pack $dw.from.jid -side left

	frame $dw.subject
	label $dw.subject.lsubj -anchor w -text [::msgcat::mc "Subject:"]
	label $dw.subject.subj -font $font -anchor w
	pack $dw.subject -fill x
	pack $dw.subject.lsubj -side left
	pack $dw.subject.subj -side left

	if {![info exists options(seencolor)]} {
	    if {[cequal $tcl_platform(platform) unix]} {
		set options(seencolor) [option get $hw disabledForeground JBrowser]
	    } else {
		set options(seencolor) [option get $hw nscolor JBrowser]
	    }
	}
	if {![info exists options(unseencolor)]} {
	    set options(unseencolor) [option get $hw fill JBrowser]
	}

        set sw [ScrolledWindow $uw.sw]
	Tree $tw -deltax 16 -deltay 18 \
	    -selectcommand [list [namespace current]::update_body \
				 $dw.date.ts $dw.from.jid $dw.subject.subj $hw.body]
        $sw setwidget $tw
        pack $sw -side top -expand yes -fill both

        $tw bindText <ButtonPress-3> \
                [list [namespace current]::select_popup $hw]
        $tw bindText <Double-ButtonPress-1> \
                [list [namespace current]::action browse $hw]
        $tw bindText <Any-Enter>  \
                [list [namespace current]::balloon $hw enter  %X %Y]
        $tw bindText <Any-Motion> \
                [list [namespace current]::balloon $hw motion %X %Y]
        $tw bindText <Any-Leave>  \
                [list [namespace current]::balloon $hw leave  %X %Y]

        # HACK
        bind $tw.c <Return> \
            "[namespace current]::action browse $hw \[$tw selection get\]"
        bind $tw.c <Delete> \
            "[namespace current]::action delete $hw \[$tw selection get\]"

        bindscroll $tw.c

	set dsw [ScrolledWindow $dw.sw]
	text $hw.body -font $font -height 12 -state disabled \
	     -wrap word -takefocus 1
	$dsw setwidget $hw.body
	pack $dsw -expand yes -fill both -anchor nw
	$hw.body tag configure emphasized -elide 1
	$hw.body tag configure nonemphasized -elide 0

	bind $hw.body <ButtonPress-1> [list focus %W]

	bind $hw.body <Key-Up>    [list Tree::_keynav up    $tw]
	bind $hw.body <Key-Down>  [list Tree::_keynav down  $tw]
	bind $hw.body <Key-Left>  [list Tree::_keynav left  $tw]
	bind $hw.body <Key-Right> [list Tree::_keynav right $tw]

	hook::run open_headlines_post_hook $hw
    }

    if {$options(multiple) > 1} {
        set text $subject
    } else {
        set text $from
    }
    set fnode [str2node $text]
    if {![$tw exists $fnode]} {
        $tw insert end root $fnode -text [string map [list "\n" " "] $text] -open 1 \
            -image browser/message -font $font \
            -fill $options(seencolor) \
            -data [list type from text $text unseen 0]
    }

    if {($options(multiple) > 1) || ([cequal $subject $desc])} {
        set snode $fnode
    } else {
        set snode $fnode-subject-[str2node $subject]

        if {![$tw exists $snode]} {
            $tw insert end $fnode $snode -text [string map [list "\n" " "] $subject] -open 1 \
                -image browser/message -font $font \
                -fill $options(seencolor) \
                -data [list type subject text $subject unseen 0]
        }
    }

    set anode $fnode-article-[incr headid]
    if {[$tw exists $anode]} {
        $tw delete $anode
    }
    array set props [list type article unseen 1 seconds $seconds]
    array set props $data
    array set props [list text $desc url $url body $body]
    set nodetext \
	[cconcat \
	     [clock format $props(seconds) -format $options(timestamp_format)] \
		    [string map [list "\n" " "] $desc]]
    $tw insert end $snode $anode -text $nodetext -open 1 \
        -fill $options(seencolor) -font $font \
        -data [array get props]
    if {$props(unseen)} {
        $tw itemconfigure $anode -fill $options(unseencolor)
    }

    set headlines($anode) [list $connid $from $type $subject $body $thread $priority $x]

    update $tw $anode
    tab_set_updated $hw 1 message
}

proc headlines::str2node {string} {
    set utf8str [encoding convertto utf-8 $string]
    if {[catch { ::md5::md5 -hex $utf8str } ret]} {
	return [::md5::md5 $utf8str]
    } else {
	return $ret
    }
}

proc headlines::update_body {wdate wfrom wsubj wbody tw node} {
    variable headlines
    
    if {[catch { array set props [$tw itemcget $node -data] }] ||
	    ![info exists props(type)] || \
	    $props(type) != "article"} {
        set from ""
	set subj ""
	set body ""
	set date ""
	set url ""
    } else {
	set from [lindex $headlines($node) 1]
	set subj [string map [list "\n" " "] $props(text)]
	set body $props(body)
	set date [clock format $props(seconds)]
	set url $props(url)
    }
    
    $wdate configure -text $date
    $wfrom configure -text $from
    $wsubj configure -text $subj
    
    $wbody configure -state normal
    $wbody delete 0.0 end
    chat::add_emoteiconed_text $wbody "$body\n\n" ""
    if {$url != ""} {
	#chat::add_url $wbody [::msgcat::mc "See more..."] $url
	chat::add_url $wbody $url $url \
	    -command [list [namespace code action] markseen \
			   [winfo parent $tw] $node]
    }
    $wbody configure -state disabled
}

proc headlines::update_menu {menu num} {
    variable send_jids

    set ind 3
    if {$num} {
	$menu delete $ind [expr $ind + $num - 1]
    }
    foreach jid $send_jids {
	$menu insert $ind command \
	    -label [format [::msgcat::mc "Forward to %s"] $jid] \
	    -command "[namespace current]::forward3 [list $menu] [list $jid] \
			  \$[namespace current]::headwindow \$[namespace current]::headnode"
	incr ind
    }
}

namespace eval headlines {
    if {[winfo exists [set m .h1popmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Browse"] \
	-command "[namespace current]::action browse \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add separator
    $m add command -label [::msgcat::mc "Forward..."] \
	-command "[namespace current]::action forward \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add separator
    $m add command -label [::msgcat::mc "Copy headline to clipboard"] \
	-command "[namespace current]::action copy_headline \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Copy URL to clipboard"] \
	-command "[namespace current]::action copy_url \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Toggle seen"] \
	-command "[namespace current]::action toggle \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Delete"] \
	-command "[namespace current]::action delete \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    update_menu $m 0

    if {[winfo exists [set m .h2popmenu]]} {
	destroy $m
    }
    menu $m -tearoff 0
    $m add command -label [::msgcat::mc "Sort"] \
	-command "[namespace current]::action sort \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Sort by date"] \
	-command "[namespace current]::action datesort \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Mark all seen"] \
	-command "[namespace current]::action markseen \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Mark all unseen"] \
	-command "[namespace current]::action markunseen \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Delete seen"] \
	-command "[namespace current]::action deleteseen \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
    $m add command -label [::msgcat::mc "Delete all"] \
	-command "[namespace current]::action delete \
		    \$[namespace current]::headwindow \$[namespace current]::headnode"
}

proc headlines::select_popup {hw node} {
    variable headwindow
    variable headnode

    $hw.tree selection set $node

    if {[catch { array set props [[set tw $hw.tree] itemcget $node -data] }]} {
        return
    }

    set headwindow $hw
    set headnode $node

    switch -- $props(type) {
        article {
           set hm .h1popmenu 
        }

        default {
           set hm .h2popmenu 
        }
    }

    tk_popup $hm [winfo pointerx .] [winfo pointery .]
}

proc headlines::action {action hw node} {
    variable headlines
    variable options

    if {[catch { array set props [[set tw $hw.tree] itemcget $node -data] }]} {
        return
    }

    switch -glob -- $props(type)/$action {
        article/browse {
            browseurl $props(url)

            if {$props(unseen)} {
                set props(unseen) 0
                $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
                update $tw $node
            }
        }

	article/forward {
	     forward .h1popmenu $tw $node
	}

	article/copy_headline {
	    clipboard clear -displayof $hw
	    clipboard append -displayof $hw "$props(text)\n$props(body)\n$props(url)"
	}

	article/copy_url {
	    clipboard clear -displayof $hw
	    clipboard append -displayof $hw $props(url)
	}

        article/toggle {
            if {$props(unseen)} {
                set props(unseen) 0
                set myfill $options(seencolor)
            } else {
                set props(unseen) 1
                set myfill $options(unseencolor)
            }
            $tw itemconfigure $node -fill $myfill -data [array get props]
            update $tw $node
        }

        article/markseen {
            set props(unseen) 0
            $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
            update $tw $node
        }

        article/markunseen {
            set props(unseen) 1
            $tw itemconfigure $node -fill $options(unseencolor) -data [array get props]
            update $tw $node
        }

        */delete {
            set props(unseen) 0
            $tw itemconfigure $node -fill $options(seencolor) -data [array get props]
            update $tw $node

            $tw delete $node
        }

        article/deleteseen {
            if {$props(unseen) == 0} {
		action delete $hw $node
	    }
        }

	from/markseen -
	subject/markseen {
            foreach child [$tw nodes $node] {
		action markseen $hw $child
            }
	}

	from/markunseen -
	subject/markunseen {
            foreach child [$tw nodes $node] {
		action markunseen $hw $child
            }
	}

	from/deleteseen -
	subject/deleteseen {
	    if {$props(unseen) > 0} {
		foreach child [$tw nodes $node] {
		    action deleteseen $hw $child
		}
	    } else {
		action delete $hw $node
	    }
	}

        from/sort -
        subject/sort {
            set children {}
            foreach child [$tw nodes $node] {
                catch { unset props }
                array set props [$tw itemcget $child -data]

                lappend children [list $child $props(text)]
            }
            set neworder {}
            foreach child [lsort -index 1 $children] {
                lappend neworder [lindex $child 0]
            }
            $tw reorder $node $neworder

            foreach child [$tw nodes $node] {
                action $action $hw $child
            }
        }

        from/datesort -
        subject/datesort {
            set children {}
	    set seconds [clock seconds]
            foreach child [$tw nodes $node] {
                catch { unset props }
		set props(seconds) $seconds
                array set props [$tw itemcget $child -data]

                lappend children [list $child $props(seconds)]
            }
            set neworder {}
            foreach child [lsort -decreasing -index 1 $children] {
                lappend neworder [lindex $child 0]
            }
            $tw reorder $node $neworder

            foreach child [$tw nodes $node] {
                action $action $hw $child
            }
        }

        default {
        }
    }
}

proc headlines::update {tw node} {
    variable options

    for {set parent [$tw parent $node]} \
            {![cequal $parent root]} \
            {set parent [$tw parent $parent]} {
        set unseen 0

        foreach child [$tw nodes $parent] {
            catch { unset props }
            array set props [$tw itemcget $child -data]

            incr unseen $props(unseen)
        }

        catch { unset props }
        array set props [$tw itemcget $parent -data]
        set props(unseen) $unseen

        set text $props(text)
        set myfill $options(seencolor)
        if {$unseen > 0} {
            append text " ($unseen)"
            set myfill $options(unseencolor)
        }
        $tw itemconfigure $parent -text $text -fill $myfill \
                -data [array get props]
    }
}

proc headlines::balloon {hw action X Y node} {
    if {[catch { array set props [$hw.tree itemcget $node -data] }]} {
        return
    }

    set width [expr {[winfo width $hw.tree] * 0.8}]
    if {$width < 400} {
	set width 400
    }

    switch -- $props(type) {
        article {
            if {![cequal $props(body) ""]} {
                balloon::default_balloon $hw:$node $action $X $Y $props(body) -width $width
            }
        }

        default {
        }
    }
}

proc headlines::save {} {
    variable options
    variable trees

    if {!$options(cache)} {
        return
    }

    if {[catch { open [set file1 [file join ~ .tkabber headlines1.tcl]] \
                      { WRONLY CREAT TRUNC } } fd]} {
        debugmsg headlines "unable to open $file: $fd"
        return
    }
    fconfigure $fd -encoding utf-8

    set code [catch {
        foreach tw $trees {
            save_aux $tw root $fd
        }
    } result]

    catch { close $fd }

    if {$code} {
        debugmsg headlines $result
        catch { file delete $file1 }

        return
    }

    set renameP 0
    if {![file exists [set file [file join ~ .tkabber headlines.tcl]]]} {
    } elseif {[file size $file] == 0} {
        catch { file delete -force $file }
    } else {
        set renameP 1
        catch { file rename -force $file \
                     [set file0 [file join ~ .tkabber headlines0.tcl]] }
    }

    if {![catch { file rename $file1 $file } result]} {
        return
    }

    debugmsg headlines "unable to rename $file1 to $file: $result"
    if {($renameP) && ([catch { file rename -force $file0 $file } result])} {
        debugmsg headlines "unable to rename $file0 back to $file: $result"
    }
    catch { file delete $file1 }

    return
}

proc headlines::save_aux {tw node fd} {
    variable headlines

    if {![winfo exists $tw]} {
        return
    }

    if {[llength [set children [$tw nodes $node]]] > 0} {
        foreach child $children {
            save_aux $tw $child $fd
        }
    } elseif {([info exists headlines($node)]) \
            && (![catch { array set props [$tw itemcget $node -data] }])} {
        puts $fd [concat [list [namespace current]::show] \
                         $headlines($node) [list [array get props]]]
    }
}

proc headlines::restore {} {
    variable options

    if {$options(cache)} {
	if {[file exists [set file [file join ~ .tkabber headlines.tcl]]]} {
	    set encoding [encoding system]
	    encoding system utf-8
	    catch { source $file }
	    encoding system $encoding
	}
    }

    return ""
}

proc headlines::forward3 {menu to tw node} {
    variable send_jids

    if {[catch { array set props [$tw.tree itemcget $node -data] } errmsg]} {
	return
    }

    # TODO: connid
    message::send_msg $to -type headline \
	-subject $props(text) \
	-body $props(body) \
	-xlist [list [jlib::wrapper:createtag x \
			  -vars [list xmlns jabber:x:oob] \
			  -subtags [list [jlib::wrapper:createtag url \
			                      -chdata $props(url)] \
					 [jlib::wrapper:createtag desc \
					      -chdata $props(text)]]]]
    set len [llength $send_jids]
    set send_jids [update_combo_list $send_jids $to 5]
    set custom::saved([namespace current]::send_jids) $send_jids
    custom::store
    update_menu $menu $len
}

proc headlines::forward2 {menu tw node} {
    global forward_hl
    variable send_jids

    if {[catch { array set props [$tw itemcget $node -data] } errmsg]} {
	return
    }

    set len [llength $send_jids]
    foreach choice [array names forward_hl] {
	if {$forward_hl($choice)} {
	    lassign $choice connid to
	    message::send_msg $to -type headline \
		-subject $props(text) \
		-body $props(body) \
		-xlist [list [jlib::wrapper:createtag x \
				  -vars [list xmlns jabber:x:oob] \
				  -subtags [list [jlib::wrapper:createtag url \
				                      -chdata $props(url)] \
						 [jlib::wrapper:createtag desc \
						      -chdata $props(text)]]]] \
		-connection $connid
	    set send_jids [update_combo_list $send_jids $to 5]
	}
    }
    set custom::saved([namespace current]::send_jids) $send_jids
    custom::store
    update_menu $menu $len
}

proc headlines::forward {menu tw node} {
    global forward_hl

    set gw .forward_headline
    catch { destroy $gw }

    set choices {}
    set balloons {}
    foreach c [jlib::connections] {
	foreach choice [roster::get_jids $c] {
	    if {![cequal [roster::itemconfig $c $choice -category] conference]} {
		lappend choices [list $c $choice] [roster::get_label $c $choice]
		lappend balloons [list $c $choice] $choice
	    }
	}
    }
    if {[llength $choices] == 0} {
        MessageDlg ${gw}_err -aspect 50000 -icon info \
	    -message [::msgcat::mc "No users in roster..."] -type user \
	    -buttons ok -default 0 -cancel 0
        return
    }

    CbDialog $gw [::msgcat::mc "Forward headline"] \
	[list [::msgcat::mc "Send"] "[namespace current]::forward2 [list $menu] [list $tw] [list $node]
				     destroy $gw" \
	      [::msgcat::mc "Cancel"] [list destroy $gw]] \
	forward_hl $choices $balloons
}

hook::add finload_hook [namespace current]::headlines::restore
hook::add quit_hook    [namespace current]::headlines::save

