/*========================================================================*\

Copyright (c) 1990-1999  Paul Vojta

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
PAUL VOJTA BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

NOTE:
	xdvi is based on prior work, as noted in the modification history
	in xdvi.c.

\*========================================================================*/

#include "xdvi-config.h"
#include <kpathsea/c-fopen.h>
#include <kpathsea/c-stat.h>
#include <kpathsea/magstep.h>
#include <kpathsea/tex-glyph.h>
#include "dvi.h"

#ifdef HTEX
#include "HTEscape.h"
#endif

#define	PK_PRE		247
#define	PK_ID		89
#define	PK_MAGIC	(PK_PRE << 8) + PK_ID
#define	GF_PRE		247
#define	GF_ID		131
#define	GF_MAGIC	(GF_PRE << 8) + GF_ID
#define	VF_PRE		247
#define	VF_ID_BYTE	202
#define	VF_MAGIC	(VF_PRE << 8) + VF_ID_BYTE
#if	defined(KANJI) && !defined(ASCIIDNP)
#define	JFMS_MAGIC	11
#ifdef PTEX
#define JFMS_TATEMAGIC  9
#endif
#endif	/* KANJI && !ASCIIDNP */

#define	dvi_oops(str)	(dvi_oops_msg = (str), longjmp(dvi_env, 1))

#ifdef SELFILE
extern void set_icon_and_title (); /* from xdvi.c */
extern void home (); /* from events.c */
#endif

static	struct stat fstatbuf;

static	Boolean	font_not_found;

/*
 * DVI preamble and postamble information.
 */
static	char	job_id[300];
static	long	numerator, denominator;

/*
 * Offset in DVI file of last page, set in read_postamble().
 */
static	long	last_page_offset;

extern int this_is_a_href;

#ifdef DVISEL
/*
 * Offset in DVI file of postamble, set in find_postamble().
 */
long postamble_offset;
#endif /* DVISEL */

/*
 *	free_vf_chain frees the vf_chain structure.
 */

static	void
free_vf_chain(tnp)
	struct tn *tnp;
{
	while (tnp != NULL) {
	    struct tn *tnp1 = tnp->next;
	    free((char *) tnp);
	    tnp = tnp1;
	}
}


/*
 *	Release all shrunken bitmaps for all fonts.
 */

void
reset_fonts()
{
	struct font *f;
	struct glyph *g;
#if	defined(KANJI) && !defined(ASCIIDNP)
	int n, maxchar;

	for (f = font_head; f != NULL; f = f->next)
	    if ((f->flags & FONT_LOADED) && !(f->flags & FONT_VIRTUAL)) {
		maxchar = (f->flags & FONT_KANJI) ? KTABLESIZE : f->maxchar + 1;
		for (n = 0; n < maxchar; ++n) {
		    g = (f->flags & FONT_KANJI) ? f->kglyph[n] : &f->glyph[n];
		    if (g != NULL) {
			if (g->bitmap2.bits) {
			    free(g->bitmap2.bits);
			    g->bitmap2.bits = NULL;
			}
#ifdef	GREY
			if (g->pixmap2) {
			    XDestroyImage(g->image2);
			    g->pixmap2 = NULL;
			}
#endif
		    }
		}
	    }
#else	/* KANJI && !ASCIIDNP */

	for (f = font_head; f != NULL; f = f->next)
	    if ((f->flags & FONT_LOADED) && !(f->flags & FONT_VIRTUAL))
		for (g = f->glyph; g <= f->glyph + f->maxchar; ++g) {
		    if (g->bitmap2.bits) {
			free(g->bitmap2.bits);
			g->bitmap2.bits = NULL;
		    }
#ifdef	GREY
		    if (g->pixmap2) {
			XDestroyImage(g->image2);
			g->pixmap2 = NULL;
			if (g->pixmap2_t != NULL) {
			    free(g->pixmap2_t);
			    g->pixmap2_t = NULL;
			}
		    }
#endif
		}
#endif	/* KANJI && !ASCIIDNP */
}

/*
 *	realloc_font allocates the font structure to contain (newsize + 1)
 *	characters.
 */

void
realloc_font(fontp, newsize)
	struct font	*fontp;
	wide_ubyte	newsize;
{
	struct glyph *glyph;

	glyph = fontp->glyph = xrealloc(fontp->glyph,
	    (unsigned int) (newsize + 1) * sizeof(struct glyph));
	if (newsize > fontp->maxchar)
	    bzero((char *) (glyph + fontp->maxchar + 1),
		(int) (newsize - fontp->maxchar) * sizeof(struct glyph));
	maxchar = fontp->maxchar = newsize;
}


/*
 *	realloc_virtual_font does the same thing for virtual fonts.
 */

void
realloc_virtual_font(fontp, newsize)
	struct font	*fontp;
	wide_ubyte	newsize;
{
	struct macro *macro;

	macro = fontp->macro = xrealloc(fontp->macro,
	    (unsigned int) (newsize + 1) * sizeof(struct macro));
	if (newsize > fontp->maxchar)
	    bzero((char *) (macro + fontp->maxchar + 1),
		(int) (newsize - fontp->maxchar) * sizeof(struct macro));
	maxchar = fontp->maxchar = newsize;
}


/*
 *	load_font locates the raster file and reads the index of characters,
 *	plus whatever other preprocessing is done (depending on the format).
 */

Boolean
load_font(fontp)
	struct font *fontp;
{
	double	fsize	= fontp->fsize;
	int	dpi	= fsize + 0.5;
	char	*font_found;
	int	size_found;
	int	magic;
	Boolean	hushcs	= hush_chk;

#ifdef	PTEX
	fontp->dir = 0;
#endif	/* PTEX */
#ifdef	ASCIIDNP
	if (iskanjifont(fontp->fontname)) return False;
#endif	/* ASCIIDNP */
	fontp->flags |= FONT_LOADED;
#ifdef	NTTZEIT
	if (isdnpfont(fontp->fontname)) {
	    fontp->timestamp = ++current_timestamp;
	    fontp->maxchar = maxchar = 255;
	    fontp->set_char_p = set_char;
	    fontp->file = NULL;
	    fontp->flags |= FONT_DNP;
	    read_NTT_index(fontp);
	    return False;
	}
#endif	/* NTTZEIT */
	fontp->file = font_open(fontp->fontname, &font_found,
	    fsize, &size_found, fontp->magstepval, &fontp->filename);
	if (fontp->file == NULL) {
	    Fprintf(stderr, "%s: Can't find font %s.%dpk\n", prog,
                    fontp->fontname, dpi);
	    return True;
	}
	--n_files_left;
	if (font_found != NULL) {
	    Fprintf(stderr,
		    "%s: can't find font %s; using %s instead at %d dpi.\n",
		    prog, fontp->fontname, font_found, dpi);
	    free(fontp->fontname);
	    fontp->fontname = font_found;
	    hushcs = True;
	}
	else if (!kpse_bitmap_tolerance ((double) size_found, fsize))
	    Fprintf(stderr,
		"%s: Can't find font %s at %d dpi; using %d dpi instead.\n",
		prog, fontp->fontname, dpi, size_found);
	fontp->timestamp = ++current_timestamp;
	fontp->maxchar = maxchar = 255;
#if	defined(KANJI) && !defined(ASCIIDNP)
	if (iskanjifont(fontp->fontname)) {
	    fontp->flags |= FONT_KANJI;
	    fontp->fsize = (float) size_found;
	    fontp->set_char_p = (set_char_proc)set_char2;
	}
	else {
	    fontp->fsize = (float) size_found; /* fixed by SATO Tatsuya, 03-Jul-2000 */
	    fontp->set_char_p = set_char;
	}
#else	/* KANJI && !ASCIIDNP */
	fontp->fsize = (float) size_found; /* fixed by SATO Tatsuya, 03-Jul-2000 */
	fontp->set_char_p = set_char;
#endif	/* KANJI && !ASCIIDNP */
	magic = two(fontp->file);
	if (magic == PK_MAGIC) read_PK_index(fontp, WIDENINT hushcs);
	else
#ifdef USE_GF
	    if (magic == GF_MAGIC) read_GF_index(fontp, WIDENINT hushcs);
	else
#endif
#if defined(Omega) || defined(KANJI)
	    if (magic == VF_MAGIC)
		maxchar = read_VF_index(fontp, WIDENINT hushcs);
#else
	    if (magic == VF_MAGIC) read_VF_index(fontp, WIDENINT hushcs);
#endif
	else
#if	defined(KANJI) && !defined(ASCIIDNP)
#ifdef PTEX
            if (magic == JFMS_MAGIC || magic == JFMS_TATEMAGIC) {
		fontp->dir = (magic == JFMS_TATEMAGIC);
#else
    if (magic == JFMS_MAGIC) {
#endif /* PTEX */
#ifdef	USE_ZEIT
		read_ZEIT_index(fontp);
#endif	/* USE_ZEIT */
		return False;
	    }
	else
#endif	/* KANJI && !ASCIIDNP */
	    oops("Cannot recognize format for font file %s", fontp->filename);

	if (fontp->flags & FONT_VIRTUAL) {
#if defined(Omega) || defined(KANJI)
#else
	    while (maxchar > 0 && fontp->macro[maxchar].pos == NULL) --maxchar;
	    if (maxchar < 255)
		realloc_virtual_font(fontp, WIDENINT maxchar);
#endif
	}
	else {
	    while (maxchar > 0 && fontp->glyph[maxchar].addr == 0) --maxchar;
	    if (maxchar < 255)
		realloc_font(fontp, WIDENINT maxchar);
	}
	return False;
}


/*
 *	MAGSTEPVALUE - If the given magnification is close to a \magstep
 *	or a \magstephalf, then return twice the number of \magsteps.
 *	Otherwise return NOMAGSTP.
 */

#define	NOMAGSTP (-29999)
#define	NOBUILD	29999

static	int
magstepvalue(mag)
	float	*mag;
{
  int m_ret;
  unsigned dpi_ret =
    kpse_magstep_fix ((unsigned) *mag, (unsigned) pixels_per_inch, &m_ret);
  *mag = (float) dpi_ret; /* MAG is actually a dpi.  */
  return m_ret ? m_ret : NOMAGSTP;
}

/*
 *	reuse_font recursively sets the flags for font structures being reused.
 */

static	void
reuse_font(fontp)
	struct font *fontp;
{
	struct font **fp;
	struct tn *tnp;

	if (fontp->flags & FONT_IN_USE) return;

	fontp->flags |= FONT_IN_USE;
	if (list_fonts)
	    Printf("xdvi: (reusing) %s at %d dpi\n", fontp->fontname,
		(int) (fontp->fsize + 0.5));
	if (fontp->flags & FONT_VIRTUAL) {
	    for (fp = fontp->vf_table; fp < fontp->vf_table + VFTABLELEN; ++fp)
		if (*fp != NULL) reuse_font(*fp);
	    for (tnp = fontp->vf_chain; tnp != NULL; tnp = tnp->next)
		reuse_font(tnp->fontp);
	}
}


/*
 *      define_font reads the rest of the fntdef command and then reads in
 *      the specified pixel file, adding it to the global linked-list holding
 *      all of the fonts used in the job.
 */
struct font *
define_font(file, cmnd, vfparent, tntable, tn_table_len, tn_headpp)
	FILE		*file;
	wide_ubyte	cmnd;
	struct font	*vfparent;	/* vf parent of this font, or NULL */
	struct font	**tntable;	/* table for low TeXnumbers */
	unsigned int	tn_table_len;	/* length of table for TeXnumbers */
	struct tn	**tn_headpp;	/* addr of head of list of TeXnumbers */
{
	int	TeXnumber;
	struct font *fontp;
	float	fsize;
	double	scale_dimconv;
	long	checksum;
	int	scale;
	int	design;
	int	magstepval;
	int	len;
	char	*fontname;
	int	size;

	TeXnumber = num(file, (int) cmnd - FNTDEF1 + 1);
	checksum = four(file);
	scale = four(file);
	design = four(file);
	len = one(file); len += one(file); /* sequence point in the middle */
	fontname = xmalloc((unsigned) len + 1);
	Fread(fontname, sizeof(char), len, file);
	fontname[len] = '\0';
	if(debug & DBG_PK)
	    Printf("xdvi: Define font \"%s\" scale=%d design=%d\n",
		fontname, scale, design);
	if (vfparent == NULL) {
	    fsize = 0.001 * scale / design * magnification * pixels_per_inch;
	    scale_dimconv = dimconv;
	}
	else {
	    /*
	     *	The scaled size is given in units of vfparent->scale * 2 ** -20
	     *	SPELL units, so we convert it into SPELL units by multiplying by
	     *		vfparent->dimconv.
	     *	The design size is given in units of 2 ** -20 pt, so we convert
	     *	into SPELL units by multiplying by
	     *		(pixels_per_inch * 2**16) / (72.27 * 2**20).
	     */
	    fsize = (72.27 * (1<<4)) * vfparent->dimconv * scale / design;
	    scale_dimconv = vfparent->dimconv;
	}
	magstepval = magstepvalue(&fsize);
	size = fsize + 0.5;
	/*
	 * reuse font if possible
	 */
	for (fontp = font_head;; fontp = fontp->next) {
	    if (fontp == NULL) {		/* if font doesn't exist yet */
		if (list_fonts)
		    Printf("xdvi: %s at %d dpi\n", fontname, (int) (fsize + 0.5));
		fontp = xmalloc((unsigned) sizeof(struct font));
		fontp->fontname = fontname;
		fontp->fsize = fsize;
		fontp->magstepval = magstepval;
		fontp->file = NULL;	/* needed if it's a virtual font */
		fontp->checksum = checksum;
		fontp->flags = FONT_IN_USE;
		fontp->dimconv = scale * scale_dimconv / (1<<20);
		fontp->set_char_p = load_n_set_char;
		/* With virtual fonts, we might be opening another font
		   (pncb.vf), instead of what we just allocated for
		   (rpncb), thus leaving garbage in the structure for
		   when close_a_file comes along looking for something.  */
		fontp->file = NULL; 
		fontp->filename = NULL;
		if (vfparent == NULL) font_not_found |= load_font(fontp);
		fontp->next = font_head;
		font_head = fontp;
		break;
	    }
	    if (strcmp(fontname, fontp->fontname) == 0
		    && size == (int) (fontp->fsize + 0.5)) {
			/* if font already in use */
		reuse_font(fontp);
		free(fontname);
		break;
	    }
	}
	if (TeXnumber < tn_table_len)
	    tntable[TeXnumber] = fontp;
	else {
	    struct tn *tnp;
	    tnp = xmalloc((unsigned) sizeof(struct tn));
	    tnp->next = *tn_headpp;
	    *tn_headpp = tnp;
	    tnp->TeXnumber = TeXnumber;
	    tnp->fontp = fontp;
	}
	return fontp;
}

#ifdef	KANJI
extern struct vfontmap *first_ptr;

int
iskanjifont(font)
	char *font;
{
	int len,i=1;
	struct vfontmap *current_ptr;

	current_ptr = first_ptr;

	while(current_ptr){
		len=strlen(current_ptr->jfmname);
		if (!strncmp(font,current_ptr->jfmname,len) &&
			(font[len] == '\0' || /* added by Yutaka OIWA */
				(font[len]>='0' && font[len]<='9')))
					return i;
		current_ptr = current_ptr->next_ptr;
		++i;
	}
 	return 0;
 }

int
isdnpfont(font)
	char *font;
{
	int i,j,len,len2;
	static char * NTT_jflist[] = {
			"jgreek", "jhira", "jka", "jkata", "jkb", "jkc", "jkd",
			"jke", "jkeisen", "jkf", "jkg", "jkh", "jki", "jkj",
			"jkk", "jkl", "jkm", "jkn", "jko", "jkp", "jkq", "jkr",
			"jks", "jkt", "jku", "jkv", "jkw", "jkx", "jky", "jkz",
			"jroma", "jrussian", "jsy", NULL };
	struct vfontmap *current_ptr;

	current_ptr = first_ptr;

	i=1;
	while(current_ptr){
		len=strlen(current_ptr->jfmname);
		if (!strncmp(font,current_ptr->jfmname,len)) {
			j=0;
			while(NTT_jflist[j]){
				len2=strlen(NTT_jflist[j]);
				if(!strncmp(font+len,NTT_jflist[j],len2) &&
				   (font[len+len2]>='0' && font[len+len2]<='9'))
								return i;
				j++;
			}
			return 0;
		}
		current_ptr = current_ptr->next_ptr;
		++i;
	}

 	return 0;
}
#endif	/* KANJI */

/*
 *      process_preamble reads the information in the preamble and stores
 *      it into global variables for later use.
 */
static	void
process_preamble()
{
	ubyte   k;

	if (one(dvi_file) != PRE)
		dvi_oops("Not a DVI file.");
#ifdef PTEX
	k = one(dvi_file);
	if (k != 2 && k != 3)
#else
	if (one(dvi_file) != 2)
#endif
		dvi_oops("Wrong version of DVI output for this program");
	numerator     = four(dvi_file);
	denominator   = four(dvi_file);
	magnification = four(dvi_file);
	dimconv = (((double) numerator * magnification)
		/ ((double) denominator * 1000.));
	dimconv = dimconv * (((long) pixels_per_inch)<<16) / 254000;
	tpic_conv = pixels_per_inch * magnification / 1000000.0;
#ifdef PTEX
        k = one(dvi_file);
        if (k != 2 && k != 3)
#else 
	k = one(dvi_file);
#endif
	Fread(job_id, sizeof(char), (int) k, dvi_file);
	job_id[k] = '\0';
}

/*
 *      find_postamble locates the beginning of the postamble
 *	and leaves the file ready to start reading at that location.
 */
#define	TMPSIZ	516	/* 4 trailer bytes + 512 junk bytes allowed */
static	void
find_postamble()
{
	long	pos;
	ubyte	temp[TMPSIZ];
	ubyte	*p;
	ubyte	*p1;
	ubyte	byte;

	Fseek(dvi_file, (long) 0, 2);
	pos = ftell(dvi_file) - TMPSIZ;
	if (pos < 0) pos = 0;
	Fseek(dvi_file, pos, 0);
	p = temp + fread((char *) temp, sizeof(char), TMPSIZ, dvi_file);
	for (;;) {
	    p1 = p;
	    while (p1 > temp && *(--p1) != TRAILER) ;
	    p = p1;
	    while (p > temp && *(--p) == TRAILER) ;
	    if (p <= p1 - 4) break;	/* found 4 TRAILER bytes */
	    if (p <= temp) dvi_oops("DVI file corrupted");
	}
	pos += p - temp;
	byte = *p;
	while (byte == TRAILER) {
	    Fseek(dvi_file, --pos, 0);
	    byte = one(dvi_file);
	}
#ifdef PTEX
        if (byte != 2 && byte != 3)
#else
	if (byte != 2)
#endif
	    dvi_oops("Wrong version of DVI output for this program");
	Fseek(dvi_file, pos - 4, 0);
#ifdef DVISEL
	Fseek(dvi_file, postamble_offset = sfour(dvi_file), 0);
#else
	Fseek(dvi_file, sfour(dvi_file), 0);
#endif /* DVISEL */
}


/*
 *      read_postamble reads the information in the postamble,
 *	storing it into global variables.
 *      It also takes care of reading in all of the pixel files for the fonts
 *      used in the job.
 */
static	void
read_postamble()
{
	ubyte   cmnd;
	struct font	*fontp;
	struct font	**fontpp;

	if (one(dvi_file) != POST)
	    dvi_oops("Postamble doesn't begin with POST");
	last_page_offset = four(dvi_file);
	if (numerator != four(dvi_file)
		|| denominator != four(dvi_file)
		|| magnification != four(dvi_file))
	    dvi_oops("Postamble doesn't match preamble");
		/* read largest box height and width */
	unshrunk_page_h = (spell_conv(sfour(dvi_file)) >> 16) + offset_y;
	if (unshrunk_page_h < unshrunk_paper_h)
	    unshrunk_page_h = unshrunk_paper_h;
	unshrunk_page_w = (spell_conv(sfour(dvi_file)) >> 16) + offset_x;
	if (unshrunk_page_w < unshrunk_paper_w)
	    unshrunk_page_w = unshrunk_paper_w;
	(void) two(dvi_file);	/* max stack size */
	total_pages = two(dvi_file);
	font_not_found = False;
	while ((cmnd = one(dvi_file)) >= FNTDEF1 && cmnd <= FNTDEF4)
	    (void) define_font(dvi_file, cmnd, (struct font *) NULL,
		tn_table, TNTABLELEN, &tn_head);
	if (cmnd != POSTPOST)
	    dvi_oops("Non-fntdef command found in postamble");
	if (font_not_found)
	    dvi_oops("Not all pixel files were found");
	/*
	 * free up fonts no longer in use
	 */
	fontpp = &font_head;
	while ((fontp = *fontpp) != NULL)
	    if (fontp->flags & FONT_IN_USE)
		fontpp = &fontp->next;
	    else {
		if (debug & DBG_PK)
		    Printf("xdvi: Discarding font \"%s\" at %d dpi\n",
			fontp->fontname, (int) (fontp->fsize + 0.5));
		*fontpp = fontp->next;		/* remove from list */
		free(fontp->fontname);
		if (fontp->flags & FONT_LOADED) {
		    if (fontp->file != NULL) {
			Fclose(fontp->file);
			++n_files_left;
		    }
		    free(fontp->filename);
		    if (fontp->flags & FONT_VIRTUAL) {
			struct macro *m;

			for (m = fontp->macro;
				m <= fontp->macro + fontp->maxchar; ++m)
			    if (m->free_me) free((char *) m->pos);
			free((char *) fontp->macro);
			free((char *) fontp->vf_table);
			free_vf_chain(fontp->vf_chain);
		    }
		    else if (fontp->glyph != NULL) { /* skip empty fonts (tjc) */
			struct glyph *g;
#if	defined(KANJI) && !defined(ASCIIDNP)
			int n, maxchar;

			maxchar = (fontp->flags & FONT_KANJI)
				  ? KTABLESIZE : fontp->maxchar;
			for (n = 0; n < maxchar; ++n) {
			    g = (fontp->flags & FONT_KANJI)
				? fontp->kglyph[n] : &fontp->glyph[n];
			    if (g != NULL) {
				if (g->bitmap.bits != NULL)
				    free(g->bitmap.bits);
				if (g->bitmap2.bits != NULL)
				    free(g->bitmap2.bits);
#ifdef	GREY
				if (g->pixmap2 != NULL)
				    XDestroyImage(g->image2);
#endif
				if (fontp->flags & FONT_KANJI)
				    free((char *) g);
			    }
			}
			if (fontp->flags & FONT_KANJI)
			    free((char **) fontp->kglyph);
			else
			    free((char *) fontp->glyph);
#else	/* KANJI && !ASCIIDNP */

			for (g = fontp->glyph;
				g <= fontp->glyph + fontp->maxchar; ++g) {
			    if (g->bitmap.bits != NULL) free(g->bitmap.bits);
			    if (g->bitmap2.bits != NULL) free(g->bitmap2.bits);
#ifdef	GREY
			    if (g->pixmap2 != NULL) {
				XDestroyImage(g->image2);
				if (g->pixmap2_t != NULL)
				    free(g->pixmap2_t);
			    }
#endif
			}
			free((char *) fontp->glyph);
#endif	/* KANJI && !ASCIIDNP */
		    }
		    free((char *) fontp);
		}
	    }
}

static	void
prepare_pages()
{
	int i;

#ifdef PAGENUM
       long offset;

       page_index = (PAGE_INDEX *)xmalloc((unsigned)total_pages
                                          * sizeof(PAGE_INDEX));
       for (offset = last_page_offset, i = total_pages; i > 0; i--) {
           Fseek(dvi_file, offset, 0);
           if (one(dvi_file) != BOP) {
               dvi_oops("page description doesn't begin with BOP");
           }
           page_index[i-1].offset = offset;
           page_index[i-1].number = four(dvi_file); /* Read c[0] */

           Fseek(dvi_file, (long)(9*4), 1);
           offset = four(dvi_file);
       }
#else
	page_offset = xmalloc((unsigned) total_pages * sizeof(long));
	i = total_pages;
	page_offset[--i] = last_page_offset;
	Fseek(dvi_file, last_page_offset, 0);
	/*
	 * Follow back pointers through pages in the DVI file,
	 * storing the offsets in the page_offset table.
	 */
	while (i > 0) {
	    Fseek(dvi_file, (long) (1+4+(9*4)), 1);
	    Fseek(dvi_file, page_offset[--i] = four(dvi_file), 0);
	}
#endif /* PAGENUM */
}

void
init_page()
{
	page_w = ROUNDUP(unshrunk_page_w, mane.shrinkfactor) + 2;
	page_h = ROUNDUP(unshrunk_page_h, mane.shrinkfactor) + 2;
}

/*
 *	init_dvi_file is the main subroutine for reading the startup
 *	information from the dvi file.  Returns True on success.
 */

static	Boolean
init_dvi_file()
{
	(void) fstat(fileno(dvi_file), &fstatbuf);
	if (S_ISDIR(fstatbuf.st_mode))
	    return False;
	dvi_time = fstatbuf.st_mtime;
	process_preamble();
	find_postamble();
	read_postamble();
	prepare_pages();
	init_page();
	if (current_page >= total_pages) current_page = total_pages - 1;
	warn_spec_now = warn_spec;
#ifdef COLOR
	InitColorStack();
#endif
#if	PS
	ps_newdoc();
#endif
	return True;
}

/**
 **	open_dvi_file opens the dvi file and calls init_dvi_file() to
 **	initialize it.
 **/

void
open_dvi_file()
{
	int	n;
	char	*file;

	if (setjmp(dvi_env)) oops(dvi_oops_msg);

	n = strlen(dvi_name);
	file = dvi_name;

	if (!this_is_a_href) {
	  /*
	   * Try foo.dvi before foo, in case there's an executable foo with
	   * documentation foo.tex.  Unless it already ends with ".dvi".
	   */
	  if (n < sizeof(".dvi")
	      || strcmp(dvi_name + n - sizeof(".dvi") + 1, ".dvi") != 0) {
	    dvi_name = xmalloc((unsigned) n + sizeof(".dvi"));
	    Strcpy(dvi_name, file);
	    Strcat(dvi_name, ".dvi");
	    dvi_file = xfopen(dvi_name, OPEN_MODE);
	    if (dvi_file != NULL && init_dvi_file())
	      return;
	    free(dvi_name);
	    dvi_name = file;
	  }
	}

	/* Then try `foo', in case the user likes DVI files without `.dvi'. */
	if ((dvi_file = xfopen(dvi_name, OPEN_MODE)) == NULL
		|| !init_dvi_file()) {
	    perror(dvi_name);
#if	PS
	    ps_destroy();
#endif
	    exit(1);
	}
}

#ifdef SELFILE
/* Allow the user to choose a new dvi file, by popping up a dialog box
   which allows the graphical selection of the correct filename,
   maybe we should only allow files ending in .dvi to be selected.  */

FILE *
select_filename(open, move_home)
    int open, move_home ;
{
  extern FILE *XsraSelFile();
  FILE *dummy_file ;
  static char *dummy_name ;

  dummy_file = XsraSelFile(top_level, "Select a dvi file: ",
			   "Ok", "Cancel",
			   "Can't open file: ", NULL,
			   OPEN_MODE, NULL, &dummy_name) ;
  if (dummy_file != NULL) {
    /* we may not want the file they returned... */
    if (!open)
      fclose (dummy_file) ;

    /* The name is what we really want, so use it, but turn it into an URL */
    free(dvi_name);
    dummy_name = HTEscape(dummy_name,URL_PATH);
    dvi_name = xmalloc((unsigned) strlen(dummy_name)+6);
    strcat(strcpy(dvi_name,"file:"),dummy_name);
    /* Should we really free dummy_name? */

    current_page = 0 ;  /* go to start of new dvi file */
    if (move_home)
      home(False);      /* Move to home position on new first page */
    
    /* We do this elsewhere if we don't open the file.  */
    if (open)
      set_icon_and_title (dvi_name, NULL, NULL, 1);

  } else if (open) { /* User cancelled, so open old file */
    dummy_file = xfopen(dvi_name, OPEN_MODE);
    if (dummy_file == NULL) 
      dvi_oops("Could not open old file");
    --dvi_time;
  }

  return dummy_file ;
}
#endif  /* SELFILE */

/**
 **	Check for changes in dvi file.
 **/

Boolean
check_dvi_file()
{
	struct font *fontp;

	if (dvi_file == NULL || fstat(fileno(dvi_file), &fstatbuf) != 0
	    || fstatbuf.st_mtime != dvi_time) {
		if (dvi_file) {
		    Fclose(dvi_file);
		    dvi_file = NULL;
		    if (list_fonts) Putchar('\n');
		}
#ifdef PAGENUM
		if (page_index != (PAGE_INDEX *) NULL) {
		    free((char *) page_index);
		    page_index = (PAGE_INDEX *) NULL;
#else
		if (page_offset != (long *) NULL) {
		    free((char *) page_offset);
		    page_offset = (long *) NULL;
#endif /* PAGENUM */
		}
		bzero((char *) tn_table, (int) sizeof(tn_table));
		free_vf_chain(tn_head);
		tn_head = NULL;
		for (fontp = font_head; fontp != NULL; fontp = fontp->next)
		    fontp->flags &= ~FONT_IN_USE;
#ifdef COLOR
		if (page_color) {
		    int i;
		    for (i = 0; i <= total_pages; i++)
			if (page_color[i].color_stack)
			    free((char *) page_color[i].color_stack);
		    free((char *)page_color);
		}
		if (color_allocated_top) {
		    XFreeColors(DISP, our_colormap,
				color_allocated, color_allocated_top, 0);
		    color_allocated_top = 0;
		}
#endif
#ifdef SELFILE
		if ((dvi_time > fstatbuf.st_mtime) && /* choose a new file */
			(dvi_file = select_filename(True, True)) == NULL)
		    dvi_oops("Cannot open new dvi file.");
		else
#endif  /* SELFILE */

#ifdef HTEX
		{
		    int url_aware_save = URL_aware;
		    URL_aware = True;
		    dvi_file = xfopen(dvi_name, OPEN_MODE);
		    URL_aware = url_aware_save;
		}
		if (dvi_file == NULL)
#else /* HTEX */
		if ((dvi_file = xfopen(dvi_name, OPEN_MODE)) == NULL)
#endif /* HTEX */
		    dvi_oops("Cannot reopen dvi file.");

		/* This function should be applied in both cases, for a
		 * new opened or a reread dvi file or we will get empty
		 * page_offset and fonts. <werner@suse.de>
		 */
		if (!init_dvi_file())
		    dvi_oops("Cannot initialise dvi file.");

		reconfig();
#ifdef HTEX
		htex_reinit();
#endif /* HTEX */
		redraw_page();
		return False;
	}
	return True;
}

#ifdef	ASCIIDNP
static	double	kanji_width[] = {
	0.962216, 0.747434, 0.504013, 0.353665,
	0.962216, 0.747434, 0.504013, 0.353665,
};

#ifdef PTEX
static        double  tkanji_width[] = {
	0.962216, 0.747434, 0.481108
};

static        char    tkanji_sym[] = {
	0,0,2,2,2,2,2,0,0,0,0,0,0,0,0,0,0,0,0,1,1,1,1,1,0,1,0,0,0,0,2,0,
	0,0,2,2,0,0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};
#endif /* PTEX */

static	char	kanji_sym[] = {
	0,0,2,2,3,3,7,7,7,6,7,3,3,7,7,6,6,0,0,6,0,6,1,1,0,1,0,0,0,0,7,0,
	0,0,7,7,0,0,7,3,6,2,6,2,6,2,6,2,6,2,6,2,6,2,6,2,6,2,6,2,0,0,0,0,
	0,0,0,0,0,0,0,0,1,1,1,2,2,2,0,0,1,1,1,0,0,0,0,0,6,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

static	char	kanji_hira[] = {
	0,1,0,1,0,1,0,1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,1,0,1,0,1,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

static	char	kanji_kata[] = {
	0,1,0,1,0,1,0,1,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,1,0,1,0,1,0,0,0,0,0,0,1,0,0,0,0,0,0,1,1,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
};

void
adj_kanjiwidth(fontp, subindex, c)
	struct font *fontp;
	unsigned int subindex;
	ubyte c;
{
	char *table;
#ifdef PTEX
        extern void rotate_bitmap();
	extern struct TateCorrection TateCorrectionTable[];
#endif

	switch (subindex) {
	  case 1:
#ifdef PTEX
	    if (fontp->dir) table = tkanji_sym;
	    else
#endif
	    table = kanji_sym;
	    break;
	  case 3:
	    table = kanji_hira;
	    break;
	  case 4:
	    table = kanji_kata;
	    break;
	  default:
#ifdef PTEX
	    if (fontp->dir) {
		fontp->glyph[c].x +=
			fontp->dimconv * 16 * 0.458221;
		fontp->glyph[c].y =
			(fontp->glyph[c].bitmap.h
				- fontp->dimconv * 16 * tkanji_width[0]) / 2;
	    }
#endif
	    fontp->glyph[c].dvi_adv =
		    fontp->dimconv * (1 << 20) * kanji_width[0];
	    return;
	}

#ifdef PTEX
	if (fontp->dir) {
		int     code;

		code = 0;
		switch (subindex) {
		case 1:
			if (c < 0x5f) code = c - 1;
			break;
		case 3:
			if (c < 0x4f) code = c - 1 + (0x7f - 0x21) + (0x2f - 0x21);
			break;
		case 4:
			if (c < 0x57)
			code = c - 1 + (0x7f - 0x21) + (0x2f - 0x21) + (0x6f - 0x21);
			break;
		}

		if (TateCorrectionTable[code].code == TCOR_ROTATE) {
			rotate_bitmap(&fontp->glyph[c].bitmap);
			fontp->glyph[c].x -= (fontp->glyph[c].bitmap.h
						- fontp->glyph[c].bitmap.w)/2;
		}
		if (TateCorrectionTable[code].code) {
			if (subindex==3 || subindex==4) { /* for small KANA */
				TateCorrectionTable[code].x /= 2;
				TateCorrectionTable[code].y /= 2;
			}
			switch (TateCorrectionTable[code].code) {
			case TCOR_AUTOSHIFT:
			  fontp->glyph[c].x -= fontp->dimconv * 24 * 0.458221
				- fontp->glyph[c].bitmap.w;
			  break;
			case TCOR_SHIFT:
			fontp->glyph[c].x -= (fontp->dimconv * 32 * 0.458221
					* TateCorrectionTable[code].x / 128);
			fontp->glyph[c].y -= (fontp->dimconv * 32 * 0.458221
					* TateCorrectionTable[code].y / 128);
			  break;
			}
		}

		fontp->glyph[c].x += fontp->dimconv * 16 * 0.458221;
		fontp->glyph[c].y =
			(fontp->glyph[c].bitmap.h
			- fontp->dimconv * 16 * tkanji_width[table[c]]) / 2;

		fontp->glyph[c].dvi_adv =
			fontp->dimconv * (1 << 20) * tkanji_width[table[c]];
		return;
	}
#endif /* PTEX */
	fontp->glyph[c].dvi_adv =
		fontp->dimconv * (1 << 20) * kanji_width[table[c]];
	if (table[c] == 1)
	    fontp->glyph[c].x = -fontp->glyph[c].y / 5;
	else if (table[c] >= 4)
	    fontp->glyph[c].x = -fontp->glyph[c].y / 10;
}

/**** jsf_names[] : borrowed from jvi2ps1.3j ****/

static	char	*jsf_names[] = {
	"", "jsy", "jroma", "jhira", "jkata",
	"jgreek", "jrussian", "jkeisen",
	"jka", "jkb", "jkc", "jkd", "jke", "jkf",
	"jkg", "jkh", "jki", "jkj", "jkk", "jkl",
	"jkm", "jkn", "jko", "jkp", "jkq", "jkr",
	"jks", "jkt", "jku", "jkv", "jkw", "jkx",
	"jky", "jkz", NULL,
};

void
define_kanjifont(n)
	unsigned int n;
{
	unsigned int base, index;
	register struct tn *tnp;
	struct font *fontp, *real_fontp;
	float fsize;
	int len;
	char *fontname, *name, *head;
	int size;
#ifdef PTEX
	int	dir;
#endif

	base = n & 0xffffff;
	index = n >> 24;

/* the following 3 lines added by Shigeharu TAKENO <shige@iee.niit.ac.jp> */
	if(base < currinf.tn_table_len)
	  real_fontp = currinf.tn_table[base];
	else
	for (tnp = currinf.tn_head; tnp != NULL; tnp = tnp->next)
	    if (tnp->TeXnumber == base) {
		real_fontp = tnp->fontp;
		break;
	    }
	tnp = (struct tn *)
		xmalloc((unsigned) sizeof(struct tn));
	tnp->next = tn_head;
	tn_head = tnp;
	currinf.tn_head = tn_head;
	tnp->TeXnumber = n;
	fsize = real_fontp->fsize;
	name = real_fontp->fontname;
#ifdef PTEX
	dir = iskanjifont(name);
	if ((dir >= 1) && (dir <= 3)) head = "dm";
#else
	if ((iskanjifont(name) >= 1) && (iskanjifont(name) <= 3)) head = "dm";
#endif
	else	head = "dg";
#ifdef PTEX
	dir = ((dir == 3) || (dir == 6));
#endif
	while (!isdigit(*name))
	    name++;
	len = strlen(head) + strlen(jsf_names[index]) + strlen(name);
	fontname = xmalloc((unsigned) len + 1);
	Sprintf(fontname, "%s%s%s", head, jsf_names[index], name);
	if (debug & DBG_PK)
	    Printf("Define kanjifont \"%s\"\n", fontname);
	/*
	 * reuse font if possible
	 */
	for (fontp = font_head;; fontp = fontp->next) {
	    if (fontp == NULL) {	/* if font doesn't exist yet */
		if (list_fonts)
		    Printf("%s at %d dpi\n", fontname, (int) (fsize + 0.5));
		fontp = (struct font *) xmalloc((unsigned) sizeof(struct font));
		fontp->dimconv = real_fontp->dimconv;
		fontp->fontname = fontname;
		fontp->fsize = fsize;
		fontp->flags = FONT_IN_USE;
		load_font(fontp);
		fontp->next = font_head;
		font_head = fontp;
		break;
	    }
	    if (strcmp(fontname, fontp->fontname) == 0
		    && size == (int) (5 * fontp->fsize + 0.5)) {
					/* if font already in use */
		reuse_font(fontp);
		free(fontname);
		break;
	    }
	}
#ifdef PTEX
	fontp->dir = dir;
#endif
	tnp->fontp = fontp;
	if (index == 1) {
	    struct glyph *g;

	    /* dainippon symbol font; Zenkaku blank is not defined in it. */
	    g = &fontp->glyph[1];
	    g->x = 0; 
	    g->y = 0; 
	    g->bitmap.w = 1; 
	    g->bitmap.h = 1;
	    alloc_bitmap(&g->bitmap);
	    bzero(g->bitmap.bits, g->bitmap.h * g->bitmap.bytes_wide);
	    adj_kanjiwidth(fontp, index, 1);
	}
}
#endif	/* ASCIIDNP */
