% -*- LaTeX -*-
% $Id: MachResult.lhs,v 1.1 2004/02/17 22:51:23 anoncvs_phyz Exp $
%
% Copyright (c) 1998-2003, Wolfgang Lux
% See LICENSE for the full license.
%
\nwfilename{MachResult.lhs}
\subsection{Building a disjunctive expression}
When we construct the answer expression from the final state of the
computation all (free) variables will be given unique names. For the
global free variables, these are the names, which where specified at
the program startup. The other variables will be assigned fresh
names. For the answer part of the result, we will consider only those
variables, which have been bound to some value (even if it is another
unbound variable). As a side effect, the process shown below will
construct the list of variables used in the answer expression.
\begin{verbatim}

> module MachResult where
> import MachTypes
> import MachNode
> import List
> import Set
> import Monad
> import Error
> import Combined

> type BrowseState a = StateT [Integer] MachState a

> newtype MachState a = MachState (MachStateT a)
> instance Functor MachState where
>   fmap f (MachState m) = MachState (fmap f m)
> instance Monad MachState where
>   return m = MachState (return m)
>   MachState m >>= f = MachState (m >>= \x -> let MachState m' = f x in m')
> instance RefMonad MachState where
>   newRef x = MachState (newRef x)
>   readRef r = MachState (readRef r)
>   writeRef r x = MachState (writeRef r x)

> browse :: [(String,NodePtr)] -> NodePtr -> MachStateT ShowS
> browse freeVars node = 
>   call (browseResult freeVars node) (map (nodeAdr . snd) freeVars)
>   where nodeAdr (Ptr adr _) = adr
>         call m is = let MachState m' = callSt m is in m'

> browseResult :: [(String,NodePtr)] -> NodePtr -> BrowseState ShowS
> browseResult freeVars ptr =
>   do
>     answer <- browseSubsts names freeVars
>     cstrs <- browseConstraints names freeVars ptr
>     exp <- browseElem names ptr
>     return (showsAnswer (answer ++ cstrs) exp)
>   where names = variableNames freeVars

> browseExpression :: Int -> [String] -> Integer -> Node -> BrowseState ShowS
> browseExpression p names adr (CharNode c) = return (shows c)
> browseExpression p names adr (IntNode i) = return (shows i)
> browseExpression p names adr (FloatNode f) = return (shows f)
> browseExpression p names adr (ConstructorNode _ name args)
>   | isTupleName name = fmap showsTuple (mapM (browseElem names) args)
>   | name == cons =
>       do
>         mbCs <- getString args
>         case mbCs of
>           Just cs -> return (shows cs)
>           Nothing -> fmap showsList (browseList names args)
>   | otherwise = fmap (showsTerm p name) (mapM (browseArg names) args)
>   where ConstructorTag _ cons _ = consTag
> browseExpression p names adr (VarNode _ _ _) =
>   fmap showString (varName names adr)
> browseExpression p names adr (SuspendNode (Ptr adr' ref) _) =
>   readRef ref >>= browseExpression p names adr'
> browseExpression p names adr (QueueMeNode _ _) =
>   return (showString "Suspended")
> browseExpression p names adr (IndirNode (Ptr adr' ref)) =
>   readRef ref >>= browseExpression p names adr'
> browseExpression p names adr (ClosureNode name _ _ args) =
>   fmap (showsTerm p name) (mapM (browseArg names) args)
> browseExpression p names adr (SearchContinuation _ _ _ _) =
>   return (showString "<search>")

> browseArg :: [String] -> NodePtr -> BrowseState ShowS
> browseArg names (Ptr adr ref) = readRef ref >>= browseExpression 1 names adr

> browseElem :: [String] -> NodePtr -> BrowseState ShowS
> browseElem names (Ptr adr ref) = readRef ref >>= browseExpression 0 names adr

> browseList :: [String] -> [NodePtr] -> BrowseState ShowS
> browseList names [head,tail] =
>   do
>     hd <- browseElem names head
>     tl <- derefPtr tail >>= browseTail names
>     return (hd . tl)

> browseTail :: [String] -> NodePtr -> BrowseState ShowS
> browseTail names (Ptr adr ref) =
>   readRef ref >>= \node ->
>   case node of
>     ConstructorNode _ cName args
>       | cName == nil -> return id
>       | cName == cons -> fmap (showString "," .) (browseList names args)
>     _ -> fmap (showString "|" .) (browseExpression 0 names adr node)
>   where ConstructorTag _ nil _  = nilTag
>         ConstructorTag _ cons _ = consTag

> getString :: [NodePtr] -> BrowseState (Maybe String)
> getString [head,tail] =
>   do
>     mbC <- getStringHead head
>     case mbC of
>       Just c ->
>         do
>           mbCs <- getStringTail tail
>           case mbCs of
>             Just cs -> return (Just (c:cs))
>             Nothing -> return Nothing
>       Nothing -> return Nothing

> getStringHead :: NodePtr -> BrowseState (Maybe Char)
> getStringHead (Ptr _ ref) =
>   readRef ref >>= \node ->
>   case node of
>     CharNode c -> return (Just c)
>     IndirNode ptr -> getStringHead ptr
>     _ -> return Nothing

> getStringTail :: NodePtr -> BrowseState (Maybe String)
> getStringTail (Ptr _ ref) =
>   readRef ref >>= \node ->
>   case node of
>     ConstructorNode _ cName args
>       | cName == nil -> return (Just [])
>       | cName == cons -> getString args
>       | otherwise -> return Nothing
>     IndirNode ptr -> getStringTail ptr
>     _ -> return Nothing
>   where ConstructorTag _ nil _  = nilTag
>         ConstructorTag _ cons _ = consTag

> browseSubsts :: [String] -> [(String,NodePtr)] -> BrowseState [ShowS]
> browseSubsts names freeVars =
>   mapM readVar freeVars >>= mapM (browseSubst names) . filter isBound
>   where readVar (name,Ptr adr ref) =
>           readRef ref >>= \node -> return (name,adr,node)
>         isBound (_,_,VarNode _ _ _) = False
>         isBound _ = True

> browseSubst :: [String] -> (String,Integer,Node) -> BrowseState ShowS
> browseSubst names (name,adr,node) =
>   fmap (showString (name ++ " = ") .) (browseExpression 0 names adr node)

> browseConstraints ::
>     [String] -> [(String,NodePtr)] -> NodePtr -> BrowseState [ShowS]
> browseConstraints names freeVars ptr =
>   foldM constrainedVars zeroSet (ptr : map snd freeVars) >>=
>   mapM constraints . toListSet >>=
>   mapM (browseConstraint names) . concat
>   where constraints (Ptr adr ref) =
>           varName names adr >>= \name ->
>           readRef ref >>= \(VarNode cs _ _) ->
>           return [(name,c) | c <- cs]

> browseConstraint :: [String] -> (String,Constraint) -> BrowseState ShowS
> browseConstraint names (name,DisEq (Ptr adr ref)) =
>   readRef ref >>= 
>   fmap (showString (name ++ " /= ") .) . browseExpression 0 names adr

> constrainedVars :: Set NodePtr -> NodePtr -> BrowseState (Set NodePtr)
> constrainedVars vars ptr@(Ptr _ ref) =
>   readRef ref >>= \node ->
>   case node of
>     ConstructorNode _ _ args -> foldM constrainedVars vars args
>     SuspendNode fn _ -> constrainedVars vars fn
>     VarNode cstrs _ _
>       | ptr `notElemSet` vars && not (null cstrs) ->
>           foldM constrainedVars (addToSet ptr vars) [ptr | DisEq ptr <- cstrs]
>       | otherwise -> return vars
>     IndirNode ptr -> constrainedVars vars ptr
>     ClosureNode _ _ _ args -> foldM constrainedVars vars args
>     _ -> return vars

> showsAnswer :: [ShowS] -> ShowS -> ShowS
> showsAnswer answer exp
>   | null answer = exp
>   | otherwise = braces ('{','}') (catBy ", " answer) . showChar ' ' . exp

> showsTerm :: Int -> String -> [ShowS] -> ShowS
> showsTerm p root args =
>   showParen (not (null args) && p > 0) (catBy " " (showString root : args))

> showsTuple :: [ShowS] -> ShowS
> showsTuple args = braces ('(',')') (catBy "," args)

> showsList :: ShowS -> ShowS
> showsList = braces ('[',']')

> catBy :: String -> [ShowS] -> ShowS
> catBy sep = cat . intersperse (showString sep)

> cat :: [ShowS] -> ShowS
> cat = foldr (.) id

> braces :: (Char,Char) -> ShowS -> ShowS
> braces (lb,rb) x = showChar lb . x . showChar rb

\end{verbatim}
The assignment of names to variables uses a list of names together
with the list of known variable addresses. For each variable a unique
name is returned and the list of known addresses may be updated. The
function \texttt{varName} assumes that the name supply is unbounded.
\begin{verbatim}

> varName :: [String] -> Integer -> BrowseState String
> varName names adr = 
>   do
>     (adrs1,adrs2) <- fmap (break (adr ==)) fetchSt
>     when (null adrs2) (updateSt_ (++ [adr]))
>     return (names !! length adrs1)

\end{verbatim}
The list of variable names is initialized with the names of the global
variables, followed by a supply of generated names. This generator
is just a copy of the function used to generate fresh variables names
in the interpreter \ToDo{so they should probably be joined into a
single utility function}. Note that we use lowercase letters for
variable names here, too.
\begin{verbatim}

> variableNames :: [(String,NodePtr)] -> [String]
> variableNames freeVars = names ++ filter (`notElem` names) genVars
>   where names = map fst freeVars
>         genVars = [genName c i | i <- [0..], c <- ['a'..'z']]
>         genName c 0 = [c]
>         genName c i = c : show i

\end{verbatim}
