<?php
/*
 +--------------------------------------------------------------------+
 | Copyright CiviCRM LLC. All rights reserved.                        |
 |                                                                    |
 | This work is published under the GNU AGPLv3 license with some      |
 | permitted exceptions and without any warranty. For full license    |
 | and copyright information, see https://civicrm.org/licensing       |
 +--------------------------------------------------------------------+
 */

/**
 * Base class for exceptions generated by CiviCRM.
 * This Exception returns more information than the default one. We are using it from the
 * form layer to allow redirects to occur without having redirects in the BAO
 * @param string $message
 *   The human friendly error message.
 * @param string $error_code
 *   A computer friendly error code. By convention, no space (but underscore allowed).
 *   ex: mandatory_missing, duplicate, invalid_format
 * @param array $data
 *   Extra params to return. eg an extra array of ids. It is not mandatory, but can help the computer using the api. Keep in mind the api consumer isn't to be trusted. eg. the database password is NOT a good extra data.
 */
class CRM_Core_Exception extends PEAR_Exception {
  private $errorData = [];

  /**
   * Class constructor.
   *
   * @param string $message
   * @param int $error_code
   * @param array $errorData
   * @param null $previous
   */
  public function __construct($message, $error_code = 0, $errorData = [], $previous = NULL) {
    parent::__construct($message);
    $this->errorData = $errorData + ['error_code' => $error_code];
  }

  /**
   * Custom string representation of object.
   *
   * @return string
   */
  public function __toString() {
    return __CLASS__ . ": [{$this->errorData['error_code']}: {$this->message}\n";
  }

  /**
   * Get error code.
   *
   * @return mixed
   */
  public function getErrorCode() {
    return $this->errorData['error_code'];
  }

  /**
   * Return specific error information.
   *
   * (Can be used for more detailed error messages or translation.)
   *
   * This method may be overridden in child exception classes in order
   * to add functionality not present in PEAR_Exception and is a placeholder
   * to define API
   *
   * The returned array must be an associative array of parameter => value like so:
   * <pre>
   * array('name' => $name, 'context' => array(...))
   * </pre>
   * @return array
   */
  public function getErrorData() {
    return $this->errorData;
  }

}
