#ifndef _RHEO_FORM_ELEMENT_H
#define _RHEO_FORM_ELEMENT_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// for checking the spectral convergence exp(-ak)
// on the reference element
//
// author: Pierre.Saramito@imag.fr
//
// date: 1 october 2017
//
#include "rheolef/skit.h"
#include "rheolef/quadrature.h"
#include "rheolef/basis.h"

namespace rheolef {

// -----------------------------------------------------------------------------
// mass
// -----------------------------------------------------------------------------
template<class T, class Basis>
void
build_mass (
  const Basis&       b,
  reference_element  hat_K,
  arma::Mat<T>&      mass)
{
  typedef reference_element::size_type size_type;
  quadrature_option qopt;
  qopt.set_order(2*b.degree());
  quadrature<Float> quad (qopt);
  size_type loc_ndof = b.size (hat_K);
  arma::Col<T> phi (loc_ndof);
  mass.resize (loc_ndof, loc_ndof);
  mass.fill (T(0));
  for (typename quadrature<T>::const_iterator iter_q = quad.begin(hat_K),
	last_q = quad.end(hat_K); iter_q != last_q; iter_q++) {	
    b.eval (hat_K, (*iter_q).x, phi);
    mass += ((*iter_q).w*phi)*trans(phi);
  }
}
// -----------------------------------------------------------------------------
// grad_grad
// -----------------------------------------------------------------------------
template<class T, class Basis>
void
build_grad_grad (
  const Basis&       b,
  reference_element  hat_K,
  arma::Mat<T>&      grad_grad)
{
  typedef reference_element::size_type size_type;
  quadrature_option qopt;
  qopt.set_order(2*b.degree());
  quadrature<Float> quad (qopt);
  size_type loc_ndof = b.size (hat_K);
  std::vector<point_basic<T> > grad_phi (loc_ndof);
  grad_grad.resize (loc_ndof, loc_ndof);
  grad_grad.fill (T(0));
  for (typename quadrature<T>::const_iterator iter_q = quad.begin(hat_K),
	last_q = quad.end(hat_K); iter_q != last_q; iter_q++) {	
    b.grad_eval (hat_K, (*iter_q).x, grad_phi);
    for (size_type loc_idof = 0; loc_idof < loc_ndof; ++loc_idof) {
    for (size_type loc_jdof = 0; loc_jdof < loc_ndof; ++loc_jdof) {
      grad_grad (loc_idof,loc_jdof) += (*iter_q).w*dot(grad_phi[loc_idof],grad_phi[loc_jdof]);
    }}
  }
}

} // namespace rheolef
#endif // _RHEO_FORM_ELEMENT_H
