package Role::HasPayload::Merged 0.007;
use Moose::Role;
# ABSTRACT: merge autogenerated payload with constructor-specified payload

#pod =head1 SYNOPSIS
#pod
#pod   package Example;
#pod   use Moose;
#pod
#pod   with qw(Role::HasPayload::Merged);
#pod
#pod   sub Payload { 'Role::HasPayload::Meta::Attribute::Payload' }
#pod
#pod   has height => (
#pod     is => 'ro',
#pod     traits   => [ Payload ],
#pod   );
#pod
#pod   has width => (
#pod     is => 'ro',
#pod     traits   => [ Payload ],
#pod   );
#pod
#pod   has color => (
#pod     is => 'ro',
#pod   );
#pod
#pod ...then...
#pod
#pod   my $example = Example->new({
#pod     height  => 10,
#pod     width   => 20,
#pod     color   => 'blue',
#pod     payload => { depth => 30 },
#pod   });
#pod
#pod   $example->payload; # { height => 10, width => 20, depth => 30 }
#pod
#pod =head1 DESCRIPTION
#pod
#pod Role::HasPayload::Merged provides a C<payload> method and a C<payload>
#pod attribute.  It computes the result of the C<payload> method when it's called,
#pod first by gathering the values of attributes marked with
#pod Role::HasPayload::Meta::Attribute::Payload, then by merging in the contents of
#pod the C<payload> attribute (provided at construction).
#pod
#pod If an entry in the constructor-provided payload already exists in the
#pod autogenerated payload, a warning is issued and the autogenerated value is used.
#pod
#pod For a bit more on the autogenerated payload, see L<Role::HasPayload::Auto>.
#pod
#pod This role is especially useful when combined with L<Role::HasMessage::Errf>.
#pod
#pod =cut

use Role::HasPayload::Meta::Attribute::Payload;

has payload => (
  reader  => '_payload_to_merge',
  isa     => 'HashRef',
  default => sub {  {}  },
);

sub payload {
  my ($self) = @_;

  my @attrs = grep { $_->does('Role::HasPayload::Meta::Attribute::Payload') }
              $self->meta->get_all_attributes;

  my %payload = map {;
    my $method = $_->get_read_method;
    ($_->name => $self->$method)
  } @attrs;

  my $manual_payload = $self->_payload_to_merge;
  KEY: for my $key (keys %$manual_payload) {
    if (exists $payload{ $key }) {
      Carp::carp("declining to override automatic payload entry $key");
      next KEY;
    }

    $payload{ $key } = $manual_payload->{ $key };
  }

  return \%payload;
}

no Moose::Role;
1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Role::HasPayload::Merged - merge autogenerated payload with constructor-specified payload

=head1 VERSION

version 0.007

=head1 SYNOPSIS

  package Example;
  use Moose;

  with qw(Role::HasPayload::Merged);

  sub Payload { 'Role::HasPayload::Meta::Attribute::Payload' }

  has height => (
    is => 'ro',
    traits   => [ Payload ],
  );

  has width => (
    is => 'ro',
    traits   => [ Payload ],
  );

  has color => (
    is => 'ro',
  );

...then...

  my $example = Example->new({
    height  => 10,
    width   => 20,
    color   => 'blue',
    payload => { depth => 30 },
  });

  $example->payload; # { height => 10, width => 20, depth => 30 }

=head1 DESCRIPTION

Role::HasPayload::Merged provides a C<payload> method and a C<payload>
attribute.  It computes the result of the C<payload> method when it's called,
first by gathering the values of attributes marked with
Role::HasPayload::Meta::Attribute::Payload, then by merging in the contents of
the C<payload> attribute (provided at construction).

If an entry in the constructor-provided payload already exists in the
autogenerated payload, a warning is issued and the autogenerated value is used.

For a bit more on the autogenerated payload, see L<Role::HasPayload::Auto>.

This role is especially useful when combined with L<Role::HasMessage::Errf>.

=head1 PERL VERSION

This library should run on perls released even a long time ago.  It should work
on any version of perl released in the last five years.

Although it may work on older versions of perl, no guarantee is made that the
minimum required version will not be increased.  The version may be increased
for any reason, and there is no promise that patches will be accepted to lower
the minimum required perl.

=head1 AUTHOR

Ricardo Signes <cpan@semiotic.systems>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2022 by Ricardo Signes.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
